/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.enterprise.security.auth.login;

import java.util.*;
import java.util.logging.Logger;
import java.util.logging.Level;
import com.sun.logging.LogDomains;

import javax.security.auth.*;
import javax.security.auth.callback.*;
import javax.security.auth.login.*;
import javax.security.auth.spi.*;

import com.sun.enterprise.security.auth.realm.ldap.LDAPRealm;

/**
 * iAS JAAS LoginModule for an LDAP Realm.
 *
 * <P>Refer to the LDAPRealm documentation for necessary and optional
 * configuration parameters for the iAS LDAP login support.
 *
 * <P>There are various ways in which a user can be authenticated using
 * an LDAP directory. Currently this login module only supports one mode,
 * 'find and bind'. Other modes may be added as schedules permit.
 *
 * <P>Mode: <i>find-bind</i>
 * <ol>
 *  <li>An LDAP search is issued on the directory starting at base-dn
 *      with the given search-filter (having substituted the user name
 *      in place of %s). If no entries match this search, login fails
 *      and authentication is over.
 *  <li>The DN of the entry which matched the search as the DN
 *      of the user in the directory. If the search-filter
 *      is properly set there should always be a single match; if there are
 *      multiple matches, the first one found is used.
 *  <li>Next an LDAP bind is attempted using the above DN and the provided
 *      password. If this fails, login is considered to have failed and
 *      authentication is over.
 *  <li>Then an LDAP search is issued on the directory starting at
 *      group-base-dn with the given group-search-filter (having
 *      substituted %d for the user DN previously found). From the
 *      matched entry(ies) all the values of group-target are taken
 *      as group names in which the user has membership. If no entries
 *      are found, the group membership is empty.
 * </ol>
 *
 *
 */
public class LDAPLoginModule extends PasswordLoginModule
{
    private LDAPRealm _ldapRealm;

    /**
     * Performs authentication for the current user.
     *
     */
    protected void authenticate ()
        throws LoginException
    {
        if (!(_currentRealm instanceof LDAPRealm)) {
            String msg = sm.getString("ldaplm.badrealm");
            throw new LoginException(msg);
        }
        _ldapRealm = (LDAPRealm)_currentRealm;
        
                                // enforce that password cannot be empty.
                                // ldap may grant login on empty password!
        if (_password == null || _password.length() == 0) {
            String msg = sm.getString("ldaplm.emptypassword", _username);
            throw new LoginException(msg);
        }
        
        String mode = _currentRealm.getProperty(LDAPRealm.PARAM_MODE);

        if (LDAPRealm.MODE_FIND_BIND.equals(mode)) {
            String[] grpList = _ldapRealm.findAndBind(_username, _password);
            String[] groupListToForward = new String[grpList.length];
            for (int i = 0; i< grpList.length; i++){
                groupListToForward[i] = grpList[i];
            }

            commitAuthentication(_username, _password,
                    _currentRealm, groupListToForward);
        } else {
            String msg = sm.getString("ldaplm.badmode", mode);
            throw new LoginException(msg);
        }
    }
}
