/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */
package com.sun.enterprise.server.logging;

import java.util.Set;

import java.util.logging.LogRecord;
import java.util.logging.Level;
import java.util.logging.Formatter;

import javax.management.ObjectName;
import javax.management.MBeanServer;

import java.lang.reflect.Method;
import java.lang.reflect.Constructor;

/**
    Hook used for AMX logging by FileandSysLogHandler.
    <b>This class assumes that the caller is holding a lock that
    will prevent more than one thread from calling {@link #publish}.</b>
 */
final class AMXLoggingHook    // don't make this class public
{
    private ObjectName    mLoggingObjectName;
    private MBeanServer   mMBeanServer;
    private boolean       mLoggingAvailable;
    private LoggingImplHook mLoggingImplHook;
    
    /**
        The minimum log level for which LogRecords will be passed
        to the AMX Logging MBean.  Generally speaking, all LogRecords
        should be passed along, letting the AMX Logging MBean decide
        their disposition.
     */
    private Level               mMinimumLogLevel;
    
   // /** signature for LOGGING_HOOK_METHOD_NAME method */
   // private static final String[]  LOGGING_HOOK_SIG  = new String[] 
   //      { LogRecord.class.getName(), Formatter.class.getName() };
    
   // /** name of the operation called on the AMX Logging MBean */
   // private static final String LOGGING_HOOK_NAME    = "privateLoggingHook";
    
    private static final boolean    DEBUG   = false;
    private final Output            mOut;
    private final String            mServerName;
    
        private void
    dumpSystemProps( final Output output )
    {
        final java.util.Properties    props   = System.getProperties();
        
        final String[]  keys    = (String[])props.keySet().toArray( new String[0] );
        java.util.Arrays.sort( keys );
        for( final String key : keys )
        {
            debug( key + "=" + props.getProperty( key ) );
        }
        
    }
    
    AMXLoggingHook()
    {
        mServerName = System.getProperty(
            com.sun.enterprise.util.SystemPropertyConstants.SERVER_NAME);
        final String  instanceRoot  = System.getProperty(
            com.sun.enterprise.util.SystemPropertyConstants.INSTANCE_ROOT_PROPERTY );
            
        mOut    = createOutput( DEBUG, instanceRoot + "/AMXLoggingHook-" + mServerName + ".debug" );
        // debug( "\n_______________________________________________________________________________");
        // debug( "AMXLoggingHook: REV = 1");
        // debug( "AMXLoggingHook: started at " + new java.util.Date() + " for server " + mServerName);
        
        // dumpSystemProps( mOut );

        mLoggingObjectName  = null;
        mMBeanServer        = getMBeanServer(); // may or may not be available
        
        mMinimumLogLevel    = Level.FINEST;
        mLoggingImplHook    = null;
        
    }
    
        private static Output
   createOutput( final boolean debug, final String fileName )
   {
        try
        {
           final java.io.File   f   = new java.io.File( fileName );
           
           return debug ? new FileOutput( f ) : new NullOutput();
        }
        catch( Throwable t )
        {
            // nothing can be done; fall through and return NullOutput
        }
        
        return new NullOutput();
   }
    
    private final void  debug( final Object o )   { mOut.println( o.toString() ); }

        private MBeanServer
    getMBeanServer()
    {
        if ( mMBeanServer == null )
        {
            mMBeanServer =
                com.sun.enterprise.admin.common.MBeanServerFactory.getMBeanServer();
            // may still be null if not yet available
            if ( mMBeanServer != null )
            {
                debug( "MBeanServer NOW EXISTS, creating LoggingImpl" );
                initLogging( mMBeanServer );
            }
        }
        
        return mMBeanServer;
    }
    
    
     private static final String  LOGGING_IMPL_CLASSNAME =
        "com.sun.enterprise.management.ext.logging.LoggingImpl";
    /**
        Called once to install the Logging MBean
     */
        private void
    initLogging( final MBeanServer mbeanServer )
    {
        try
        {
            final Class  loggingClass = Class.forName( LOGGING_IMPL_CLASSNAME );
            
            final Constructor constructor = loggingClass.getConstructor( String.class );
            mLoggingImplHook = (LoggingImplHook)constructor.newInstance( mServerName );
            
            final Method   getObjectNameMethod =
                loggingClass.getMethod( "getObjectName", String.class );
            mLoggingObjectName    = (ObjectName)getObjectNameMethod.invoke( mLoggingImplHook, mServerName );
            debug( "registering Logging as: " + mLoggingObjectName );
            mbeanServer.registerMBean( mLoggingImplHook, mLoggingObjectName );
         }
         catch ( Exception e )
         {
            final String msg = "Can't load " + LOGGING_IMPL_CLASSNAME + ", caught: " + e;
            debug( msg );
            throw new Error( msg );
         }
    }
    
        public Level
    getMinimumLogLevel( )
    {
        return mMinimumLogLevel;
    }
    
        public void
    setMinimumLogLevel( final Level level )
    {
        mMinimumLogLevel    = level;
    }
    
        // call the Logging MBean
        void
    publish( 
        final LogRecord record,
        final Formatter theFormatter )
    {
        if ( record.getLevel().intValue() < mMinimumLogLevel.intValue() )
        {
            return;
        }

        debug( "publish: " + theFormatter.format( record ) );

        final MBeanServer   server    = getMBeanServer();
        if ( server != null )
        {
            if ( ! mLoggingAvailable )
            {
                mLoggingAvailable   = server.isRegistered( mLoggingObjectName );
                if ( mLoggingAvailable )
                {
                    debug( "Logging MBean is now available: " + mLoggingObjectName );
                }
            }
            
            if ( mLoggingAvailable )
            {
                try
                {
                    mLoggingImplHook.privateLoggingHook( record, theFormatter );
                    //final Object[]  args  = new Object[]  { record, theFormatter };
                    //server.invoke( mLoggingObjectName, LOGGING_HOOK_NAME , args, LOGGING_HOOK_SIG );
                }
                catch( Throwable t )
                {
                    debug(
                        "AMXLoggingHook.publish: Exception calling privateLoggingHook: " + t );
                    // squelch--we can't log it or we'll have a recursive call
                }
            }
            else
            {
                debug( "publish: no Logging yet." );
            }
        }
        else
        {
            debug( "publish: no MBeanServer yet." );
        }
    }
    

    
    /**
        Maintains an ordered list of LogRecords which
        could not be sent to the not-yet-existing AMX Logging MBean.
    private static final class StartupRecords {
        private final List<LogRecord>  mList;
        
        public StartupRecords() {
            mList   = new ArrayList<LogRecord>();
        }
        
        public void add( final LogRecord r ){mList.add( r );}
        public void clear(){    mList.clear(); }
        public List   getAll()  { return Collections.unmodifiableList( mList ); }
    };
    
     */
    
    private interface Output
    {
        public void println( Object o );
    }

    /**
    	Directs output to a file; used internally for debugging to avoid
    	infinite recursion between logging and this hook.
     */
    private static final class FileOutput implements Output
    {
        private java.io.PrintStream mOut;
        
        // passing null means output is not emitted
    		public
    	FileOutput( final java.io.File f)
    	    throws java.io.IOException
    	{
    	    mOut    = new java.io.PrintStream( new java.io.FileOutputStream( f ) );
    	}
    	
    		public void
    	println( Object o )
    	{
    	    mOut.println( o.toString() );
    	}
    	
    		public void
    	close( )
    	{
    	    if ( mOut != null )
    	    try
    	    {
    	        mOut.close();
    	    }
    	    finally
    	    {
    	        mOut    = null;
    	    }
    	}
    };
    
    // ignores all output
    private static final class NullOutput implements Output
    {
        NullOutput()    {}
        public void println( Object o ) {}
    }
}




