/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.enterprise.web.accesslog;

import java.nio.CharBuffer;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.LinkedList;
import java.util.logging.Logger;
import java.util.logging.Level;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import org.apache.catalina.Container;
import org.apache.catalina.HttpResponse;
import org.apache.catalina.Request;
import org.apache.catalina.Response;
import com.sun.logging.LogDomains;
import com.sun.enterprise.web.Constants;

/**
 * Access log formatter using the SJSAS format.
 */
public class DefaultAccessLogFormatterImpl extends AccessLogFormatter {

    private static final Logger _logger =
        LogDomains.getLogger(LogDomains.WEB_LOGGER);

    /*
     * HTTP header names
     */
    private static final String HTTP_HEADER_ACCEPT = "Accept";
    private static final String HTTP_HEADER_AUTHORIZATION = "Authorization";
    private static final String HTTP_HEADER_DATE = "Date";
    private static final String HTTP_HEADER_IF_MODIFIED_SINCE = "If-Modified-Since";

    /*
     * Supported access log entry tokens
     */
    private static final String AUTH_USER_NAME = "auth-user-name";
    private static final String CLIENT_DNS = "client.dns";
    private static final String CLIENT_NAME = "client.name";
    private static final String COOKIE_VALUE = "cookie.value";
    private static final String DATE_TIME = "datetime";
    private static final String HEADER_ACCEPT = "header.accept";
    private static final String HEADER_ANY = "header.";
    private static final int HEADER_ANY_LEN = HEADER_ANY.length();
    private static final String HEADER_AUTH = "header.auth";
    private static final String HEADER_DATE = "header.date";
    private static final String HEADER_IF_MOD_SINCE = "header.if-mod-since";
    private static final String HEADER_USER_AGENT = "header.user-agent";
    private static final String HEADER_REFERER = "header.referer";
    private static final String HTTP_METHOD = "http-method";
    private static final String HTTP_URI = "http-uri";
    private static final String HTTP_VERSION = "http-version";
    private static final String QUERY_STR = "query-str";
    private static final String REFERER = "referer";
    private static final String REQUEST = "request";
    private static final String RESPONSE_LENGTH = "response.length";
    private static final String STATUS = "status";
    private static final String TIME_TAKEN = "time-taken";
    private static final String USER_AGENT = "user.agent";
    private static final String VS_ID = "vs.id";

    private Container container;

    /**
     * List of access log pattern components
     */
    private LinkedList<String> patternComponents;


    /**
     * Constructor.
     *
     * @param pattern The access log pattern
     * @param container The container associated with the access log valve
     */
    public DefaultAccessLogFormatterImpl(String pattern, Container container) {

        super();

        this.patternComponents = parsePattern(pattern);
        this.container = container;

        dayFormatter = new SimpleDateFormat("dd");
        dayFormatter.setTimeZone(tz);
        monthFormatter = new SimpleDateFormat("MM");
        monthFormatter.setTimeZone(tz);
        yearFormatter = new SimpleDateFormat("yyyy");
        yearFormatter.setTimeZone(tz);
        timeFormatter = new SimpleDateFormat("HH:mm:ss");
        timeFormatter.setTimeZone(tz);
    }


    /**
     * Appends an access log entry line, with info obtained from the given
     * request and response objects, to the given CharBuffer.
     *
     * @param request The request object from which to obtain access log info
     * @param response The response object from which to obtain access log info
     * @param charBuffer The CharBuffer to which to append access log info
     */
    public void appendLogEntry(Request request,
                               Response response,
                               CharBuffer charBuffer) {

        ServletRequest req = request.getRequest();
        HttpServletRequest hreq = (HttpServletRequest) req;

        for (int i=0; i<patternComponents.size(); i++) {
            String pc = patternComponents.get(i);
            if (AUTH_USER_NAME.equals(pc)) {
                appendAuthUserName(charBuffer, hreq);
            } else if (CLIENT_DNS.equals(pc)) {
                appendClientDNS(charBuffer, req);
            } else if (CLIENT_NAME.equals(pc)) {
                appendClientName(charBuffer, req);
            } else if (COOKIE_VALUE.equals(pc)) {
                appendCookieValue(charBuffer, hreq);
            } else if (DATE_TIME.equals(pc)) {
                appendCurrentDate(charBuffer);       
            } else if (HEADER_ACCEPT.equals(pc)) {
                appendHeaderAccept(charBuffer, hreq);
            } else if (HEADER_AUTH.equals(pc)) {
                appendHeaderAuth(charBuffer, hreq);
            } else if (HEADER_DATE.equals(pc)) {
                appendHeaderDate(charBuffer, hreq);
            } else if (HEADER_IF_MOD_SINCE.equals(pc)) {
                appendHeaderIfModSince(charBuffer, hreq);
            } else if (HEADER_USER_AGENT.equals(pc)) {
                appendUserAgent(charBuffer, hreq);
            } else if (HEADER_REFERER.equals(pc)) {
                appendReferer(charBuffer, hreq);
            } else if (HTTP_METHOD.equals(pc)) {
                appendHTTPMethod(charBuffer, hreq);
            } else if (HTTP_URI.equals(pc)) {
                appendHTTPUri(charBuffer, hreq);
            } else if (HTTP_VERSION.equals(pc)) {
                appendHTTPVersion(charBuffer, hreq);
            } else if (QUERY_STR.equals(pc)) {
                appendQueryString(charBuffer, hreq);
            } else if (REFERER.equals(pc)) {
                appendReferer(charBuffer, hreq);
            } else if (REQUEST.equals(pc)) {
                appendRequestInfo(charBuffer, hreq);
            } else if (RESPONSE_LENGTH.equals(pc)) {
                appendResponseLength(charBuffer, response);
            } else if (STATUS.equals(pc)) {
                appendResponseStatus(charBuffer, response);
            } else if (TIME_TAKEN.equals(pc)) {
                appendTimeTaken(charBuffer, request);
            } else if (USER_AGENT.equals(pc)) {
                appendUserAgent(charBuffer, hreq);
            } else if (VS_ID.equals(pc)) {
                appendVirtualServerId(charBuffer);
            } else if (pc.startsWith(HEADER_ANY)) {
                appendHeaderAny(charBuffer, pc.substring(HEADER_ANY_LEN),
                                hreq);
            }

            charBuffer.put(SPACE);
        }
    }


    /*
     * Parses the access log pattern (that was specified via setPattern) into
     * its individual components, and returns them as a list.
     *
     * @param pattern The pattern to parse
     *
     * @return List containing the access log pattern components
     */
    private LinkedList<String> parsePattern(String pattern) {

        LinkedList list = new LinkedList();

        int from = 0;
        int end = -1;
        int index = -1;
        while ((index = pattern.indexOf('%', from)) >= 0) {
            end = pattern.indexOf('%', index+1);
            if (end < 0) {
                _logger.log(
                    Level.SEVERE,
                    "peaccesslogvalve.missingAccessLogPatternEndDelimiter",
                    pattern);
            }
            String component = pattern.substring(index+1, end);

            if (!AUTH_USER_NAME.equals(component) 
                    && !CLIENT_DNS.equals(component) 
                    && !CLIENT_NAME.equals(component) 
                    && !COOKIE_VALUE.equals(component) 
                    && !DATE_TIME.equals(component) 
                    && !HEADER_ACCEPT.equals(component) 
                    && !HEADER_AUTH.equals(component) 
                    && !HEADER_DATE.equals(component) 
                    && !HEADER_IF_MOD_SINCE.equals(component) 
                    && !HEADER_USER_AGENT.equals(component) 
                    && !HEADER_REFERER.equals(component) 
                    && !HTTP_METHOD.equals(component) 
                    && !HTTP_URI.equals(component) 
                    && !HTTP_VERSION.equals(component) 
                    && !QUERY_STR.equals(component) 
                    && !REFERER.equals(component) 
                    && !REQUEST.equals(component) 
                    && !RESPONSE_LENGTH.equals(component) 
                    && !STATUS.equals(component) 
                    && !TIME_TAKEN.equals(component) 
                    && !USER_AGENT.equals(component) 
                    && !VS_ID.equals(component)
                    && !component.startsWith(HEADER_ANY)) {
                _logger.log(
                    Level.SEVERE,
                    "peaccesslogvalve.invalidAccessLogPatternComponent",
                    new Object[] { component, pattern });
            }

            if (TIME_TAKEN.equals(component)) {
                needTimeTaken = true;
            }

            list.add(component);
            from = end + 1;    
        }

        return list;
    }


    /*
     * Appends the client host name of the given request to the given char
     * buffer.
     */
    private void appendClientName(CharBuffer cb, ServletRequest req) {
        cb.put("\"");
        String value = req.getRemoteHost();
        if (value == null) {
            value = "NULL-CLIENT-NAME";
        }
        cb.put(value);
        cb.put("\"");
    }


    /*
     * Appends the client DNS of the given request to the given char
     * buffer.
     */
    private void appendClientDNS(CharBuffer cb, ServletRequest req) {
        cb.put("\"");
        String value = req.getRemoteAddr();
        if (value == null) {
            value = "NULL-CLIENT-DNS";
        }
        cb.put(value);
        cb.put("\"");
    }


    /*
     * Appends the authenticated user (if any) to the given char buffer.
     */
    private void appendAuthUserName(CharBuffer cb, HttpServletRequest hreq) {
        cb.put("\"");
        String user = hreq.getRemoteUser();
        if (user == null) {
            user = "NULL-AUTH-USER";
        }
        cb.put(user);
        cb.put("\"");
    }


    /*
     * Appends the current date to the given char buffer.
     */
    private void appendCurrentDate(CharBuffer cb) {
        cb.put("\"");
        Date date = getDate();
        cb.put(dayFormatter.format(date));           // Day
        cb.put('/');
        cb.put(lookup(monthFormatter.format(date))); // Month
        cb.put('/');
        cb.put(yearFormatter.format(date));          // Year
        cb.put(':');
        cb.put(timeFormatter.format(date));          // Time
        cb.put(SPACE);
        cb.put(timeZone);                            // Time Zone
        cb.put("\"");
    }


    /*
     * Appends info about the given request to the given char buffer.
     */
    private void appendRequestInfo(CharBuffer cb, HttpServletRequest hreq) {
        cb.put("\"");
        cb.put(hreq.getMethod());
        cb.put(SPACE);
        cb.put(hreq.getRequestURI());
        if (hreq.getQueryString() != null) {
            cb.put('?');
            cb.put(hreq.getQueryString());
        }
        cb.put(SPACE);
        cb.put(hreq.getProtocol());
        cb.put("\"");
    }


    /*
     * Appends the response status to the given char buffer.
     */
    private void appendResponseStatus(CharBuffer cb, Response response) {
        cb.put(String.valueOf(((HttpResponse) response).getStatus()));
    }


    /*
     * Appends the content length of the given response to the given char
     * buffer.
     */
    private void appendResponseLength(CharBuffer cb, Response response) {
        cb.put("" + response.getContentCount());
    }


    /*
     * Appends the value of the 'user-agent' header of the given request to
     * the given char buffer.
     */
    private void appendUserAgent(CharBuffer cb, HttpServletRequest hreq) {
        cb.put("\"");
        String ua = hreq.getHeader("user-agent");
        if (ua == null) {
            ua = "NULL-USER-AGENT";
        }
        cb.put(ua);
        cb.put("\"");
    }


    /*
     * Appends the time (in milliseconds) it has taken to service the given
     * request to the given char buffer.
     */
    private void appendTimeTaken(CharBuffer cb, Request req) {

        String timeTaken = "NULL-TIME-TAKEN";

        cb.put("\"");
        Long startTimeObj = (Long) req.getNote(
            Constants.REQUEST_START_TIME_NOTE);
        if (startTimeObj != null) {
            long startTime = startTimeObj.longValue();
            long endTime = System.currentTimeMillis();
            timeTaken = String.valueOf(endTime - startTime);
        } 
        cb.put(timeTaken);
        cb.put("\"");
    }


    /*
     * Appends the value of the 'referer' header of the given request to
     * the given char buffer.
     */
    private void appendReferer(CharBuffer cb, HttpServletRequest hreq) {
        cb.put("\"");
        String referer = hreq.getHeader("referer");
        if (referer == null) {
            referer = "NULL-REFERER";
        }
        cb.put(referer);
        cb.put("\"");
    }


    /*
     * Appends the Accept header of the given request to the given char
     * buffer.
     */
    private void appendHeaderAccept(CharBuffer cb, HttpServletRequest hreq) {
        cb.put("\"");
        String accept = hreq.getHeader(HTTP_HEADER_ACCEPT);
        if (accept == null) {
            accept = "NULL-HEADER-ACCEPT";
        }
        cb.put(accept);
        cb.put("\"");
    }


    /*
     * Appends the Authorization header of the given request to the given char
     * buffer.
     */
    private void appendHeaderAuth(CharBuffer cb, HttpServletRequest hreq) {
        cb.put("\"");
        String auth = hreq.getHeader(HTTP_HEADER_AUTHORIZATION);
        if (auth == null) {
            auth = "NULL-HEADER-AUTHORIZATION";
        }
        cb.put(auth);
        cb.put("\"");
    }


    /*
     * Appends the Date header of the given request to the given char buffer.
     */
    private void appendHeaderDate(CharBuffer cb, HttpServletRequest hreq) {
        cb.put("\"");
        String date = hreq.getHeader(HTTP_HEADER_DATE);
        if (date == null) {
            date = "NULL-HEADER-DATE";
        }
        cb.put(date);
        cb.put("\"");
    }


    /*
     * Appends the If-Modified-Since header of the given request to the
     * given char buffer.
     */
    private void appendHeaderIfModSince(CharBuffer cb,
                                        HttpServletRequest hreq) {
        cb.put("\"");
        String ifModSince = hreq.getHeader(HTTP_HEADER_IF_MODIFIED_SINCE);
        if (ifModSince == null) {
            ifModSince = "NULL-HEADER-IF-MODIFIED-SINCE";
        }
        cb.put(ifModSince);
        cb.put("\"");
    }


    /*
     * Appends the value of the specified header name of the given request
     * to the given char buffer.
     */
    private void appendHeaderAny(CharBuffer cb,
                                 String headerName,
                                 HttpServletRequest hreq) {
        cb.put("\"");
        String value = hreq.getHeader(headerName);
        if (value == null) {
            value = "NULL-HEADER-" + headerName.toUpperCase();
        }
        cb.put(value);
        cb.put("\"");
    }


    /*
     * Appends the value of the first cookie of the given request to the
     * given char buffer.
     */
    private void appendCookieValue(CharBuffer cb, HttpServletRequest hreq) {
        cb.put("\"");
        String cookieValue = "NULL-COOKIE-VALUE";
        Cookie[] cookies = hreq.getCookies();
        if (cookies != null && cookies.length > 0) {
            cookieValue = cookies[0].getValue();
        }
        cb.put(cookieValue);
        cb.put("\"");
    }


    /*
     * Appends the HTTP method of the given request to the given char buffer.
     */
    private void appendHTTPMethod(CharBuffer cb, HttpServletRequest hreq) {
        cb.put("\"");
        String method = hreq.getMethod();
        if (method == null) {
            method = "NULL-HTTP-METHOD";
        }
        cb.put(method);
        cb.put("\"");
    }


    /*
     * Appends the URI of the given request to the given char buffer.
     */
    private void appendHTTPUri(CharBuffer cb, HttpServletRequest hreq) {
        cb.put("\"");
        String uri = hreq.getRequestURI();
        if (uri == null) {
            uri = "NULL-HTTP-URI";
        }
        cb.put(uri);
        cb.put("\"");
    }


    /*
     * Appends the HTTP protocol version of the given request to the given
     * char buffer.
     */
    private void appendHTTPVersion(CharBuffer cb, HttpServletRequest hreq) {
        cb.put("\"");
        String protocol = hreq.getProtocol();
        if (protocol == null) {
            protocol = "NULL-HTTP-PROTOCOL";
        }
        cb.put(protocol);
        cb.put("\"");
    }


    /*
     * Appends the query string of the given request to the given char buffer.
     */
    private void appendQueryString(CharBuffer cb, HttpServletRequest hreq) {
        cb.put("\"");
        String query = hreq.getQueryString();
        if (query == null) {
            query = "NULL-QUERY";
        }
        cb.put(query);
        cb.put("\"");
    }


    /*
     * Appends the id of the virtual server with which this access log valve
     * has been associated to the given char buffer.
     */
    private void appendVirtualServerId(CharBuffer cb) {
        String vsId = "NULL-VIRTUAL-SERVER";
        if (container != null) {
            vsId = container.getName();
        }
        cb.put(vsId);
    }

}
