/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.enterprise.webservice;

import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.HashMap;
import java.util.Map;

import javax.security.auth.Subject;
import javax.security.auth.message.config.*;
import javax.security.auth.message.AuthException;
import javax.security.auth.message.AuthStatus;
import javax.xml.ws.WebServiceException;

import com.sun.enterprise.deployment.ServiceReferenceDescriptor;
import com.sun.enterprise.security.jmac.PacketMessageInfo;
import com.sun.enterprise.security.jmac.PacketMapMessageInfo;
import com.sun.enterprise.security.jmac.config.PipeHelper;
import com.sun.enterprise.util.LocalStringManagerImpl;
import com.sun.logging.LogDomains;

import com.sun.xml.ws.api.SOAPVersion;
import com.sun.xml.ws.api.model.wsdl.WSDLPort;
import com.sun.xml.ws.api.model.wsdl.WSDLService;
import com.sun.xml.ws.api.pipe.Pipe;
import com.sun.xml.ws.api.pipe.PipeCloner;
import com.sun.xml.ws.api.pipe.helper.AbstractFilterPipeImpl;
import com.sun.xml.ws.api.message.Packet;
import com.sun.xml.ws.api.message.Message;

import com.sun.xml.ws.security.secconv.SecureConversationInitiator;
import com.sun.xml.ws.security.secconv.WSSecureConversationException;
import javax.xml.bind.JAXBElement;

/**
 * This pipe is used to do client side security for app server
 */
public class ClientSecurityPipe extends AbstractFilterPipeImpl
    implements SecureConversationInitiator {

    protected PipeHelper helper;
   
    protected static Logger _logger = LogDomains.getLogger(
        LogDomains.SECURITY_LOGGER);
    protected static LocalStringManagerImpl localStrings = 
        new LocalStringManagerImpl(ClientSecurityPipe.class);

    public ClientSecurityPipe(Map props, Pipe next) {

        super(next);

	props.put(PipeConstants.SECURITY_PIPE,this);

        WSDLPort wsdlModel = (WSDLPort)props.get(PipeConstants.WSDL_MODEL);
        if (wsdlModel != null) {
            props.put(PipeConstants.WSDL_SERVICE,
                wsdlModel.getOwner().getName());
        }

	this.helper = new PipeHelper(PipeConstants.SOAP_LAYER,props,null);

    }

    protected ClientSecurityPipe(ClientSecurityPipe that, PipeCloner cloner) {
        super(that, cloner);
	// XXX all the cloned pipes will share the same helper
	// is that what we want?
        this.helper = that.helper;
    }
		       
    public void preDestroy() {

	// XXX calling cleanSubject to afford context ability to
	// cleanup session/conversation, but don't have info, context, 
	// or subject

	Subject s = getClientSubject(null);

	PacketMessageInfo info = new PacketMapMessageInfo(null,null);

	try {
	    ClientAuthContext cAC = helper.getClientAuthContext(info,s);
	    if (cAC != null) {
		cAC.cleanSubject(info,s);
	    }
	} catch (AuthException e) {
	    
	    WSDLPort wsdlModel = 
		(WSDLPort) helper.getProperty(PipeConstants.WSDL_MODEL);
	    
	    throw new WebServiceException
		(localStrings.getLocalString
		 ("enterprise.webservice.cantCleanupSession",
		  "Cannot cleanup session for {0}",
		  new Object[] { 
		     wsdlModel == null ? "unknown" : wsdlModel.getName() }),e);

	} finally {
	    // XXX all the cloned pipes will share the same helper
	    // is it ok to disable the helper
	    helper.disable();
	}
    }    
    
    public final Pipe copy(PipeCloner cloner) {
        return new ClientSecurityPipe(this, cloner);
    }
    
    public Packet process(Packet request) {

	/*
	 * XXX should there be code like the following?
	 if(isHttpBinding) {
	     return next.process(request);
	 }
        */

	PacketMessageInfo info= new PacketMapMessageInfo(request,new Packet());
       
        info.getMap().put(javax.xml.ws.Endpoint.WSDL_SERVICE,
            helper.getProperty(PipeConstants.WSDL_SERVICE));

        AuthStatus status = AuthStatus.SEND_SUCCESS;

	Subject clientSubject = getClientSubject(request);

	ClientAuthContext cAC = null;

	try {

	    cAC = helper.getClientAuthContext(info,clientSubject);

	    if (cAC != null) {

		// proceed to process message sescurity
		status = cAC.secureRequest(info, clientSubject);
	    }

	} catch(Exception e) {

	    _logger.log(Level.SEVERE,"ws.error_secure_request", e);
	    
	    WSDLPort wsdlModel = 
		(WSDLPort) helper.getProperty(PipeConstants.WSDL_MODEL);

	    throw new WebServiceException
		(localStrings.getLocalString
		 ("enterprise.webservice.cantSecureRequst",
		  "Cannot secure request for {0}",
		  new Object[] { 
		     wsdlModel == null ? "unknown" : wsdlModel.getName() }),e);
	} 

	Packet response = null;

	if (status == AuthStatus.FAILURE) {
	    if (_logger.isLoggable(Level.FINE)) {
		_logger.log(Level.FINE,"ws.status_secure_request", status);
	    }
	    response = info.getResponsePacket();
	} else {
	    response = processSecureRequest(info,cAC,clientSubject);
	}

        return response;
    }    
	
    private Packet processSecureRequest(PacketMessageInfo info,
	ClientAuthContext cAC, Subject clientSubject) 
	throws WebServiceException {
        
        AuthStatus status = AuthStatus.SUCCESS;

	Packet response = next.process(info.getRequestPacket());
	
	// check for SOAP fault
	Message m = response.getMessage();
	if (m != null && m.isFault()) {
	    return response;
	}

	if (cAC != null) {

	    info.setResponsePacket(response);

	    try {
		status = cAC.validateResponse(info,clientSubject,null);
	    } catch (Exception e) {

		WSDLPort wsdlModel = 
		    (WSDLPort) helper.getProperty(PipeConstants.WSDL_MODEL);
		
		throw new WebServiceException
		    (localStrings.getLocalString
		     ("enterprise.webservice.cantValidateResponse",
		      "Cannot validate response for {0}",
		      new Object[] { 
			 wsdlModel == null ? "unknown" : wsdlModel.getName() }),e);
	    }

	    if (status == AuthStatus.SEND_CONTINUE) {
		response = processSecureRequest(info, cAC, clientSubject);
	    } else {

		response = info.getResponsePacket();
	    }
	}

	return response;
    }

    private static Subject getClientSubject(Packet p) {

	Subject s = null;

	if (p != null) {
	    s = (Subject) 
		p.invocationProperties.get(PipeConstants.CLIENT_SUBJECT);
	}

	if (s == null) {

	    s = PipeHelper.getClientSubject();

            if (p != null) {
                p.invocationProperties.put(PipeConstants.CLIENT_SUBJECT,s);
            }
	}
	
	return s;
    }
			
    public JAXBElement startSecureConversation(Packet packet) 
            throws WSSecureConversationException {

	PacketMessageInfo info = new PacketMapMessageInfo(packet,new Packet());
	JAXBElement token = null;

	try {

	    // gets the subject from the packet (puts one there if not found)
	    Subject clientSubject = getClientSubject(packet);

	    // put MessageInfo in properties map, since MessageInfo 
	    // is not passed to getAuthContext, key idicates function
	    HashMap map = new HashMap();
	    map.put(PipeConstants.SECURITY_TOKEN,info);

	    helper.getSessionToken(map,info,clientSubject);

	    // helper returns token in map of msgInfo, using same key
	    Object o = info.getMap().get(PipeConstants.SECURITY_TOKEN);

	    if (o != null && o instanceof JAXBElement) {
		token = (JAXBElement) o;
	    }

	} catch(Exception e) {

	    if (e instanceof WSSecureConversationException) {
		throw (WSSecureConversationException) e;
	    } else {
		throw new WSSecureConversationException
		    ("Secure Conversation failure: ", e);
	    }
	} 

	return token;
    }
}












