/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */
package com.sun.grizzly.rails;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URLConnection;
import java.nio.channels.FileChannel;
import java.nio.channels.SocketChannel;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Locale;
import java.util.logging.Level;

import org.apache.coyote.ActionCode;
import org.apache.coyote.Adapter;
import org.apache.coyote.Request;
import org.apache.coyote.Response;
import org.apache.coyote.http11.InternalInputBuffer;
import org.apache.coyote.http11.InternalOutputBuffer;
import org.apache.tomcat.util.buf.Ascii;
import org.apache.tomcat.util.buf.ByteChunk;
import org.apache.tomcat.util.buf.ByteChunk.ByteInputChannel;
import org.apache.tomcat.util.buf.MessageBytes;
import org.apache.tomcat.util.http.MimeHeaders;
import org.jruby.IRuby;
import org.jruby.RubyException;
import org.jruby.RubyIO;
import org.jruby.exceptions.RaiseException;
import org.jruby.javasupport.JavaEmbedUtils;
import org.jruby.runtime.builtin.IRubyObject;

import com.sun.enterprise.web.connector.grizzly.Constants;
import com.sun.enterprise.web.connector.grizzly.DefaultProcessorTask;
import com.sun.enterprise.web.connector.grizzly.FileCache;
import com.sun.enterprise.web.connector.grizzly.FileCacheFactory;
import com.sun.enterprise.web.connector.grizzly.SelectorThread;
import com.sun.enterprise.web.connector.grizzly.SocketChannelOutputBuffer;

/**
 * Adapter implementation that brige JRuby on Rails with Grizzly.
 *
 * @author TAKAI Naoto
 * @author Jean-Francois Arcand
 */
public class RailsAdapter implements Adapter, ByteChunk.ByteInputChannel{

    private static final String RFC_2616_FORMAT = "EEE, d MMM yyyy HH:mm:ss z";

    private final String publicDirectory;

    private RubyObjectPool pool = null;
    
    /**
     * Grizzly FileCache
     */
    private FileCache fileCache;
           
    private ByteChunk readChunk;
      
    private ByteChunk writeChunk;
    
    private Request req;
    
    private Response res;

    private RailsInputStream inputStream;
    
    
    public RailsAdapter(RubyObjectPool pool) {
        this.pool = pool;
        this.publicDirectory = pool.getRailsRoot() + "/public";
        readChunk = new ByteChunk();
        readChunk.setByteInputChannel(this);
        readChunk.setBytes(new byte[8192],0,8192);
        
        inputStream = new RailsInputStream();
    }

    public void afterService(Request req, Response res) throws Exception {
        try {
            req.action(ActionCode.ACTION_POST_REQUEST, null);
        } catch (Throwable t) {
            t.printStackTrace();
        } finally {
            // Recycle the wrapper request and response
            req.recycle();
            res.recycle();
            readChunk.recycle();
        }
    }

    public void fireAdapterEvent(String type, Object data) {
    }

    public void service(Request req, Response res) throws Exception { 
        MessageBytes mb = req.requestURI();
        ByteChunk requestURI = mb.getByteChunk();
        this.req = req;

        if (fileCache == null){                   
            fileCache = FileCacheFactory.getFactory(req.getLocalPort())
                            .getFileCache();
        }
        
        if (!fileCache.sendCache(requestURI.getBytes(), requestURI.getStart(),
                                 requestURI.getLength(), getChannel(res),
                                 keepAlive(req))){

            try {
                String uri = requestURI.toString();
                File file = new File(publicDirectory,uri);
                if (file.isDirectory()) {
                    uri += "index.html";
                    file = new File(file,uri);
                }

                if (file.canRead()) {
                    serviceFile(req, res, file);
                    fileCache.add(FileCache.DEFAULT_SERVLET_NAME,
                                  publicDirectory,
                                  uri,res.getMimeHeaders(),false);
                                  
                } else {
                    serviceRails(req, res);
                }

                res.finish();
            } catch (Exception e) {
                if (SelectorThread.logger().isLoggable(Level.SEVERE)) {
                    SelectorThread.logger().log(Level.SEVERE, e.getMessage());
                }

                throw e;
            }
        } 
    }

    private SocketChannel getChannel(Response res) {
        SocketChannelOutputBuffer buffer = (SocketChannelOutputBuffer) res.getOutputBuffer();
        SocketChannel channel = buffer.getChannel();

        return channel;
    }

    private boolean modifiedSince(Request req, File file) {
        try {
            String since = req.getMimeHeaders().getHeader("If-Modified-Since");
            if (since == null) {
                return false;
            }

            Date date = new SimpleDateFormat(RFC_2616_FORMAT, Locale.US).parse(since);
            if (date.getTime() > file.lastModified()) {
                return true;
            } else {
                return false;
            }
        } catch (ParseException e) {
            return false;
        }
    }

    private void serviceFile(Request req, Response res, File file) throws IOException, FileNotFoundException {

        if (modifiedSince(req, file)) {
            // Not Modified
            res.setStatus(304);

            return;
        }

        // set headers
        String type = URLConnection.guessContentTypeFromName(file.getName());
        res.setContentType(type);
        res.setContentLength((int) file.length());

        // comit header
        ((SocketChannelOutputBuffer) res.getOutputBuffer()).flush();

        // transfer file
        SocketChannel socketChannel = getChannel(res);
        FileInputStream stream = new FileInputStream(file);
        FileChannel fileChannel = stream.getChannel();

        long nread = 0;
        while (nread < file.length() && socketChannel.isOpen()) {
            nread += fileChannel.transferTo(nread, fileChannel.size(), socketChannel);
        }

        try {
            stream.close();
        } catch (IOException ex) {
            ;
        }

        try {
            fileChannel.close();
        } catch (IOException ex) {
            ;
        }

    }

    private void serviceRails(Request req, Response res) throws IOException {
        IRuby runtime = null;
        try {
            runtime = pool.bollowRuntime();

            if (runtime == null) {
                res.setStatus(503);
                return;
            }
            req.doRead(readChunk);

            IRubyObject reqObj = JavaEmbedUtils.javaToRuby(runtime, req);
            IRubyObject loggerObj = JavaEmbedUtils.javaToRuby(runtime, SelectorThread.logger());

            OutputStream os = 
                ((InternalOutputBuffer)res.getOutputBuffer()).getOutputStream();
            
            RubyIO iObj = new RubyIO(runtime, inputStream);
            RubyIO oObj = new RubyIO(runtime, os);

            runtime.defineReadonlyVariable("$req", reqObj);
            runtime.defineReadonlyVariable("$stdin", iObj);
            runtime.defineReadonlyVariable("$stdout", oObj);
            runtime.defineReadonlyVariable("$logger", loggerObj);

            runtime.getLoadService().load("dispatch.rb");

            res.setCommitted(true);
        } catch (RaiseException e) {
            RubyException exception = e.getException();

            System.err.println(e.getMessage());
            exception.printBacktrace(System.err);

            throw e;
        } finally {
            if (runtime != null) {
                pool.returnRuntime(runtime);
            }
        }
    }
    
    
    /**
     * Get the keep-alive header.
     */
    private boolean keepAlive(Request request){
        MimeHeaders headers = request.getMimeHeaders();

        // Check connection header
        MessageBytes connectionValueMB = headers.getValue("connection");
        if (connectionValueMB != null) {
            ByteChunk connectionValueBC = connectionValueMB.getByteChunk();
            if (findBytes(connectionValueBC, Constants.CLOSE_BYTES) != -1) {
                return false;
            } else if (findBytes(connectionValueBC, 
                                 Constants.KEEPALIVE_BYTES) != -1) {
                return true;
            }
        }
        return true;
    }    
        
    
    /**
     * Specialized utility method: find a sequence of lower case bytes inside
     * a ByteChunk.
     */
    protected int findBytes(ByteChunk bc, byte[] b) {

        byte first = b[0];
        byte[] buff = bc.getBuffer();
        int start = bc.getStart();
        int end = bc.getEnd();

        // Look for first char 
        int srcEnd = b.length;

        for (int i = start; i <= (end - srcEnd); i++) {
            if (Ascii.toLower(buff[i]) != first) continue;
            // found first char, now look for a match
            int myPos = i+1;
            for (int srcPos = 1; srcPos < srcEnd; ) {
                    if (Ascii.toLower(buff[myPos++]) != b[srcPos++])
                break;
                    if (srcPos == srcEnd) return i - start; // found it
            }
        }
        return -1;
    }

    public int realReadBytes(byte[] b, int off, int len) throws IOException {
        req.doRead(readChunk);
        return readChunk.substract(b,off,len);
    }
    
    private class RailsInputStream extends InputStream{
        public int read() throws IOException {
            return readChunk.substract();
        }
        
        public int read(byte[] b) throws IOException {
            return read(b,0,b.length);
        } 
        
        public int read(byte[] b, int off, int len) throws IOException {
            return readChunk.substract(b,off,len);
        }
    }
}
