/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */
package com.sun.enterprise.web.connector.grizzly;

import java.nio.ByteBuffer;
import java.util.logging.Level;



/**
 * Simple worker thread used for processing HTTP requests. All threads are
 * synchronized using a <code>Pipeline</code> object
 *
 * @author Jean-Francois Arcand
 */
public class WorkerThreadImpl extends Thread implements WorkerThread{

    /**
     * What will be run.
     */
    protected Runnable target;
    
    
    /**
     * The <code>ByteBuffer</code> used when <code>Task</code> are executed.
     */
    protected ByteBuffer byteBuffer;
    
    
    /**
     * The <code>Pipeline</code> on which this thread synchronize.
     */
    protected Pipeline pipeline;

    /**
     * Looing variable.
     */
    protected volatile boolean doTask = true;

    
    /**
     * The <code>ThreadGroup</code> used.
     */
    protected final static ThreadGroup threadGroup = new ThreadGroup("Grizzly");
    
    /** 
     * Create a Thread that will synchronizes/block on 
     * <code>Pipeline</code> instance.
     */
    public WorkerThreadImpl(ThreadGroup threadGroup, Runnable runnable){
        super(threadGroup, runnable);                    
        setDaemon(true);
        target = runnable;
    }    
    
    
    /** 
     * Create a Thread that will synchronizes/block on 
     * <code>Pipeline</code> instance.
     */
    public WorkerThreadImpl(Pipeline pipeline, String name){
        super(threadGroup, name);                    
        this.pipeline = pipeline;
        setDaemon(true);        
    }

    
    /**
     * Execute a <code>Task</code>.
     */
    public void run(){

        if ( target != null ){
            target.run();
            return;
        }
        
        while (doTask) {
            try{
                // Wait for a Task to be added to the pipeline.
                Task t = pipeline.getTask();
                if ( t != null){
                    t.run();                
                    t = null;
                }
            } catch (Throwable t) {
                if ( byteBuffer != null){
                    byteBuffer.clear();
                }
                SelectorThread.logger().log(Level.FINE,
                        "workerThread.httpException",t);
            }
        }
    }
    
    
    /**
     * Stop this thread. If this Thread is performing atask, the task will be
     * completed.
     */
    public void terminate(){
        doTask = false;
    }
    
    
    /**
     * Set the <code>ByteBuffer</code> shared this thread
     */
    public void setByteBuffer(ByteBuffer byteBuffer){
        this.byteBuffer = byteBuffer;
    }
    
    
    /**
     * Return the <code>ByteBuffer</code> shared this thread
     */
    public ByteBuffer getByteBuffer(){
        return byteBuffer;
    }

}

