/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.enterprise.web.connector.grizzly.comet;

import com.sun.enterprise.web.connector.grizzly.AsyncTask;
import com.sun.enterprise.web.connector.grizzly.Pipeline;
import com.sun.enterprise.web.connector.grizzly.SelectorThread;
import com.sun.enterprise.web.connector.grizzly.async.AsyncProcessorTask;
import java.io.IOException;
import java.nio.channels.SelectionKey;
import java.util.Iterator;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * Main class allowing Comet support on top of Grizzly Asynchronous
 * Request Processing mechanism. This class is the entry point to any
 * component interested to execute Comet request style. Components can be
 * Servlets, JSP, JSF or pure Java class. A component interested to support
 * Comet request must do:
 *
 * (1) First, register the cometContext path on which Comet support will be applied:
 *     <code>CometEngine cometEngine = CometEngine.getEngine()</code>
 *     <code>CometContext cometContext = cometEngine.register(contextPath)</code>
 * (2) Second, add an instance of <code>CometHandler</code> to the
 *     <code>CometContext</code> returned by the register method:
 *     <code>cometContext.addCometHandler(handler);</code>
 * (3) Finally, you can notify other <code>CometHandler</code> by doing:
 *     <code>cometContext.notify(Object)(handler);</code>
 *
 * You can also select the stage where the request polling happens when
 * registering the cometContext path (see register(String,int);
 *
 *
 * @author Jeanfrancois Arcand
 */
public class CometEngine {
    
    /**
     * The token used to support BEFORE_REQUEST_PROCESSING polling.
     */
    public final static int BEFORE_REQUEST_PROCESSING = 0;
    
    
    /**
     * The token used to support AFTER_SERVLET_PROCESSING polling.
     */
    public final static int AFTER_SERVLET_PROCESSING = 1;
    
    
    /**
     * The token used to support BEFORE_RESPONSE_PROCESSING polling.
     */
    public final static int AFTER_RESPONSE_PROCESSING = 2;
    
    
    /**
     * Main logger
     */
    private final static Logger logger = SelectorThread.logger();
    
    
    /**
     * The <code>Pipeline</code> used to execute <code>CometTask</code>
     */
    private Pipeline pipeline;
    
    
    /**
     * The single instance of this class.
     */
    private static CometEngine cometEngine;
    
    
    /**
     * The current active <code>CometContext</code> keyed by context path.
     */
    private ConcurrentHashMap<String,CometContext> activeContexts;
    
    
    /**
     * Cache of <code>CometTask</code> instance
     */
    private ConcurrentLinkedQueue<CometTask> cometTasks;
    
    
    /**
     * Cache of <code>CometContext</code> instance.
     */
    private ConcurrentLinkedQueue<CometContext> cometContexts;
    
    
    /**
     * The <code>CometSelector</code> used to poll requests.
     */
    private CometSelector cometSelector;
    
    
    /**
     * Updated keys list.
     */
    private ConcurrentLinkedQueue<SelectionKey> updatedKeys =
            new ConcurrentLinkedQueue<SelectionKey>();
    
    
    // --------------------------------------------------------------------- //
    
    
    /**
     * Creat a singleton and initialize all lists required. Also create and
     * start the <code>CometSelector</code>
     */
    private CometEngine() {
        activeContexts = new ConcurrentHashMap<String,CometContext>();
        cometTasks = new ConcurrentLinkedQueue<CometTask>();
        cometContexts = new ConcurrentLinkedQueue<CometContext>();
        
        cometSelector = new CometSelector(this);
        cometSelector.start();
    }
    
    
    /**
     * Return a singleton of this Class.
     * @return CometEngine the singleton.
     */
    public synchronized final static CometEngine getEngine(){
        if (cometEngine == null) {
            cometEngine = new CometEngine();
        }
        return cometEngine;
    }
    
    
    /**
     * Unregister the <code>CometHandler</code> to the list of the
     * <code>CometContext</code>.
     */
    public CometContext unregister(String contextPath){
        CometContext cometContext = activeContexts.get(contextPath);
        try{
            cometContext.notify(cometContext,CometEvent.TERMINATE);
        } catch (IOException ex){
            logger.log(Level.WARNING,"unregister",ex);
        }
        finalizeContext(cometContext);
        
        return activeContexts.remove(contextPath);
    }
    
    
    /**
     * Register a context path with this <code>CometEngine</code>. The
     * <code>CometContext</code> returned will be of type
     * AFTER_SERVLET_PROCESSING, which means the request target (most probably
     * a Servlet) will be executed first and then polled.
     * @param contextPath the context path used to create the
     *        <code>CometContext</code>
     * @return CometContext a configured <code>CometContext</code>.
     */
    public CometContext register(String contextPath){
        return register(contextPath,AFTER_SERVLET_PROCESSING);
    }
    
    
    /**
     * Register a context path with this <code>CometEngine</code>. The
     * <code>CometContext</code> returned will be of type
     * <code>type</code>.
     * @param contextPath the context path used to create the
     *        <code>CometContext</code>
     * @return CometContext a configured <code>CometContext</code>.
     */
    public CometContext register(String contextPath, int type){
        CometContext cometContext = activeContexts.get(contextPath);
        if (cometContext == null){
            cometContext = cometContexts.poll();
            if ( cometContext == null){
                cometContext = new CometContext(contextPath,type);
                cometContext.setCometSelector(cometSelector);
            }
            
            activeContexts.put(contextPath,cometContext);
        }
        return cometContext;
    }
    
    
    /**
     * Handle an interrupted(or polled) request by matching the current context
     * path with the registered one.
     * If required, the bring the target component (Servlet) to the proper
     * execution stage and then notify the <code>CometHandler</code>
     * @param apt the current apt representing the request.
     * @return boolean true if the request can be polled.
     */
    protected boolean handle(AsyncProcessorTask apt) throws IOException{
        
        if (pipeline == null){
            pipeline = apt.getPipeline();
        }
        
        String contextPath = apt.getProcessorTask().getRequestURI();
        CometContext cometContext = null;       
        if (contextPath != null){
            cometContext = activeContexts.get(contextPath);
        }
        
        if (cometContext == null){
            apt.getProcessorTask().invokeAdapter();
            return false;
        }
        
        SelectionKey key = apt.getProcessorTask().getSelectionKey();
        
        cometContext.attachKeyToThreadId(key);
        CometTask cometTask = getCometTask(cometContext,key);
        cometTask.setSelectorThread(apt.getProcessorTask().getSelectorThread());
        executeServlet(cometContext,apt);
        
        boolean interrupt = true;
        if (updatedKeys.remove(key)) {
            cometContext.addAsyncProcessorTask(apt);
            cometContext.initialize(key);
            cometTask.setExpirationDelay(cometContext.getExpirationDelay());
            cometSelector.registerKey(key,cometTask);
        } else {
            interrupt = false;
            cometTask.recycle();
            cometTasks.offer(cometTask);
        }
        return interrupt;
    }
    
    
    /**
     * Return a clean and configured <code>CometTask</code>
     * @param cometContext the CometContext to clean
     * @param key The current <code>SelectionKey</code>
     * @return a new CometContext
     */
    protected CometTask getCometTask(CometContext cometContext,SelectionKey key){
        CometTask cometTask = cometTasks.poll();
        if (cometTask == null){
            cometTask = new CometTask();
        }
        cometTask.setCometContext(cometContext);
        cometTask.setSelectionKey(key);
        cometTask.setCometSelector(cometSelector);
        cometTask.setPipeline(pipeline);
        return cometTask;
    }
    
    
    /**
     * Cleanup the <code>CometContext</code>
     * @param cometContext the CometContext to clean
     */
    private void finalizeContext(CometContext cometContext) {
        Iterator<String> iterator = activeContexts.keySet().iterator();
        String contextPath;
        while(iterator.hasNext()){
            contextPath = iterator.next();
            if ( activeContexts.get(contextPath).equals(cometContext) ){
                activeContexts.remove(contextPath);
                break;
            }
        }
        
        ConcurrentLinkedQueue<AsyncProcessorTask> asyncTasks =
                cometContext.getAsyncProcessorTask();
        for (AsyncProcessorTask apt: asyncTasks){
            flushResponse(apt);
        }
        cometContext.recycle();
        cometContexts.offer(cometContext);
    }
    
    
    /**
     * Return the <code>CometContext</code> associated with the cometContext path.
     * XXX: This is not secure as a you can get a CometContext associated
     * with another cometContext path. But this allow interesting application...
     * MUST BE REVISTED.
     * @param contextPath the request's cometContext path.
     */
    public CometContext getCometContext(String contextPath){
        return activeContexts.get(contextPath);
    }
    
    
    /**
     * The <code>CometSelector</code> is expiring idle <code>SelectionKey</code>,
     * hence we need to resume the current request.
     * @param key the expired SelectionKey
     */
    protected void interrupt(SelectionKey key) {
        CometTask cometTask = (CometTask)key.attachment();
        
        // Make sure there is no leak.
        updatedKeys.remove(key);
        
        if (cometTask == null)
            throw new IllegalStateException("cometTask cannot be null");
        
        CometContext cometContext = cometTask.getCometContext();
        Iterator<AsyncProcessorTask> iterator =
                cometContext.getAsyncProcessorTask().iterator();
        
        while (iterator.hasNext()){
            AsyncProcessorTask apt = iterator.next();
            if (apt.getProcessorTask().getSelectionKey() 
                    == cometTask.getSelectionKey()){
                flushResponse(apt);
                cometContext.getAsyncProcessorTask().remove(apt);
                cometTask.getSelectionKey().attach(0L);
                apt.getAsyncExecutor().getAsyncHandler()
                    .removeFromInterruptedQueue(apt);
                break;
            }
        }
        cometTask.recycle();
        cometTasks.offer(cometTask);
    }
    
    
    /**
     * Complete the asynchronous request.
     */
    private void flushResponse(AsyncProcessorTask apt){
        apt.setStage(AsyncTask.POST_EXECUTE);
        try{
            apt.doTask();
        } catch (IOException ex) {
            logger.log(Level.SEVERE,"interrupt",ex);
        }
    }
    
    
    /**
     * Bring the cometContext path target (most probably a Servlet) to the processing
     * stage we need for Comet request processing.
     * @param cometContext The CometContext associated with the Servlet
     * @param apt the AsyncProcessorTask
     */
    private void executeServlet(CometContext cometContext,
            AsyncProcessorTask apt){
        
        switch (cometContext.type){
            case BEFORE_REQUEST_PROCESSING:
                apt.setStage(AsyncTask.PRE_EXECUTE);
                break;
            case AFTER_SERVLET_PROCESSING:
                apt.getProcessorTask().invokeAdapter();
                return;
            case AFTER_RESPONSE_PROCESSING:
                apt.setStage(AsyncTask.POST_EXECUTE);
                break;
            default:
                throw new IllegalStateException("Invalid state");
        }
        
        try{
            apt.doTask();
        } catch (IOException ex){
            logger.log(Level.SEVERE,"executeServlet",ex);
        }
    }
    
    
    /**
     * Add a key to the updated list.
     */
    protected boolean addUpdateKey(SelectionKey key){
        return updatedKeys.offer(key);
    }
}
