/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.enterprise.web.connector.grizzly.comet;

import com.sun.enterprise.web.connector.grizzly.Constants;
import com.sun.enterprise.web.connector.grizzly.SelectorThread;
import java.io.IOException;
import java.nio.channels.CancelledKeyException;
import java.nio.channels.SelectionKey;
import java.nio.channels.Selector;
import java.nio.channels.SocketChannel;
import java.util.Iterator;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * NIO <code>Selector</code> allowing <code>CometHandler</code> to receive
 * non-blocking requests bytes during request polling.
 *
 * @author Jeanfrancois Arcand
 */
public class CometSelector {

    /**
     * The <code>CometEngine</code> singleton
     */
    protected CometEngine cometEngine;
     
    
    /**
     * The <code>java.nio.channels.Selector</code>
     */
    private Selector selector;
       
    
    /**
     * Logger.
     */
    private Logger logger = SelectorThread.logger();
    
    
    /**
     * The list of <code>SelectionKey</code> to register with the 
     * <code>Selector</code>
     */
    private ConcurrentHashMap<SelectionKey,CometTask> keysToRegister 
            = new ConcurrentHashMap<SelectionKey,CometTask>();

    
    /**
     * New <code>CometSelector</code>
     * @param cometEngine The <code>CometEngine</code> singleton 
     */
    public CometSelector(CometEngine cometEngine) {
        this.cometEngine = cometEngine;
    }

    
    /**
     * Start the <code>java.nio.channels.Selector</code> running on its 
     * Thread.
     */
    public void start(){
        new Thread("CometSelector"){
            {
               setDaemon(true);                
            }
            
            public void run(){       
                try{
                    selector = Selector.open();
                } catch(IOException ex){
                    // Most probably a fd leak.
                    logger.log(Level.SEVERE,"CometSelector.open()",ex);
                    return;
                }
                while (true){
                    SelectionKey key = null;
                    Set readyKeys;
                    Iterator<SelectionKey> iterator;
                    int selectorState; 

                    try{
                        selectorState = 0;
  
                        try{
                            selectorState = selector.select(1000);
                        } catch (CancelledKeyException ex){
                            ;
                        }

                        readyKeys = selector.selectedKeys();
                        iterator = readyKeys.iterator();
                        while (iterator.hasNext()) {
                            key = iterator.next();
                            iterator.remove();
                            if (key.isValid()) {
                                key.interestOps(key.interestOps() 
                                    & (~SelectionKey.OP_READ));
                                ((CometTask)key.attachment()).execute();
                            } else {
                                cancelKey(key);
                            }
                        }
                        
                        Iterator<SelectionKey> keys = 
                                keysToRegister.keySet().iterator();
                        SelectionKey mainKey;
                        SocketChannel channel;
                        CometTask cometTask;
                        while (keys.hasNext()){
                            mainKey = keys.next();
                            channel =  (SocketChannel)mainKey.channel();
                            if (mainKey.isValid() && channel.isOpen()) {
                                key = channel
                                    .register(selector,SelectionKey.OP_READ);  
                                cometTask = keysToRegister.remove(mainKey);
                                cometTask.setCometKey(key);
                                key.attach(cometTask); 
                                keys.remove();
                            } 
                        }                             
                        expireIdleKeys();
                        
                        if (selectorState <= 0){
                            selector.selectedKeys().clear();
                        }
                    } catch (Throwable t){
                        if (key != null){
                            key.attach(null);
                            key.cancel();
                        }
                        logger.log(Level.INFO,"CometSelector",t);
                    }      
                }   
            }
        }.start();
    }   
    
    
    /**
     * Expires registered <code>SelectionKey</code>. If a 
     * <code>SelectionKey</code> is expired, the request will be resumed and the 
     * HTTP request will complete,
     */
    protected void expireIdleKeys(){       
        Set<SelectionKey> readyKeys = selector.keys();
        if (readyKeys.isEmpty()){
            return;
        }
        long current = System.currentTimeMillis();
        Iterator<SelectionKey> iterator = readyKeys.iterator();
        SelectionKey key;
        while (iterator.hasNext()) {
            key = iterator.next();    
            CometTask cometTask = (CometTask)key.attachment();
            
            if (cometTask == null) return;

            long expire = cometTask.getExpireTime();
            if (current - expire >= cometTask.getExpirationDelay()) {
                cancelKey(key);
            } 
        }                    
    }  
    
    
    /**
     * Cancel a <code>SelectionKey</code>, and delegate the request
     * polling interruption to the <code>CometEngine</code>
     * @param key the expired <code>SelectionKey</code>
     */
    public void cancelKey(SelectionKey key){
        CometTask cometTask = (CometTask)key.attachment();
        key.cancel();
        cometTask.getCometContext().interrupt(cometTask.getSelectionKey());        
        cometEngine.interrupt(key);         
        key.attach(null);
        selector.wakeup();
    }
    
    
    /**
     * Register the <code>SelectionKey/code> to the <code>Selector</code>. We
     * cannot register the <code>SelectionKey/code> directy on the
     * <code>Selector</code> because there is a deadlock in the VM (see bug XXX).
     */
    public void registerKey(SelectionKey key, CometTask cometTask){
        cometTask.setExpireTime(System.currentTimeMillis());
        keysToRegister.put(key, cometTask);
        selector.wakeup();
    }
    
    
    /**
     * Wakes up the <code>java.nio.channels.Selector</code> 
     */
    public void wakeup(){
        selector.wakeup();
    }
    
}
