/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.enterprise.web.portunif;

import com.sun.enterprise.web.connector.grizzly.SelectorThread;
import com.sun.enterprise.web.connector.grizzly.ssl.SSLSelectorThread;
import com.sun.enterprise.web.connector.grizzly.ssl.SSLUtils;
import com.sun.enterprise.web.portunif.util.ProtocolInfo;
import java.io.EOFException;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.SelectionKey;
import java.nio.channels.SocketChannel;
import java.util.Enumeration;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLEngine;
import javax.net.ssl.SSLEngineResult.HandshakeStatus;

/**
 * <code>ProtocolFinder</code> that will first try to execute an handshake.
 * If the handshake is succesfull, the https protocol will be assumed. If the 
 * handshake fail, the http protocol will be assumed.
 *
 * This object shoudn't be called by several threads simultaneously.
 *
 * @author Jeanfrancois Arcand
 */
public class TlsProtocolFinder implements ProtocolFinder{
    
    /**
     * Decrypted ByteBuffer default size.
     */
    private final static int appBBSize = 5 * 4096;
    
    // ---------------------------------------------------------------------- //
    
    
    public TlsProtocolFinder() {
    }
    
    
    /**
     * Try to initialize an SSL|TLS handshake to determine if https 
     */
    public void find(ProtocolInfo protocolInfo) throws IOException{
        if (protocolInfo.sslContext == null){
            return;
        }

        SelectionKey key = protocolInfo.key;    
        SocketChannel socketChannel = (SocketChannel)key.channel();

        SSLEngine sslEngine = protocolInfo.sslEngine;        
        if (protocolInfo.sslContext != null && sslEngine == null) {
            sslEngine = protocolInfo.sslContext.createSSLEngine();
            sslEngine.setUseClientMode(false);
            sslEngine.setNeedClientAuth(true);
            sslEngine.setWantClientAuth(true);
        }
        
        ByteBuffer inputBB = null;
        ByteBuffer outputBB =  null;
        ByteBuffer byteBuffer =  null;  
        int inputBBSize = sslEngine.getSession().getPacketBufferSize();        
        if (protocolInfo.inputBB == null 
                || (inputBB != null && inputBBSize > inputBB.capacity())){
            inputBB = ByteBuffer.allocate(inputBBSize * 2);
            outputBB = ByteBuffer.allocate(inputBBSize * 2);
            byteBuffer = ByteBuffer.allocate(inputBBSize * 2);  

            inputBBSize = sslEngine.getSession().getApplicationBufferSize();
            if ( inputBBSize > byteBuffer.capacity() ) {
                ByteBuffer newBB = ByteBuffer.allocate(inputBBSize);
                byteBuffer.flip();
                newBB.put(byteBuffer);
                byteBuffer = newBB;
            }   

        } else {
            inputBB = protocolInfo.inputBB;
            outputBB = protocolInfo.outputBB;
            byteBuffer = protocolInfo.byteBuffer;                     
        }
        outputBB.position(0);
        outputBB.limit(0); 
        
        if ( protocolInfo.bytesRead > 0){
            inputBB.put((ByteBuffer)protocolInfo.byteBuffer.flip());
        }

        HandshakeStatus handshakeStatus = HandshakeStatus.NEED_UNWRAP;
        boolean OK = true;  
        if (protocolInfo.handshake){
            try{
                byteBuffer = SSLUtils.doHandshake(key,byteBuffer,inputBB,outputBB,
                                                  sslEngine,handshakeStatus,
                                                  SSLUtils.getReadTimeout());
                protocolInfo.handshake = false;
            } catch (EOFException ex) {
                ; // DO nothing, as the client closed the connection
            } catch (Throwable ex) {            
                // An exception means the handshake failed.
                OK = false;
                byteBuffer.put(inputBB);            
            }
        }
          
        try {
            if (OK) {  
                int byteRead = SSLUtils.doRead(key,inputBB,sslEngine,
                        SSLUtils.getReadTimeout());
                if (byteRead > -1){
                    protocolInfo.byteBuffer = 
                            SSLUtils.unwrapAll(byteBuffer,inputBB,sslEngine);
                    protocolInfo.bytesRead = byteBuffer.position();
                    
                    // Not enough bytes to decrypt the request.
                    if (protocolInfo.bytesRead == 0){
                        OK = false;
                    }                  
                } else {
                   throw new EOFException(); 
                }
            }
        } finally {
            String protocol = protocolInfo.protocol;  
            protocolInfo.inputBB = inputBB;
            protocolInfo.byteBuffer = byteBuffer;
            protocolInfo.outputBB = outputBB; 
            protocolInfo.sslEngine = sslEngine;     
            protocolInfo.isSecure = true;            
            if (!OK){
                protocolInfo.bytesRead = byteBuffer.position();
                protocolInfo.protocol = null;
                protocolInfo.isSecure = false;
            }
            protocolInfo.socketChannel = (SocketChannel)key.channel();                
        }
        return;
    }
}
