/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the "License").  You may not use this file except 
 * in compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * glassfish/bootstrap/legal/CDDLv1.0.txt or 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html. 
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * HEADER in each file and include the License file at 
 * glassfish/bootstrap/legal/CDDLv1.0.txt.  If applicable, 
 * add the following below this CDDL HEADER, with the 
 * fields enclosed by brackets "[]" replaced with your 
 * own identifying information: Portions Copyright [yyyy] 
 * [name of copyright owner]
 */
// Copyright (c) 1998, 2007, Oracle. All rights reserved.  
package oracle.toplink.essentials.internal.expressions;

import java.util.*;
import oracle.toplink.essentials.expressions.*;
import oracle.toplink.essentials.queryframework.ObjectBuildingQuery;

/**
 * <b>Purpose:</b>Represents The FOR UPDATE OF fine-grained pessimistically
 * locking clause.
 * @author  Stephen McRitchie
 * @since   Oracle Toplink 10g AS
 */
public class ForUpdateOfClause extends ForUpdateClause {
    protected Vector lockedExpressions;

    public ForUpdateOfClause() {
    }

    public void addLockedExpression(ObjectExpression expression) {
        getLockedExpressions().addElement(expression);
    }

    public Vector getLockedExpressions() {
        if (lockedExpressions == null) {
            lockedExpressions = oracle.toplink.essentials.internal.helper.NonSynchronizedVector.newInstance();
        }
        return lockedExpressions;
    }

    public boolean isForUpdateOfClause() {
        return true;
    }

    public boolean isReferenceClassLocked() {
        if (lockedExpressions == null) {
            return false;
        }

        // Normally the expressionBuilder is stored first but not necessarily 
        // when a child ForUpdateOfClause is built for a nested query, or if a 
        //user made this clause.
        for (int i = 0; i < lockedExpressions.size(); i++) {
            if (((Expression)lockedExpressions.elementAt(i)).isExpressionBuilder()) {
                return true;
            }
        }
        return false;
    }

    public void setLockedExpressions(Vector lockedExpressions) {
        this.lockedExpressions = lockedExpressions;
    }

    public void setLockMode(short lockMode) {
        this.lockMode = lockMode;
    }

    /**
     * INTERNAL:
     * Prints the as of clause for an expression inside of the FROM clause.
     */
    public void printSQL(ExpressionSQLPrinter printer, SQLSelectStatement statement) {
        // assert(lockedExpressions != null && lockedExpressions.size() > 0);
        // assert(	getLockMode() == ObjectBuildingQuery.LOCK || 
        //			getLockMode() == ObjectBuildingQuery.LOCK_NOWAIT);
        ExpressionBuilder clonedBuilder = statement.getBuilder();

        printer.printString(printer.getSession().getPlatform().getSelectForUpdateOfString());

        printer.setIsFirstElementPrinted(false);
        for (Enumeration enumtr = getLockedExpressions().elements(); enumtr.hasMoreElements();) {
            ObjectExpression next = (ObjectExpression)enumtr.nextElement();

            // Neccessary as this was determined in query framework.
            next = (ObjectExpression)next.rebuildOn(clonedBuilder);
            next.writeForUpdateOfFields(printer, statement);
        }
        if (lockMode == ObjectBuildingQuery.LOCK_NOWAIT) {
            printer.printString(printer.getSession().getPlatform().getSelectForUpdateNoWaitString());
        }
    }
}
