/*
 * Copyright (c) 2026, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

package com.sun.javafx.css.parser;

public final class CssNumberParser {

    private CssNumberParser() {}

    // Up to 19 decimal digits fit in an unsigned 64-bit long
    private static final int MAX_SIG_DIGITS = 19;

    // Saturation: beyond this, doubles overflow/underflow anyway
    private static final int EXP_LIMIT = 10000;

    /**
     * Parses a number according to the W3C "Consume a number" algorithm. A number can have integer
     * and fractional parts, and it supports E notation (exponential).
     * <p>
     * This method operates on the substring {@code [start, end)} of {@code s}, which is expected to
     * contain the string representation of a number without leading or trailing whitespace.
     * <p>
     * This implementation is allocation-free for significands with less than 20 digits and returns
     * the nearest representable double value of the input string, with ties rounding to even.
     *
     * @throws NumberFormatException if the substring does not consist entirely of a valid number representation
     * @see <a href="https://www.w3.org/TR/css-syntax-3/#consume-number">Consume a number</a>
     */
    public static double parseDouble(String s, int start, int end) {
        int p = start;
        boolean negative = false;

        // Optional sign
        if (p < end) {
            char c = s.charAt(p);
            if (c == '+' || c == '-') {
                negative = (c == '-');
                p++;
            }
        }

        long significand = 0;
        int sigDigits = 0;
        int exp10 = 0; // base-10 exponent applied to significand
        int digitsConsumed = 0; // digits in the main number (excluding exponent)
        boolean seenNonZero = false;
        boolean truncated = false;

        // While the next input code point is a digit, consume and append.
        while (p < end) {
            char c = s.charAt(p);
            if (c >= '0' && c <= '9') {
                digitsConsumed++;
                int d = c - '0';
                if (d != 0 || seenNonZero) {
                    seenNonZero = true;
                    if (sigDigits < MAX_SIG_DIGITS) {
                        // May go negative as signed, but bit pattern is correct uint64
                        significand = significand * 10L + d;
                        sigDigits++;
                    } else {
                        if (d != 0) {
                            truncated = true;
                        }

                        exp10++; // truncate extra integer digits by increasing the exponent
                    }
                }

                // Leading zeros in the integer part don't affect significand or exponent.
                p++;
            } else {
                break;
            }
        }

        // If '.' followed by a digit, consume fractional digits.
        if (p + 1 < end && s.charAt(p) == '.') {
            char next = s.charAt(p + 1);
            if (next >= '0' && next <= '9') {
                p++; // consume '.'
                while (p < end) {
                    char c = s.charAt(p);
                    if (c >= '0' && c <= '9') {
                        digitsConsumed++;
                        int d = c - '0';
                        if (d != 0 || seenNonZero) {
                            seenNonZero = true;
                            if (sigDigits < MAX_SIG_DIGITS) {
                                significand = significand * 10L + d;
                                sigDigits++;
                                exp10--; // appended a fractional digit
                            } else if (d != 0) {
                                truncated = true;
                            }
                        } else {
                            // Leading zeros immediately after the decimal point shift the exponent.
                            exp10--;
                        }

                        p++;
                    } else {
                        break;
                    }
                }
            }
        }

        // Exponent part (E/e, optional sign, then a digit)
        if (p < end) {
            char e = s.charAt(p);
            if (e == 'E' || e == 'e') {
                boolean expNegative = false;
                int q = p + 1;
                if (q < end) {
                    char sign = s.charAt(q);
                    if (sign == '+' || sign == '-') {
                        expNegative = (sign == '-');
                        q++;
                    }
                }

                if (q < end) {
                    char firstExpDigit = s.charAt(q);
                    if (firstExpDigit >= '0' && firstExpDigit <= '9') {
                        int exp = 0;
                        while (q < end) {
                            char c = s.charAt(q);
                            if (c >= '0' && c <= '9') {
                                if (exp < EXP_LIMIT) {
                                    exp = exp * 10 + (c - '0');
                                    if (exp > EXP_LIMIT) {
                                        exp = EXP_LIMIT;
                                    }
                                }

                                q++;
                            } else {
                                break;
                            }
                        }

                        p = q;
                        exp10 += expNegative ? -exp : exp;
                    }
                }
            }
        }

        // At least one digit must have been consumed from either the integer or fractional part.
        if (digitsConsumed == 0 || p != end) {
            throw new NumberFormatException("Invalid number: " + s.substring(start, end));
        }

        // If the numeric value is zero, preserve the sign.
        if (significand == 0) {
            return negative ? -0.0 : 0.0;
        }

        // Use Lemire's algorithm if possible, otherwise fall back to Double.parseDouble()
        if (!truncated && exp10 >= -342 && exp10 <= 308) {
            double value = convertToNearestDouble(significand, exp10);
            return negative ? -value : value;
        }

        return Double.parseDouble(s.substring(start, end));
    }

    /**
     * Converts a decimal floating-point number {@code w*10^q} to the nearest representable double.
     * <p>
     * Based on Algorithm 1, "Number Parsing at a Gigabyte per Second" (Lemire, 2021), p. 8, without
     * the fallback condition; see "Fast Number Parsing Without Fallback" (Mushtak, Lemire, 2022).
     * <p>
     * Precondition: {@code w != 0} and {@code q in [-342, 308]}
     *
     * @see <a href="https://arxiv.org/pdf/2101.11408">Number Parsing at a Gigabyte per Second</a>
     * @see <a href="https://arxiv.org/pdf/2212.06644">Fast Number Parsing Without Fallback</a>
     */
    private static double convertToNearestDouble(long w, int q) {
        // Normalize w to [2^63, 2^64), see Algorithm 1 lines 3-4
        int l = Long.numberOfLeadingZeros(w);
        long wNorm = w << l;

        // Load 128-bit T[q] as two 64-bit words
        int idx = (q + 342) << 1;
        long tHi = T[idx];
        long tLo = T[idx + 1];

        // z = (T[q] * wNorm) >> 64 (a 128-bit value split into zHi/zLo), see line 5
        long p1Hi = Math.unsignedMultiplyHigh(tLo, wNorm);
        long p2Hi = Math.unsignedMultiplyHigh(tHi, wNorm);
        long p2Lo = tHi * wNorm;
        long zLo = p2Lo + p1Hi;
        long carry = ((p2Lo & p1Hi) | ((p2Lo | p1Hi) & ~zLo)) >>> 63;
        long zHi = p2Hi + carry;

        // u = most significant bit of z (line 8)
        long u = zHi >>> 63;

        // m = most significant 54 bits of z, ignoring an eventual leading zero bit (line 7)
        long m = zHi >>> (u + 9);

        // p = floor(log2(10) * q) + 63 - l + u, with log2(10) approximated as 217706/2^16 (line 9)
        long p = ((217706L * (long)q) >> 16) + 63 - l + u;

        // Too small => underflow to zero (line 10)
        if (p <= -1022 - 64) {
            return 0;
        }

        // Subnormals (lines 11-15)
        if (p <= -1022) {
            m >>>= -1022 - p;

            if ((m & 1L) != 0) {
                m++;
            }

            m >>>= 1;
            return Double.longBitsToDouble(m);
        }

        // Round ties to even (lines 16-18)
        if (Long.compareUnsigned(zLo, 1L) <= 0 && (m & 1L) != 0 && ((m >>> 1) & 1L) == 0 && q >= -4 && q <= 23) {
            long quo = Long.divideUnsigned(zHi, m);
            long rem = Long.remainderUnsigned(zHi, m);
            if (rem == 0 && quo != 0 && (quo & (quo - 1)) == 0) {
                m--;
            }
        }

        // Round the binary significand (line 19)
        if ((m & 1L) != 0) {
            m++;
        }

        m >>>= 1;

        // Handle significand overflow (line 20)
        if (m == (1L << 53)) {
            m >>>= 1;
            p += 1;
        }

        // Overflow => infinity (line 21)
        if (p > 1023) {
            return Double.POSITIVE_INFINITY;
        }

        return normalDouble(m, p);
    }

    // Packs the specified mantissa and exponent into a double
    private static double normalDouble(long m, long p) {
        long expBits = p + 1023;
        long fracBits = m & ((1L << 52) - 1);
        long bits = (expBits << 52) | fracBits;
        return Double.longBitsToDouble(bits);
    }

    /**
     * 128-bit reciprocal and normalized powers of five, stored as two 64-bit words.
     * See "Table Generation Script" in "Number Parsing at a Gigabyte per Second" (Lemire, 2021), p. 32
     *
     * @see <a href="https://arxiv.org/pdf/2101.11408">Number Parsing at a Gigabyte per Second</a>
     */
    private static final long[] T = new long[] {
        -1228264617323800998L, 1242899115359157055L, -7685194413468457480L, 5388497965526861063L,
        -4994806998408183946L, 6735622456908576329L, -1631822729582842029L, -803843965719055396L,
        -7937418233630358124L, 8720969558280366185L, -5310086773610559751L, -7545532125859093884L,
        -2025922448585811785L, -208543120469091547L, -8183730558007214222L, -130339450293182217L,
        -5617977179081629873L, -4774610331293865675L, -2410785455424649437L, -5968262914117332094L,
        -8424269937281487754L, 5493207715531443249L, -5918651403174471789L, -2356862392440471747L,
        -2786628235540701832L, -2946077990550589683L, -8659171674854020501L, -8758827771735200408L,
        -6212278575140137722L, 7498209359040551106L, -3153662200497784248L, 149389661945913074L,
        -8888567902952197011L, 93368538716195671L, -6499023860262858360L, 4728396691822632493L,
        -3512093806901185046L, 5910495864778290617L, -9112587656954322510L, 8305745933913819539L,
        -6779048552765515233L, 1158810380537498616L, -3862124672529506138L, -3163173042755514634L,
        -215969822234494768L, -8565652321871781196L, -7052510166537641086L, 6175682344898606512L,
        -4203951689744663454L, -1503769105731517667L, -643253593753441413L, -6491397400591784988L,
        -7319562523736982739L, 5166248661484910190L, -4537767136243840520L, -7377247228426025974L,
        -1060522901877412746L, -4609873017105144563L, -7580355841314464822L, 4036358391950366504L,
        -4863758783215693124L, -4177924046916817678L, -1468012460592228501L, -610719040218634194L,
        -7835036815511224669L, 8841672636718129437L, -5182110000961642932L, 6440404777470273892L,
        -1865951482774665761L, 8050505971837842365L, -8083748704375247957L, -6497648813669818282L,
        -5492999862041672042L, -8122061017087272852L, -2254563809124702148L, 3682481783923072647L,
        -8326631408344020699L, -6921820921902855404L, -5796603242002637969L, 571095884476206553L,
        -2634068034075909558L, -3897816162832129712L, -8563821548938525330L, -4741978110983775022L,
        -6093090917745768758L, 7907585416552444934L, -3004677628754823043L, 661109733835780360L,
        -8795452545612846258L, 2719036592861056677L, -6382629663588669919L, -5824576295778454962L,
        -3366601061058449494L, 1942651667131707105L, -9021654690802612790L, 5825843310384704845L,
        -6665382345075878084L, -1941067898873894752L, -3720041912917459700L, 2185351144835019464L,
        -38366372719436721L, 2731688931043774330L, -6941508010590729807L, 8624834609543440812L,
        -4065198994811024355L, -3054014793352862697L, -469812725086392539L, 5405853545163697437L,
        -7211161980820077193L, 5684501474941004850L, -4402266457597708587L, 2493940825248868159L,
        -891147053569747830L, 7729112049988473103L, -7474495936122174250L, -9004363024039368023L,
        -4731433901725329908L, 2579604275232953683L, -1302606358729274481L, 3224505344041192104L,
        -7731658001846878407L, 8932844867666826921L, -5052886483881210105L, -2669001970698630061L,
        -1704422086424124727L, -3336252463373287576L, -7982792831656159810L, 2526528228819083169L,
        -5366805021142811859L, -6065211750830921846L, -2096820258001126919L, 1641857348316123500L,
        -8228041688891786181L, -5891368184943504669L, -5673366092687344822L, -7364210231179380836L,
        -2480021597431793123L, 4629795266307937667L, -8467542526035952558L, 5199465050656154994L,
        -5972742139117552794L, -2724040723534582065L, -2854241655469553088L, -8016736922845615486L,
        -8701430062309552536L, 6518754469289960081L, -6265101559459552766L, 8148443086612450102L,
        -3219690930897053053L, 962181821410786819L, -8929835859451740015L, -1704479370831952190L,
        -6550608805887287114L, 7092772823314835570L, -3576574988931720989L, -357406007711231345L,
        -9152888395723407474L, 8999993282035256217L, -6829424476226871438L, 2026619565689294464L,
        -3925094576856201394L, -6690097579743157728L, -294682202642863838L, 5472436080603216552L,
        -7101705404292871755L, 8031958568804398249L, -4265445736938701790L, -3795109844276665901L,
        -720121152745989333L, 9091170749936331336L, -7367604748107325189L, 3376138709496513133L,
        -4597819916706768583L, -391512631556746488L, -1135588877456072824L, 8733981247408842698L,
        -7627272076051127371L, 5458738279630526686L, -4922404076636521310L, -7011635205744005354L,
        -1541319077368263733L, 5070514048102157020L, -7880853450996246689L, 863228270850154185L,
        -5239380795317920458L, -3532650679864695173L, -1937539975720012668L, -9027499368258256870L,
        -8128491512466089774L, -3336344095947716592L, -5548928372155224313L, -8782116138362033643L,
        -2324474446766642487L, 7469098900757009562L, -8370325556870233411L, -2249342214667950880L,
        -5851220927660403859L, 6411694268519837208L, -2702340141148116920L, -5820440219632367202L,
        -8606491615858654931L, 7891439908798240259L, -6146428501395930760L, -3970758169284363389L,
        -3071349608317525546L, -351761693178066332L, -8837122532839535322L, 6697677969404790399L,
        -6434717147622031249L, -851274575098787810L, -3431710416100151157L, -1064093218873484762L,
        -9062348037703676329L, 8558313775058847832L, -6716249028702207507L, 6086206200396171886L,
        -3783625267450371480L, -6227300304786948855L, -117845565885576446L, -3172439362556298164L,
        -6991182506319567135L, -4288617610811380305L, -4127292114472071014L, 3862600023340550427L,
        -547429124662700864L, -4395122007679087774L, -7259672230555269896L, 8782263791269039901L,
        -4462904269766699466L, -7468914334623251740L, -966944318780986428L, 4498915137003099037L,
        -7521869226879198374L, -6411550076227838910L, -4790650515171610063L, 5820620459997365075L,
        -1376627125537124675L, -6559282480285457368L, -7777920981101784778L, -8711237568605798759L,
        -5110715207949843068L, 2946011094524915263L, -1776707991509915931L, 3682513868156144079L,
        -8027971522334779313L, 4607414176811284001L, -5423278384491086237L, 1147581702586717097L,
        -2167411962186469893L, -3177208890193991532L, -8272161504007625539L, 7237616480483531100L,
        -5728515861582144020L, -4788037454677749837L, -2548958808550292121L, -1373360799919799392L,
        -8510628282985014432L, -858350499949874620L, -6026599335303880135L, 3538747893490044629L,
        -2921563150702462265L, 9035120885289943691L, -8743505996830120772L, -5882264492762254953L,
        -6317696477610263061L, -2741144597525430788L, -3285434578585440922L, -3426430746906788485L,
        -8970925639256982432L, 4776009810824339053L, -6601971030643840136L, 5970012263530423816L,
        -3640777769877412266L, 7462515329413029771L, -9193015133814464522L, 52386062455755702L,
        -6879582898840692749L, -9157889458785081180L, -3987792605123478032L, 6999382250228200141L,
        -373054737976959636L, 8749227812785250177L, -7150688238876681629L, -3755104653863994448L,
        -4326674280168464132L, -4693880817329993060L, -796656831783192261L, -1255665003235103420L,
        -7415439547505577019L, 8438581409832836170L, -4657613415954583370L, -3286831292991118499L,
        -1210330751515841308L, -8720225134666286028L, -7673985747338482674L, -3144297699952734816L,
        -4980796165745715438L, -8542058143368306423L, -1614309188754756393L, 3157485376071780683L,
        -7926472270612804602L, 8890957387685944783L, -5296404319838617848L, 1890324697752655170L,
        -2008819381370884406L, 2362905872190818963L, -8173041140997884610L, 6088502188546649756L,
        -5604615407819967859L, -1612744301171463613L, -2394083241347571919L, 7207441660390446292L,
        -8413831053483314306L, -2412877989897052924L, -5905602798426754978L, -7627783505798704059L,
        -2770317479606055818L, 4300328673033783639L, -8648977452394866743L, -1923980597781273130L,
        -6199535797066195524L, 6818396289628184396L, -3137733727905356501L, 8522995362035230495L,
        -8878612607581929669L, 3021029092058325107L, -6486579741050024183L, -835399653354481520L,
        -3496538657885142324L, 8179122470161673908L, -9102865688819295809L, -4111420493003729616L,
        -6766896092596731857L, -5139275616254662020L, -3846934097318526917L, -6424094520318327524L,
        -196981603220770742L, -8030118150397909405L, -7040642529654063570L, -7324666853212387330L,
        -4189117143640191558L, 4679224488766679549L, -624710411122851544L, -3374341425896426372L,
        -7307973034592864071L, -9026492418826348338L, -4523280274813692185L, -2059743486678159615L,
        -1042414325089727327L, -2574679358347699519L, -7569037980822161435L, 3002511419460075705L,
        -4849611457600313890L, 8364825292752482535L, -1450328303573004458L, 1232659579085827361L,
        -7823984217374209643L, -3841273781498745804L, -5168294253290374149L, 4421779809981343554L,
        -1848681798185579782L, 915538744049291538L, -8072955151507069220L, 5183897733458195115L,
        -5479507920956448621L, 6479872166822743894L, -2237698882768172872L, 3488154190101041964L,
        -8316090829371189901L, 2180096368813151227L, -5783427518286599473L, -1886565557410948870L,
        -2617598379430861437L, -2358206946763686087L, -8553528014785370254L, 7749492695127472003L,
        -6080224000054324913L, 463493832054564196L, -2988593981640518238L, -4032318728359182659L,
        -8785400266166405755L, -4826042214438183114L, -6370064314280619289L, 3190819268807046916L,
        -3350894374423386208L, -623161932418579259L, -9011838011655698236L, -7307005235402693893L,
        -6653111496142234891L, -4522070525825979462L, -3704703351750405709L, 3570783879572301480L,
        -19193171260619233L, -148206168962011054L, -6929524759678968877L, -92628855601256909L,
        -4050219931171323192L, -115786069501571136L, -451088895536766085L, 4466953431550423984L,
        -7199459587351560659L, 486002885505321038L, -4387638465762062920L, 5219189625309039202L,
        -872862063775190746L, 6523987031636299002L, -7463067817500576073L, -534194123654701028L,
        -4717148753448332187L, -667742654568376285L, -1284749923383027329L, 8388693718644305452L,
        -7720497729755473937L, -6286281471915778852L, -5038936143766954517L, -7857851839894723565L,
        -1686984161281305242L, 8624429273841147159L, -7971894128441897632L, 778582277723329070L,
        -5353181642124984136L, 973227847154161338L, -2079791034228842266L, 1216534808942701673L,
        -8217398424034108273L, -3851351762838199359L, -5660062011615247437L, -4814189703547749198L,
        -2463391496091671392L, -6017737129434686498L, -8457148712698376476L, 7768129340171790699L,
        -5959749872445582691L, -8736582398494813242L, -2838001322129590460L, -1697355961263740745L,
        -8691279853972075893L, 1244995533423855986L, -6252413799037706963L, -3055441601647567921L,
        -3203831230369745799L, 5404070034795315907L, -8919923546622172981L, -3539985255894009414L,
        -6538218414850328322L, -4424981569867511768L, -3561087000135522498L, 8303831092947774002L,
        -9143208402725783417L, 578208414664970847L, -6817324484979841368L, -3888925500096174345L,
        -3909969587797413806L, -249470856692830027L, -275775966319379353L, -4923524589293425438L,
        -7089889006590693952L, -3077202868308390899L, -4250675239810979535L, 765182433041899281L,
        -701658031336336515L, 5568164059729762005L, -7356065297226292178L, 5785945546544795205L,
        -4583395603105477319L, -1990940103673781802L, -1117558485454458744L, 6734696907262548556L,
        -7616003081050118571L, 4209185567039092847L, -4908317832885260310L, -8573576096483297653L,
        -1523711272679187483L, 3118087934678041646L, -7869848573065574033L, 4254647968387469981L,
        -5225624697904579637L, 706623942056949572L, -1920344853953336643L, -3728406090856200939L,
        -8117744561361917258L, -6941939825212513491L, -5535494683275008668L, 5157633273766521849L,
        -2307682335666372931L, 6447041592208152311L, -8359830487432564938L, 6335244004343789146L,
        -5838102090863318269L, -1304317031425039375L, -2685941595151759932L, -1630396289281299219L,
        -8596242524610931813L, 1286845328412881940L, -6133617137336276863L, -3003129357911285479L,
        -3055335403242958174L, 5469460339465668959L, -8827113654667930715L, 8030098730593431003L,
        -6422206049907525490L, -3797434642040374958L, -3416071543957018958L, 9088264752731695015L,
        -9052573742614218705L, -8154892584824854328L, -6704031159840385477L, 8253128342678483706L,
        -3768352931373093942L, 5704724409920716729L, -98755145788979524L, -2092466524453879896L,
        -6979250993759194058L, 998051431430019017L, -4112377723771604669L, -7975807747567252037L,
        -528786136287117932L, 8476984389250486570L, -7248020362820530564L, -3925256793573221702L,
        -4448339435098275301L, -294884973539139224L, -948738275445456222L, -368606216923924029L,
        -7510490449794491995L, -2536221894791146470L, -4776427043815727089L, 6053094668365842720L,
        -1358847786342270957L, 2954682317029915496L, -7766808894105001205L, -459166561069996767L,
        -5096825099203863602L, -573958201337495959L, -1759345355577441598L, -5329133770099257852L,
        -8017119874876982855L, -5636551615525730110L, -5409713825168840664L, 2177682517447613171L,
        -2150456263033662926L, 2722103146809516464L, -8261564192037121185L, 6313000485183335694L,
        -5715269221619013577L, 3279564588051781713L, -2532400508596379068L, -512230283362660763L,
        -8500279345513818773L, 1985699082112030975L, -6013663163464885563L, -2129562165787349185L,
        -2905392935903719049L, 6561419329620589327L, -8733399612580906262L, -7428327965055601431L,
        -6305063497298744923L, 4549648098962661924L, -3269643353196043250L, -8147997931578836307L,
        -8961056123388608887L, 1825030320404309164L, -6589634135808373205L, 6892973918932774359L,
        -3625356651333078602L, 4004531380238580045L, -9183376934724255983L, -2108853905778275376L,
        -6867535149977932074L, 6587304654631931588L, -3972732919045027189L, -989241218564861323L,
        -354230130378896082L, -1236551523206076654L, -7138922859127891907L, 6144684325637283947L,
        -4311967555482476980L, -6154202648235558778L, -778273425925708321L, -3081067291867060568L,
        -7403949918844649557L, -1925667057416912855L, -4643251380128424042L, -2407083821771141069L,
        -1192378206733142148L, -7620540795641314240L, -7662765406849295699L, -2456994988062127448L,
        -4966770740134231719L, 6152128301777116498L, -1596777406740401745L, -6144897678060768090L,
        -7915514906853832947L, -3840561048787980056L, -5282707615139903279L, 4422670725869800738L,
        -1991698500497491195L, -8306719647944912790L, -8162340590452013853L, 8643358275316593218L,
        -5591239719637629412L, 6192511825718353619L, -2377363631119648861L, 7740639782147942024L,
        -8403381297090862394L, 2532056854628769813L, -5892540602936190089L, -6058300968568813542L,
        -2753989735242849707L, -7572876210711016927L, -8638772612167862923L, 9102010423587778132L,
        -6186779746782440750L, -2457545025797441047L, -3121788665050663033L, -7683617300674189212L,
        -8868646943297746252L, -4802260812921368258L, -6474122660694794911L, -1391139997724322418L,
        -3480967307441105734L, 7484447039699372786L, -9093133594791772940L, -9157278655470055721L,
        -6754730975062328271L, -6834912300910181747L, -3831727700400522434L, 679731660717048624L,
        -177973607073265139L, -8373707460958465028L, -7028762532061872568L, 8601490892183123070L,
        -4174267146649952806L, -7694880458480647778L, -606147914885053103L, 4216457482181353989L,
        -7296371474444240046L, -4282243101277735613L, -4508778324627912153L, 8482254178684994196L,
        -1024286887357502287L, 5991131704928854841L, -7557708332239520786L, -3173071712060547580L,
        -4835449396872013078L, -8578025658503072379L, -1432625727662628443L, 3112525982153323238L,
        -7812920107430224633L, 4251171748059520976L, -5154464115860392887L, 702278666647013315L,
        -1831394126398103205L, 5489534351736154548L, -8062150356639896359L, 1125115960621402641L,
        -5466001927372482545L, 6018080969204141205L, -2220816390788215277L, 2910915193077788602L,
        -8305539271883716405L, -486521013540076076L, -5770238071427257602L, -608151266925095095L,
        -2601111570856684098L, -5371875102083756772L, -8543223759426509417L, 3560107088838733873L,
        -6067343680855748868L, -161552157378970562L, -2972493582642298180L, 4409745821703674701L,
        -8775337516792518219L, -6467280898289979120L, -6357485877563259869L, 1139270913992301908L,
        -3335171328526686933L, -3187597375937010519L, -9002011107970261189L, 7231123676894144234L,
        -6640827866535438582L, 4427218577690292388L, -3689348814741910324L, -3689348814741910323L,
        -9223372036854775808L, 0L, -6917529027641081856L, 0L,
        -4035225266123964416L, 0L, -432345564227567616L, 0L,
        -7187745005283311616L, 0L, -4372995238176751616L, 0L,
        -854558029293551616L, 0L, -7451627795949551616L, 0L,
        -4702848726509551616L, 0L, -1266874889709551616L, 0L,
        -7709325833709551616L, 0L, -5024971273709551616L, 0L,
        -1669528073709551616L, 0L, -7960984073709551616L, 0L,
        -5339544073709551616L, 0L, -2062744073709551616L, 0L,
        -8206744073709551616L, 0L, -5646744073709551616L, 0L,
        -2446744073709551616L, 0L, -8446744073709551616L, 0L,
        -5946744073709551616L, 0L, -2821744073709551616L, 0L,
        -8681119073709551616L, 0L, -6239712823709551616L, 0L,
        -3187955011209551616L, 0L, -8910000909647051616L, 0L,
        -6525815118631426616L, 0L, -3545582879861895366L, 0L,
        -9133518327554766460L, 4611686018427387904L, -6805211891016070171L, 5764607523034234880L,
        -3894828845342699810L, -6629298651489370112L, -256850038250986858L, 5548434740920451072L,
        -7078060301547948643L, -1143914305352105984L, -4235889358507547899L, 7793479155164643328L,
        -683175679707046970L, -4093209111326359552L, -7344513827457986212L, 4359273333062107136L,
        -4568956265895094861L, 5449091666327633920L, -1099509313941480672L, 2199678564482154496L,
        -7604722348854507276L, 1374799102801346560L, -4894216917640746191L, 1718498878501683200L,
        -1506085128623544835L, 6759809616554491904L, -7858832233030797378L, 6530724019560251392L,
        -5211854272861108819L, -1059967012404461568L, -1903131822648998119L, 7898413271349198848L,
        -8106986416796705681L, -1981020733047832576L, -5522047002568494197L, -2476275916309790720L,
        -2290872734783229842L, -3095344895387238400L, -8349324486880600507L, 4982938468024057856L,
        -5824969590173362730L, -7606384970252091392L, -2669525969289315508L, 4327076842467049472L,
        -8585982758446904049L, -6518949010312869888L, -6120792429631242157L, -8148686262891087360L,
        -3039304518611664792L, 8260886245095692416L, -8817094351773372351L, 5163053903184807760L,
        -6409681921289327535L, -7381240676301154012L, -3400416383184271515L, -3178808521666707L,
        -9042789267131251553L, -4613672773753429596L, -6691800565486676537L, -5767090967191786995L,
        -3753064688430957767L, -7208863708989733744L, -79644842111309304L, 212292400617608628L,
        -6967307053960650171L, 132682750386005392L, -4097447799023424810L, 4777539456409894645L,
        -510123730351893109L, -3251447716342407502L, -7236356359111015049L, 7191217214140771119L,
        -4433759430461380907L, 4377335499248575995L, -930513269649338230L, -8363388681221443718L,
        -7499099821171918250L, -7532960934977096276L, -4762188758037509908L, 4418856886560793367L,
        -1341049929119499481L, 5523571108200991709L, -7755685233340769032L, -8076983103442849942L,
        -5082920523248573386L, -5484542860876174524L, -1741964635633328828L, 6979379479186945558L,
        -8006256924911912374L, -4861259862362934835L, -5396135137712502563L, 7758483227328495169L,
        -2133482903713240300L, -4136954021121544751L, -8250955842461857044L, -279753253987271518L,
        -5702008784649933400L, 4261994450943298507L, -2515824962385028846L, 5327493063679123134L,
        -8489919629131724885L, 7941369183226839863L, -6000713517987268202L, 5315025460606161924L,
        -2889205879056697349L, -2579590211097073402L, -8723282702051517699L, 7611128154919104931L,
        -6292417359137009220L, -4321147861633282548L, -3253835680493873621L, -789748808614215280L,
        -8951176327949752869L, 8729779031470891258L, -6577284391509803182L, 6300537770911226168L,
        -3609919470959866074L, -1347699823215743098L, -9173728696990998152L, 6075216638131242420L,
        -6855474852811359786L, 7594020797664053025L, -3957657547586811828L, 269153960225290473L,
        -335385916056126881L, 336442450281613091L, -7127145225176161157L, 7127805559067090038L,
        -4297245513042813542L, 4298070930406474644L, -759870872876129024L, -3850783373846682503L,
        -7392448323188662496L, 9122475437414293195L, -4628874385558440216L, -7043649776941685122L,
        -1174406963520662366L, -4192876202749718498L, -7651533379841495835L, -4926390635932268014L,
        -4952730706374481889L, 3065383741939440791L, -1579227364540714458L, -779956341003086915L,
        -7904546130479028392L, 6430056314514152534L, -5268996644671397586L, 8037570393142690668L,
        -1974559787411859078L, 823590954573587527L, -8151628894773493780L, 5126430365035880108L,
        -5577850100039479321L, 6408037956294850135L, -2360626606621961247L, 3398361426941174765L,
        -8392920656779807636L, -4793553135802847628L, -5879464802547371641L, -1380255401326171631L,
        -2737644984756826647L, -1725319251657714539L, -8628557143114098510L, 3533361486141316317L,
        -6174010410465235234L, -4806670179178130411L, -3105826994654156138L, 7826720331309500698L,
        -8858670899299929442L, 280014188641050032L, -6461652605697523899L, -8873354301053463268L,
        -3465379738694516970L, -1868320839462053277L, -9083391364325154962L, 5749828502977298558L,
        -6742553186979055799L, -2036086408133152611L, -3816505465296431844L, 6678264026688335045L,
        -158945813193151901L, 8347830033360418806L, -7016870160886801794L, 2911550761636567802L,
        -4159401682681114339L, -5583933584809066056L, -587566084924005019L, 2243455055843443238L,
        -7284757830718584993L, 3708002419115845976L, -4494261269970843337L, 23317005467419566L,
        -1006140569036166268L, -4582539761593113446L, -7546366883288685774L, -558244341782001952L,
        -4821272585683469313L, -5309491445654890344L, -1414904713676948737L, -6636864307068612930L,
        -7801844473689174817L, -4148040191917883081L, -5140619573684080617L, -5185050239897353852L,
        -1814088448677712867L, -6481312799871692315L, -8051334308064652398L, -8662506518347195601L,
        -5452481866653427593L, 3006924907348169211L, -2203916314889396588L, -853029884242176390L,
        -8294976724446954723L, 1772699331562333708L, -5757034887131305500L, 6827560182880305039L,
        -2584607590486743971L, 8534450228600381299L, -8532908771695296838L, 7639874402088932264L,
        -6054449946191733143L, 326470965756389522L, -2956376414312278525L, 5019774725622874806L,
        -8765264286586255934L, 831516194300602802L, -6344894339805432014L, -8183976793979022306L,
        -3319431906329402113L, 3605087062808385830L, -8992173969096958177L, 9170708441896323000L,
        -6628531442943809817L, 6851699533943015846L, -3673978285252374367L, 3952938399001381903L,
        -9213765455923815836L, -4446942528265218167L, -6905520801477381891L, -946992141904134804L,
        -4020214983419339459L, 8039631859474607303L, -413582710846786420L, -3785518230938904583L,
        -7176018221920323369L, -60105885123121413L, -4358336758973016307L, -75132356403901766L,
        -836234930288882479L, 9129456591349898601L, -7440175859071633406L, -1211618658047395231L,
        -4688533805412153853L, -6126209340986631942L, -1248981238337804412L, -7657761676233289928L,
        -7698142301602209614L, -2480258038432112253L, -5010991858575374113L, -7712008566467528220L,
        -1652053804791829737L, 8806733365625141341L, -7950062655635975442L, -6025006692552756422L,
        -5325892301117581398L, 6303799689591218185L, -2045679357969588844L, -1343622424865753077L,
        -8196078626372074883L, 1466078993672598279L, -5633412264537705700L, 6444284760518135752L,
        -2430079312244744221L, 8055355950647669691L, -8436328597794046994L, 2728754459941099604L,
        -5933724728815170839L, -5812428961928401302L, -2805469892591575644L, 1957835834444274180L,
        -8670947710510816634L, -7999724640327104446L, -6226998619711132888L, 3835402254873283155L,
        -3172062256211528206L, 4794252818591603944L, -8900067937773286985L, 7608094030047140369L,
        -6513398903789220827L, 4898431519131537557L, -3530062611309138130L, -7712018656367741765L,
        -9123818159709293187L, 2097517367411243253L, -6793086681209228580L, 7233582727691441970L,
        -3879672333084147821L, 9041978409614302462L, -237904397927796872L, 6690786993590490174L,
        -7066219276345954901L, 4181741870994056359L, -4221088077005055722L, 615491320315182544L,
        -664674077828931749L, -8454007886460797627L, -7332950326284164199L, 3939617107816777291L,
        -4554501889427817345L, -8910536670511192099L, -1081441343357383777L, 7308573235570561493L,
        -7593429867239446717L, -6961356773836868827L, -4880101315621920492L, -8701695967296086034L,
        -1488440626100012711L, -6265433940692719638L, -7847804418953589800L, 695789805494438130L,
        -5198069505264599346L, 869737256868047663L, -1885900863153361279L, -8136200465769716230L,
        -8096217067111932656L, -473439272678684740L, -5508585315462527915L, 4019886927579031980L,
        -2274045625900771990L, -8810199395808373737L, -8338807543829064350L, -7812217631593927538L,
        -5811823411358942533L, 4069786015789754290L, -2653093245771290262L, 475546501309804958L,
        -8575712306248138270L, 4908902581746016003L, -6107954364382784934L, -3087243809672255805L,
        -3023256937051093263L, -8470740780517707660L, -8807064613298015146L, -682526969396179383L,
        -6397144748195131028L, -5464844730172612133L, -3384744916816525881L, -2219369894288377262L,
        -9032994600651410532L, -1387106183930235789L, -6679557232386875260L, 2877803288514593168L,
        -3737760522056206171L, 3597254110643241460L, -60514634142869810L, 9108253656731439729L,
        -6955350673980375487L, 1080972517029761926L, -4082502324048081455L, 5962901664714590312L,
        -491441886632713915L, -6381430974388925822L, -7224680206786528053L, -8600080377420466543L,
        -4419164240055772162L, 7696643601933968437L, -912269281642327298L, 397432465562684739L,
        -7487697328667536418L, -4363290727450709942L, -4747935642407032618L, 8380944645968776284L,
        -1323233534581402868L, 1252808770606194547L, -7744549986754458649L, -8440366555225904216L,
        -5069001465015685407L, 7896285879677171346L, -1724565812842218855L, -3964700705685699529L,
        -7995382660667468640L, 2133748077373825698L, -5382542307406947896L, 2667185096717282123L,
        -2116491865831296966L, 3333981370896602653L, -8240336443785642460L, 6695424375237764562L,
        -5688734536304665171L, 8369280469047205703L, -2499232151953443560L, -3373457468973156583L,
        -8479549122611984081L, -9025939945749304721L, -5987750384837592197L, 7164319141522920715L,
        -2873001962619602342L, 4343712908476262990L, -8713155254278333320L, 7326506586225052273L,
        -6279758049420528746L, 9158133232781315341L, -3238011543348273028L, 2224294504121868368L,
        -8941286242233752499L, -7833187971778608078L, -6564921784364802720L, -568112927868484289L,
        -3594466212028615495L, 3901544858591782542L, -9164070410158966541L, -4479063491021217767L,
        -6843401994271320272L, -5598829363776522209L, -3942566474411762436L, -2386850686293264857L,
        -316522074587315140L, 1628122660560806833L, -7115355324258153819L, -8205795374004271538L,
        -4282508136895304370L, -1033872180650563614L, -741449152691742558L, -5904026244240592421L,
        -7380934748073420955L, -5995859411864064215L, -4614482416664388289L, 1728547772024695539L,
        -1156417002403097458L, -2451001303396518480L, -7640289654143017767L, 5385653213018257806L,
        -4938676049251384305L, -7102991539009341455L, -1561659043136842477L, -8878739423761676819L,
        -7893565929601608404L, 3674159897003727796L, -5255271393574622601L, 4592699871254659745L,
        -1957403223540890347L, 1129188820640936778L, -8140906042354138323L, 3011586022114279438L,
        -5564446534515285000L, 8376168546070237202L, -2343872149716718346L, -7976533391121755114L,
        -8382449121214030822L, 1932195658189984910L, -5866375383090150624L, -6808127464117294671L,
        -2721283210435300376L, -3898473311719230434L, -8618331034163144591L, 9092669226243950738L,
        -6161227774276542835L, -2469221522477225289L, -3089848699418290639L, 6136845133758244197L,
        -8848684464777513506L, -3082000819042179233L, -6449169562544503978L, -8464187042230111945L,
        -3449775934753242068L, 3254824252494523781L, -9073638986861858149L, -7189106879045698445L,
        -6730362715149934782L, -8986383598807123057L, -3801267375510030573L, 2602078556773259891L,
        -139898200960150313L, -1359087822460813040L, -7004965403241175802L, -849429889038008150L,
        -4144520735624081848L, -5673473379724898091L, -568964901102714406L, -2480155706228734710L,
        -7273132090830278360L, -3855940325606653146L, -4479729095110460046L, -208239388580928528L,
        -987975350460687153L, -4871985254153548564L, -7535013621679011327L, -3044990783845967853L,
        -4807081008671376254L, 5417133557047315992L, -1397165242411832414L, -2451955090545630818L,
        -7790757304148477115L, -3838314940804713213L, -5126760611758208489L, 4425478360848884291L,
        -1796764746270372707L, 920161932633717460L, -8040506994060064798L, 2880944217109767365L,
        -5438947724147693094L, -5622191765467566602L, -2186998636757228463L, 6807318348447705459L,
        -8284403175614349646L, -2662955059861265944L, -5743817951090549153L, -7940379843253970334L,
        -2568086420435798537L, 8521269269642088699L, -8522583040413455942L, -6203421752542164323L,
        -6041542782089432023L, 6080780864604458308L, -2940242459184402125L, -6234081974526590827L,
        -8755180564631333184L, 5327070802775656541L, -6332289687361778576L, 6658838503469570676L,
        -3303676090774835316L, 8323548129336963345L, -8982326584375353929L, -4021154456019173717L,
        -6616222212041804507L, -5026443070023967147L, -3658591746624867729L, 2940318199324816875L,
        -9204148869281624187L, 8755227902219092403L, -6893500068174642330L, -2891023177508298209L,
        -4005189066790915008L, -8225464990312760665L, -394800315061255856L, -5670145219463562927L,
        -7164279224554366766L, 7985374283903742931L, -4343663012265570553L, 758345818024902856L,
        -817892746904575288L, -3663753745896259334L, -7428711994456441411L, -9207375118826243940L,
        -4674203974643163860L, -2285846861678029117L, -1231068949876566920L, 1754377441329851508L,
        -7686947121313936181L, 1096485900831157192L, -4996997883215032323L, -3241078642388441414L,
        -1634561335591402499L, 5172023733869224041L, -7939129862385708418L, 5538357842881958977L,
        -5312226309554747619L, -2300424733252327086L, -2028596868516046619L, 6347841120289366950L,
        -8185402070463610993L, 6273243709394548296L
    };
}
