/*
 * Copyright (c) 2026, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */
package jdk.jpackage.internal;

import java.io.PrintWriter;
import java.util.Optional;
import java.util.function.Supplier;

public final class Globals {

    private Globals() {
    }

    public Globals objectFactory(ObjectFactory v) {
        checkMutable();
        objectFactory = Optional.ofNullable(v).orElse(ObjectFactory.DEFAULT);
        return this;
    }

    public ObjectFactory objectFactory() {
        return objectFactory;
    }

    public Globals executorFactory(ExecutorFactory v) {
        return objectFactory(ObjectFactory.build(objectFactory).executorFactory(v).create());
    }

    Log.Logger logger() {
        return logger;
    }

    public void loggerOutputStreams(PrintWriter out, PrintWriter err) {
        logger.setPrintWriter(out, err);
    }

    public void loggerVerbose() {
        logger.setVerbose();
    }

    public static int main(Supplier<Integer> mainBody) {
        if (INSTANCE.isBound()) {
            return mainBody.get();
        } else {
            return ScopedValue.where(INSTANCE, new Globals()).call(mainBody::get);
        }
    }

    public static Globals instance() {
        return INSTANCE.orElse(DEFAULT);
    }

    private void checkMutable() {
        if (this == DEFAULT) {
            throw new UnsupportedOperationException("Can't modify immutable instance");
        }
    }

    private ObjectFactory objectFactory = ObjectFactory.DEFAULT;
    private final Log.Logger logger = new Log.Logger();

    private static final ScopedValue<Globals> INSTANCE = ScopedValue.newInstance();
    private static final Globals DEFAULT = new Globals();
}
