/*
 * Copyright (c) 2017, 2026, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */
package jdk.incubator.vector;

import java.lang.foreign.MemorySegment;
import java.lang.foreign.ValueLayout;
import java.nio.ByteOrder;
import java.util.Arrays;
import java.util.Objects;
import java.util.function.IntUnaryOperator;

import jdk.internal.vm.annotation.ForceInline;
import jdk.internal.vm.vector.VectorSupport;

import static jdk.internal.vm.vector.VectorSupport.*;

import static jdk.incubator.vector.VectorOperators.*;

// -- This file was mechanically generated: Do not edit! -- //

@SuppressWarnings("cast")  // warning: redundant cast
final class ShortVector256 extends ShortVector {
    static final ShortSpecies VSPECIES =
        (ShortSpecies) ShortVector.SPECIES_256;

    static final VectorShape VSHAPE =
        VSPECIES.vectorShape();

    static final Class<ShortVector256> VCLASS = ShortVector256.class;

    static final int VSIZE = VSPECIES.vectorBitSize();

    static final int VLENGTH = VSPECIES.laneCount(); // used by the JVM

    static final Class<Short> CTYPE = short.class; // carrier type used by the JVM

    static final Class<Short> ETYPE = short.class; // used by the JVM

    ShortVector256(short[] v) {
        super(v);
    }

    // For compatibility as ShortVector256::new,
    // stored into species.vectorFactory.
    ShortVector256(Object v) {
        this((short[]) v);
    }

    static final ShortVector256 ZERO = new ShortVector256(new short[VLENGTH]);
    static final ShortVector256 IOTA = new ShortVector256(VSPECIES.iotaArray());

    static {
        // Warm up a few species caches.
        // If we do this too much we will
        // get NPEs from bootstrap circularity.
        VSPECIES.dummyVector();
        VSPECIES.withLanes(LaneType.BYTE);
    }

    // Specialized extractors

    @ForceInline
    final @Override
    public ShortSpecies vspecies() {
        // ISSUE:  This should probably be a @Stable
        // field inside AbstractVector, rather than
        // a megamorphic method.
        return VSPECIES;
    }

    @ForceInline
    @Override
    public final Class<Short> elementType() { return short.class; }

    @ForceInline
    final Class<Short> carrierType() { return CTYPE; }

    @ForceInline
    @Override
    public final int elementSize() { return Short.SIZE; }

    @ForceInline
    @Override
    public final VectorShape shape() { return VSHAPE; }

    @ForceInline
    @Override
    public final int length() { return VLENGTH; }

    @ForceInline
    @Override
    public final int bitSize() { return VSIZE; }

    @ForceInline
    @Override
    public final int byteSize() { return VSIZE / Byte.SIZE; }

    /*package-private*/
    @ForceInline
    final @Override
    short[] vec() {
        return (short[])getPayload();
    }

    /*package-private*/
    @ForceInline
    final @Override
    int laneTypeOrdinal() {
        return LANE_TYPE_ORDINAL;
    }

    // Virtualized constructors

    @Override
    @ForceInline
    public final ShortVector256 broadcast(short e) {
        return (ShortVector256) super.broadcastTemplate(e);  // specialize
    }

    @Override
    @ForceInline
    public final ShortVector256 broadcast(long e) {
        return (ShortVector256) super.broadcastTemplate(e);  // specialize
    }

    @Override
    @ForceInline
    ShortMask256 maskFromArray(boolean[] bits) {
        return new ShortMask256(bits);
    }

    @Override
    @ForceInline
    ShortShuffle256 iotaShuffle() { return ShortShuffle256.IOTA; }

    @Override
    @ForceInline
    ShortShuffle256 iotaShuffle(int start, int step, boolean wrap) {
        return (ShortShuffle256) iotaShuffleTemplate((short) start, (short) step, wrap);
    }

    @Override
    @ForceInline
    ShortShuffle256 shuffleFromArray(int[] indices, int i) { return new ShortShuffle256(indices, i); }

    @Override
    @ForceInline
    ShortShuffle256 shuffleFromOp(IntUnaryOperator fn) { return new ShortShuffle256(fn); }

    // Make a vector of the same species but the given elements:
    @ForceInline
    final @Override
    ShortVector256 vectorFactory(short[] vec) {
        return new ShortVector256(vec);
    }

    @ForceInline
    final @Override
    ByteVector256 asByteVectorRaw() {
        return (ByteVector256) super.asByteVectorRawTemplate();  // specialize
    }

    @ForceInline
    final @Override
    AbstractVector<?> asVectorRaw(LaneType laneType) {
        return super.asVectorRawTemplate(laneType);  // specialize
    }

    // Unary operator

    @ForceInline
    final @Override
    ShortVector256 uOp(FUnOp f) {
        return (ShortVector256) super.uOpTemplate(f);  // specialize
    }

    @ForceInline
    final @Override
    ShortVector256 uOp(VectorMask<Short> m, FUnOp f) {
        return (ShortVector256)
            super.uOpTemplate((ShortMask256)m, f);  // specialize
    }

    // Binary operator

    @ForceInline
    final @Override
    ShortVector256 bOp(Vector<Short> v, FBinOp f) {
        return (ShortVector256) super.bOpTemplate((ShortVector256)v, f);  // specialize
    }

    @ForceInline
    final @Override
    ShortVector256 bOp(Vector<Short> v,
                     VectorMask<Short> m, FBinOp f) {
        return (ShortVector256)
            super.bOpTemplate((ShortVector256)v, (ShortMask256)m,
                              f);  // specialize
    }

    // Ternary operator

    @ForceInline
    final @Override
    ShortVector256 tOp(Vector<Short> v1, Vector<Short> v2, FTriOp f) {
        return (ShortVector256)
            super.tOpTemplate((ShortVector256)v1, (ShortVector256)v2,
                              f);  // specialize
    }

    @ForceInline
    final @Override
    ShortVector256 tOp(Vector<Short> v1, Vector<Short> v2,
                     VectorMask<Short> m, FTriOp f) {
        return (ShortVector256)
            super.tOpTemplate((ShortVector256)v1, (ShortVector256)v2,
                              (ShortMask256)m, f);  // specialize
    }

    @ForceInline
    final @Override
    short rOp(short v, VectorMask<Short> m, FBinOp f) {
        return super.rOpTemplate(v, m, f);  // specialize
    }

    @Override
    @ForceInline
    public final <F>
    Vector<F> convertShape(VectorOperators.Conversion<Short,F> conv,
                           VectorSpecies<F> rsp, int part) {
        return super.convertShapeTemplate(conv, rsp, part);  // specialize
    }

    @Override
    @ForceInline
    public final <F>
    Vector<F> reinterpretShape(VectorSpecies<F> toSpecies, int part) {
        return super.reinterpretShapeTemplate(toSpecies, part);  // specialize
    }

    // Specialized algebraic operations:

    // The following definition forces a specialized version of this
    // crucial method into the v-table of this class.  A call to add()
    // will inline to a call to lanewise(ADD,), at which point the JIT
    // intrinsic will have the opcode of ADD, plus all the metadata
    // for this particular class, enabling it to generate precise
    // code.
    //
    // There is probably no benefit to the JIT to specialize the
    // masked or broadcast versions of the lanewise method.

    @Override
    @ForceInline
    public ShortVector256 lanewise(Unary op) {
        return (ShortVector256) super.lanewiseTemplate(op);  // specialize
    }

    @Override
    @ForceInline
    public ShortVector256 lanewise(Unary op, VectorMask<Short> m) {
        return (ShortVector256) super.lanewiseTemplate(op, ShortMask256.class, (ShortMask256) m);  // specialize
    }

    @Override
    @ForceInline
    public ShortVector256 lanewise(Binary op, Vector<Short> v) {
        return (ShortVector256) super.lanewiseTemplate(op, v);  // specialize
    }

    @Override
    @ForceInline
    public ShortVector256 lanewise(Binary op, Vector<Short> v, VectorMask<Short> m) {
        return (ShortVector256) super.lanewiseTemplate(op, ShortMask256.class, v, (ShortMask256) m);  // specialize
    }

    /*package-private*/
    @Override
    @ForceInline ShortVector256
    lanewiseShift(VectorOperators.Binary op, int e) {
        return (ShortVector256) super.lanewiseShiftTemplate(op, e);  // specialize
    }

    /*package-private*/
    @Override
    @ForceInline ShortVector256
    lanewiseShift(VectorOperators.Binary op, int e, VectorMask<Short> m) {
        return (ShortVector256) super.lanewiseShiftTemplate(op, ShortMask256.class, e, (ShortMask256) m);  // specialize
    }

    /*package-private*/
    @Override
    @ForceInline
    public final
    ShortVector256
    lanewise(Ternary op, Vector<Short> v1, Vector<Short> v2) {
        return (ShortVector256) super.lanewiseTemplate(op, v1, v2);  // specialize
    }

    @Override
    @ForceInline
    public final
    ShortVector256
    lanewise(Ternary op, Vector<Short> v1, Vector<Short> v2, VectorMask<Short> m) {
        return (ShortVector256) super.lanewiseTemplate(op, ShortMask256.class, v1, v2, (ShortMask256) m);  // specialize
    }

    @Override
    @ForceInline
    public final
    ShortVector256 addIndex(int scale) {
        return (ShortVector256) super.addIndexTemplate(scale);  // specialize
    }

    // Type specific horizontal reductions

    @Override
    @ForceInline
    public final short reduceLanes(VectorOperators.Associative op) {
        return super.reduceLanesTemplate(op);  // specialized
    }

    @Override
    @ForceInline
    public final short reduceLanes(VectorOperators.Associative op,
                                    VectorMask<Short> m) {
        return super.reduceLanesTemplate(op, ShortMask256.class, (ShortMask256) m);  // specialized
    }

    @Override
    @ForceInline
    public final long reduceLanesToLong(VectorOperators.Associative op) {
        return (long) super.reduceLanesTemplate(op);  // specialized
    }

    @Override
    @ForceInline
    public final long reduceLanesToLong(VectorOperators.Associative op,
                                        VectorMask<Short> m) {
        return (long) super.reduceLanesTemplate(op, ShortMask256.class, (ShortMask256) m);  // specialized
    }

    @Override
    @ForceInline
    final <F> VectorShuffle<F> bitsToShuffle(AbstractSpecies<F> dsp) {
        return bitsToShuffleTemplate(dsp);
    }

    @Override
    @ForceInline
    public final ShortShuffle256 toShuffle() {
        return (ShortShuffle256) toShuffle(vspecies(), false);
    }

    // Specialized unary testing

    @Override
    @ForceInline
    public final ShortMask256 test(Test op) {
        return super.testTemplate(ShortMask256.class, op);  // specialize
    }

    @Override
    @ForceInline
    public final ShortMask256 test(Test op, VectorMask<Short> m) {
        return super.testTemplate(ShortMask256.class, op, (ShortMask256) m);  // specialize
    }

    // Specialized comparisons

    @Override
    @ForceInline
    public final ShortMask256 compare(Comparison op, Vector<Short> v) {
        return super.compareTemplate(ShortMask256.class, op, v);  // specialize
    }

    @Override
    @ForceInline
    public final ShortMask256 compare(Comparison op, short s) {
        return super.compareTemplate(ShortMask256.class, op, s);  // specialize
    }

    @Override
    @ForceInline
    public final ShortMask256 compare(Comparison op, long s) {
        return super.compareTemplate(ShortMask256.class, op, s);  // specialize
    }

    @Override
    @ForceInline
    public final ShortMask256 compare(Comparison op, Vector<Short> v, VectorMask<Short> m) {
        return super.compareTemplate(ShortMask256.class, op, v, (ShortMask256) m);
    }


    @Override
    @ForceInline
    public ShortVector256 blend(Vector<Short> v, VectorMask<Short> m) {
        return (ShortVector256)
            super.blendTemplate(ShortMask256.class,
                                (ShortVector256) v,
                                (ShortMask256) m);  // specialize
    }

    @Override
    @ForceInline
    public ShortVector256 slice(int origin, Vector<Short> v) {
        return (ShortVector256) super.sliceTemplate(origin, v);  // specialize
    }

    @Override
    @ForceInline
    public ShortVector256 slice(int origin) {
        return (ShortVector256) super.sliceTemplate(origin);  // specialize
    }

    @Override
    @ForceInline
    public ShortVector256 unslice(int origin, Vector<Short> w, int part) {
        return (ShortVector256) super.unsliceTemplate(origin, w, part);  // specialize
    }

    @Override
    @ForceInline
    public ShortVector256 unslice(int origin, Vector<Short> w, int part, VectorMask<Short> m) {
        return (ShortVector256)
            super.unsliceTemplate(ShortMask256.class,
                                  origin, w, part,
                                  (ShortMask256) m);  // specialize
    }

    @Override
    @ForceInline
    public ShortVector256 unslice(int origin) {
        return (ShortVector256) super.unsliceTemplate(origin);  // specialize
    }

    @Override
    @ForceInline
    public ShortVector256 rearrange(VectorShuffle<Short> s) {
        return (ShortVector256)
            super.rearrangeTemplate(ShortShuffle256.class,
                                    (ShortShuffle256) s);  // specialize
    }

    @Override
    @ForceInline
    public ShortVector256 rearrange(VectorShuffle<Short> shuffle,
                                  VectorMask<Short> m) {
        return (ShortVector256)
            super.rearrangeTemplate(ShortShuffle256.class,
                                    ShortMask256.class,
                                    (ShortShuffle256) shuffle,
                                    (ShortMask256) m);  // specialize
    }

    @Override
    @ForceInline
    public ShortVector256 rearrange(VectorShuffle<Short> s,
                                  Vector<Short> v) {
        return (ShortVector256)
            super.rearrangeTemplate(ShortShuffle256.class,
                                    (ShortShuffle256) s,
                                    (ShortVector256) v);  // specialize
    }

    @Override
    @ForceInline
    public ShortVector256 compress(VectorMask<Short> m) {
        return (ShortVector256)
            super.compressTemplate(ShortMask256.class,
                                   (ShortMask256) m);  // specialize
    }

    @Override
    @ForceInline
    public ShortVector256 expand(VectorMask<Short> m) {
        return (ShortVector256)
            super.expandTemplate(ShortMask256.class,
                                   (ShortMask256) m);  // specialize
    }

    @Override
    @ForceInline
    public ShortVector256 selectFrom(Vector<Short> v) {
        return (ShortVector256)
            super.selectFromTemplate((ShortVector256) v);  // specialize
    }

    @Override
    @ForceInline
    public ShortVector256 selectFrom(Vector<Short> v,
                                   VectorMask<Short> m) {
        return (ShortVector256)
            super.selectFromTemplate((ShortVector256) v,
                                     ShortMask256.class, (ShortMask256) m);  // specialize
    }

    @Override
    @ForceInline
    public ShortVector256 selectFrom(Vector<Short> v1,
                                   Vector<Short> v2) {
        return (ShortVector256)
            super.selectFromTemplate((ShortVector256) v1, (ShortVector256) v2);  // specialize
    }

    @ForceInline
    @Override
    public short lane(int i) {
        switch(i) {
            case 0: return laneHelper(0);
            case 1: return laneHelper(1);
            case 2: return laneHelper(2);
            case 3: return laneHelper(3);
            case 4: return laneHelper(4);
            case 5: return laneHelper(5);
            case 6: return laneHelper(6);
            case 7: return laneHelper(7);
            case 8: return laneHelper(8);
            case 9: return laneHelper(9);
            case 10: return laneHelper(10);
            case 11: return laneHelper(11);
            case 12: return laneHelper(12);
            case 13: return laneHelper(13);
            case 14: return laneHelper(14);
            case 15: return laneHelper(15);
            default: throw new IllegalArgumentException("Index " + i + " must be zero or positive, and less than " + VLENGTH);
        }
    }

    @ForceInline
    public short laneHelper(int i) {
        return (short) VectorSupport.extract(
                                VCLASS, LANE_TYPE_ORDINAL, VLENGTH,
                                this, i,
                                (vec, ix) -> {
                                    short[] vecarr = vec.vec();
                                    return (long)vecarr[ix];
                                });
    }

    @ForceInline
    @Override
    public ShortVector256 withLane(int i, short e) {
        switch (i) {
            case 0: return withLaneHelper(0, e);
            case 1: return withLaneHelper(1, e);
            case 2: return withLaneHelper(2, e);
            case 3: return withLaneHelper(3, e);
            case 4: return withLaneHelper(4, e);
            case 5: return withLaneHelper(5, e);
            case 6: return withLaneHelper(6, e);
            case 7: return withLaneHelper(7, e);
            case 8: return withLaneHelper(8, e);
            case 9: return withLaneHelper(9, e);
            case 10: return withLaneHelper(10, e);
            case 11: return withLaneHelper(11, e);
            case 12: return withLaneHelper(12, e);
            case 13: return withLaneHelper(13, e);
            case 14: return withLaneHelper(14, e);
            case 15: return withLaneHelper(15, e);
            default: throw new IllegalArgumentException("Index " + i + " must be zero or positive, and less than " + VLENGTH);
        }
    }

    @ForceInline
    public ShortVector256 withLaneHelper(int i, short e) {
        return VectorSupport.insert(
                                VCLASS, LANE_TYPE_ORDINAL, VLENGTH,
                                this, i, (long)e,
                                (v, ix, bits) -> {
                                    short[] res = v.vec().clone();
                                    res[ix] = (short)bits;
                                    return v.vectorFactory(res);
                                });
    }

    // Mask

    static final class ShortMask256 extends AbstractMask<Short> {
        static final int VLENGTH = VSPECIES.laneCount();    // used by the JVM

        static final Class<Short> CTYPE = short.class; // used by the JVM

        ShortMask256(boolean[] bits) {
            this(bits, 0);
        }

        ShortMask256(boolean[] bits, int offset) {
            super(prepare(bits, offset));
        }

        ShortMask256(boolean val) {
            super(prepare(val));
        }

        private static boolean[] prepare(boolean[] bits, int offset) {
            boolean[] newBits = new boolean[VSPECIES.laneCount()];
            for (int i = 0; i < newBits.length; i++) {
                newBits[i] = bits[offset + i];
            }
            return newBits;
        }

        private static boolean[] prepare(boolean val) {
            boolean[] bits = new boolean[VSPECIES.laneCount()];
            Arrays.fill(bits, val);
            return bits;
        }

        @ForceInline
        final @Override
        public ShortSpecies vspecies() {
            // ISSUE:  This should probably be a @Stable
            // field inside AbstractMask, rather than
            // a megamorphic method.
            return VSPECIES;
        }

        @ForceInline
        boolean[] getBits() {
            return (boolean[])getPayload();
        }

        @Override
        ShortMask256 uOp(MUnOp f) {
            boolean[] res = new boolean[vspecies().laneCount()];
            boolean[] bits = getBits();
            for (int i = 0; i < res.length; i++) {
                res[i] = f.apply(i, bits[i]);
            }
            return new ShortMask256(res);
        }

        @Override
        ShortMask256 bOp(VectorMask<Short> m, MBinOp f) {
            boolean[] res = new boolean[vspecies().laneCount()];
            boolean[] bits = getBits();
            boolean[] mbits = ((ShortMask256)m).getBits();
            for (int i = 0; i < res.length; i++) {
                res[i] = f.apply(i, bits[i], mbits[i]);
            }
            return new ShortMask256(res);
        }

        @ForceInline
        @Override
        public final
        ShortVector256 toVector() {
            return (ShortVector256) super.toVectorTemplate();  // specialize
        }

        /**
         * Helper function for lane-wise mask conversions.
         * This function kicks in after intrinsic failure.
         */
        @ForceInline
        private final <E>
        VectorMask<E> defaultMaskCast(AbstractSpecies<E> dsp) {
            if (length() != dsp.laneCount())
                throw new IllegalArgumentException("VectorMask length and species length differ");
            boolean[] maskArray = toArray();
            return  dsp.maskFactory(maskArray).check(dsp);
        }

        @Override
        @ForceInline
        public <E> VectorMask<E> cast(VectorSpecies<E> dsp) {
            AbstractSpecies<E> species = (AbstractSpecies<E>) dsp;
            if (length() != species.laneCount())
                throw new IllegalArgumentException("VectorMask length and species length differ");

            return VectorSupport.convert(VectorSupport.VECTOR_OP_CAST,
                this.getClass(), LANE_TYPE_ORDINAL, VLENGTH,
                species.maskType(), species.laneTypeOrdinal(), VLENGTH,
                this, species,
                (m, s) -> s.maskFactory(m.toArray()).check(s));
        }

        @Override
        @ForceInline
        /*package-private*/
        ShortMask256 indexPartiallyInUpperRange(long offset, long limit) {
            return (ShortMask256) VectorSupport.indexPartiallyInUpperRange(
                ShortMask256.class, LANE_TYPE_ORDINAL, VLENGTH, offset, limit,
                (o, l) -> (ShortMask256) TRUE_MASK.indexPartiallyInRange(o, l));
        }

        // Unary operations

        @Override
        @ForceInline
        public ShortMask256 not() {
            return xor(maskAll(true));
        }

        @Override
        @ForceInline
        public ShortMask256 compress() {
            return (ShortMask256)VectorSupport.compressExpandOp(VectorSupport.VECTOR_OP_MASK_COMPRESS,
                ShortVector256.class, ShortMask256.class, LANE_TYPE_ORDINAL, VLENGTH, null, this,
                (v1, m1) -> VSPECIES.iota().compare(VectorOperators.LT, m1.trueCount()));
        }


        // Binary operations

        @Override
        @ForceInline
        public ShortMask256 and(VectorMask<Short> mask) {
            Objects.requireNonNull(mask);
            ShortMask256 m = (ShortMask256)mask;
            return VectorSupport.binaryOp(VECTOR_OP_AND, ShortMask256.class, null, LANEBITS_TYPE_ORDINAL, VLENGTH,
                                          this, m, null,
                                          (m1, m2, vm) -> m1.bOp(m2, (i, a, b) -> a & b));
        }

        @Override
        @ForceInline
        public ShortMask256 or(VectorMask<Short> mask) {
            Objects.requireNonNull(mask);
            ShortMask256 m = (ShortMask256)mask;
            return VectorSupport.binaryOp(VECTOR_OP_OR, ShortMask256.class, null, LANEBITS_TYPE_ORDINAL, VLENGTH,
                                          this, m, null,
                                          (m1, m2, vm) -> m1.bOp(m2, (i, a, b) -> a | b));
        }

        @Override
        @ForceInline
        public ShortMask256 xor(VectorMask<Short> mask) {
            Objects.requireNonNull(mask);
            ShortMask256 m = (ShortMask256)mask;
            return VectorSupport.binaryOp(VECTOR_OP_XOR, ShortMask256.class, null, LANEBITS_TYPE_ORDINAL, VLENGTH,
                                          this, m, null,
                                          (m1, m2, vm) -> m1.bOp(m2, (i, a, b) -> a ^ b));
        }

        // Mask Query operations

        @Override
        @ForceInline
        public int trueCount() {
            return (int) VectorSupport.maskReductionCoerced(VECTOR_OP_MASK_TRUECOUNT, ShortMask256.class, LANEBITS_TYPE_ORDINAL, VLENGTH, this,
                                                      (m) -> trueCountHelper(m.getBits()));
        }

        @Override
        @ForceInline
        public int firstTrue() {
            return (int) VectorSupport.maskReductionCoerced(VECTOR_OP_MASK_FIRSTTRUE, ShortMask256.class, LANEBITS_TYPE_ORDINAL, VLENGTH, this,
                                                      (m) -> firstTrueHelper(m.getBits()));
        }

        @Override
        @ForceInline
        public int lastTrue() {
            return (int) VectorSupport.maskReductionCoerced(VECTOR_OP_MASK_LASTTRUE, ShortMask256.class, LANEBITS_TYPE_ORDINAL, VLENGTH, this,
                                                      (m) -> lastTrueHelper(m.getBits()));
        }

        @Override
        @ForceInline
        public long toLong() {
            if (length() > Long.SIZE) {
                throw new UnsupportedOperationException("too many lanes for one long");
            }
            return VectorSupport.maskReductionCoerced(VECTOR_OP_MASK_TOLONG, ShortMask256.class, LANEBITS_TYPE_ORDINAL, VLENGTH, this,
                                                      (m) -> toLongHelper(m.getBits()));
        }

        // laneIsSet

        @Override
        @ForceInline
        public boolean laneIsSet(int i) {
            Objects.checkIndex(i, length());
            return VectorSupport.extract(ShortMask256.class, LANE_TYPE_ORDINAL, VLENGTH,
                                         this, i, (m, idx) -> (m.getBits()[idx] ? 1L : 0L)) == 1L;
        }

        // Reductions

        @Override
        @ForceInline
        public boolean anyTrue() {
            return VectorSupport.test(BT_ne, ShortMask256.class, LANEBITS_TYPE_ORDINAL, VLENGTH,
                                         this, vspecies().maskAll(true),
                                         (m, __) -> anyTrueHelper(((ShortMask256)m).getBits()));
        }

        @Override
        @ForceInline
        public boolean allTrue() {
            return VectorSupport.test(BT_overflow, ShortMask256.class, LANEBITS_TYPE_ORDINAL, VLENGTH,
                                         this, vspecies().maskAll(true),
                                         (m, __) -> allTrueHelper(((ShortMask256)m).getBits()));
        }

        @ForceInline
        /*package-private*/
        static ShortMask256 maskAll(boolean bit) {
            return VectorSupport.fromBitsCoerced(ShortMask256.class, LANEBITS_TYPE_ORDINAL, VLENGTH,
                                                 (bit ? -1 : 0), MODE_BROADCAST, null,
                                                 (v, __) -> (v != 0 ? TRUE_MASK : FALSE_MASK));
        }
        private static final ShortMask256  TRUE_MASK = new ShortMask256(true);
        private static final ShortMask256 FALSE_MASK = new ShortMask256(false);

    }

    // Shuffle

    static final class ShortShuffle256 extends AbstractShuffle<Short> {
        static final int VLENGTH = VSPECIES.laneCount();    // used by the JVM

        static final Class<Short> CTYPE = short.class; // used by the JVM

        ShortShuffle256(short[] indices) {
            super(indices);
            assert(VLENGTH == indices.length);
            assert(indicesInRange(indices));
        }

        ShortShuffle256(int[] indices, int i) {
            this(prepare(indices, i));
        }

        ShortShuffle256(IntUnaryOperator fn) {
            this(prepare(fn));
        }

        short[] indices() {
            return (short[])getPayload();
        }

        @Override
        @ForceInline
        public ShortSpecies vspecies() {
            return VSPECIES;
        }

        static {
            // There must be enough bits in the shuffle lanes to encode
            // VLENGTH valid indexes and VLENGTH exceptional ones.
            assert(VLENGTH < Short.MAX_VALUE);
            assert(Short.MIN_VALUE <= -VLENGTH);
        }
        static final ShortShuffle256 IOTA = new ShortShuffle256(IDENTITY);

        @Override
        @ForceInline
        public ShortVector256 toVector() {
            return toBitsVector();
        }

        @Override
        @ForceInline
        ShortVector256 toBitsVector() {
            return (ShortVector256) super.toBitsVectorTemplate();
        }

        @Override
        ShortVector256 toBitsVector0() {
            return ((ShortVector256) vspecies().asIntegral().dummyVector()).vectorFactory(indices());
        }

        @Override
        @ForceInline
        public int laneSource(int i) {
            return (int)toBitsVector().lane(i);
        }

        @Override
        @ForceInline
        public void intoArray(int[] a, int offset) {
            VectorSpecies<Integer> species = IntVector.SPECIES_256;
            Vector<Short> v = toBitsVector();
            v.convertShape(VectorOperators.S2I, species, 0)
                    .reinterpretAsInts()
                    .intoArray(a, offset);
            v.convertShape(VectorOperators.S2I, species, 1)
                    .reinterpretAsInts()
                    .intoArray(a, offset + species.length());
        }

        @Override
        @ForceInline
        public void intoMemorySegment(MemorySegment ms, long offset, ByteOrder bo) {
            VectorSpecies<Integer> species = IntVector.SPECIES_256;
            Vector<Short> v = toBitsVector();
            v.convertShape(VectorOperators.S2I, species, 0)
                    .reinterpretAsInts()
                    .intoMemorySegment(ms, offset, bo);
            v.convertShape(VectorOperators.S2I, species, 1)
                    .reinterpretAsInts()
                    .intoMemorySegment(ms, offset + species.vectorByteSize(), bo);
         }

        @Override
        @ForceInline
        public final ShortMask256 laneIsValid() {
            return (ShortMask256) toBitsVector().compare(VectorOperators.GE, 0)
                    .cast(vspecies());
        }

        @ForceInline
        @Override
        public final ShortShuffle256 rearrange(VectorShuffle<Short> shuffle) {
            ShortShuffle256 concreteShuffle = (ShortShuffle256) shuffle;
            return (ShortShuffle256) toBitsVector().rearrange(concreteShuffle)
                    .toShuffle(vspecies(), false);
        }

        @ForceInline
        @Override
        public final ShortShuffle256 wrapIndexes() {
            ShortVector256 v = toBitsVector();
            if ((length() & (length() - 1)) == 0) {
                v = (ShortVector256) v.lanewise(VectorOperators.AND, length() - 1);
            } else {
                v = (ShortVector256) v.blend(v.lanewise(VectorOperators.ADD, length()),
                            v.compare(VectorOperators.LT, 0));
            }
            return (ShortShuffle256) v.toShuffle(vspecies(), false);
        }

        private static short[] prepare(int[] indices, int offset) {
            short[] a = new short[VLENGTH];
            for (int i = 0; i < VLENGTH; i++) {
                int si = indices[offset + i];
                si = partiallyWrapIndex(si, VLENGTH);
                a[i] = (short)si;
            }
            return a;
        }

        private static short[] prepare(IntUnaryOperator f) {
            short[] a = new short[VLENGTH];
            for (int i = 0; i < VLENGTH; i++) {
                int si = f.applyAsInt(i);
                si = partiallyWrapIndex(si, VLENGTH);
                a[i] = (short)si;
            }
            return a;
        }

        private static boolean indicesInRange(short[] indices) {
            int length = indices.length;
            for (short si : indices) {
                if (si >= (short)length || si < (short)(-length)) {
                    String msg = ("index "+si+"out of range ["+length+"] in "+
                                  java.util.Arrays.toString(indices));
                    throw new AssertionError(msg);
                }
            }
            return true;
        }
    }

    // ================================================

    // Specialized low-level memory operations.

    @ForceInline
    @Override
    final
    ShortVector fromArray0(short[] a, int offset) {
        return super.fromArray0Template(a, offset);  // specialize
    }

    @ForceInline
    @Override
    final
    ShortVector fromArray0(short[] a, int offset, VectorMask<Short> m, int offsetInRange) {
        return super.fromArray0Template(ShortMask256.class, a, offset, (ShortMask256) m, offsetInRange);  // specialize
    }

    @ForceInline
    @Override
    final
    ShortVector fromArray0(short[] a, int offset, int[] indexMap, int mapOffset, VectorMask<Short> m) {
        return super.fromArray0Template(ShortMask256.class, a, offset, indexMap, mapOffset, (ShortMask256) m);
    }

    @ForceInline
    @Override
    final
    ShortVector fromCharArray0(char[] a, int offset) {
        return super.fromCharArray0Template(a, offset);  // specialize
    }

    @ForceInline
    @Override
    final
    ShortVector fromCharArray0(char[] a, int offset, VectorMask<Short> m, int offsetInRange) {
        return super.fromCharArray0Template(ShortMask256.class, a, offset, (ShortMask256) m, offsetInRange);  // specialize
    }


    @ForceInline
    @Override
    final
    ShortVector fromMemorySegment0(MemorySegment ms, long offset) {
        return super.fromMemorySegment0Template(ms, offset);  // specialize
    }

    @ForceInline
    @Override
    final
    ShortVector fromMemorySegment0(MemorySegment ms, long offset, VectorMask<Short> m, int offsetInRange) {
        return super.fromMemorySegment0Template(ShortMask256.class, ms, offset, (ShortMask256) m, offsetInRange);  // specialize
    }

    @ForceInline
    @Override
    final
    void intoArray0(short[] a, int offset) {
        super.intoArray0Template(a, offset);  // specialize
    }

    @ForceInline
    @Override
    final
    void intoArray0(short[] a, int offset, VectorMask<Short> m) {
        super.intoArray0Template(ShortMask256.class, a, offset, (ShortMask256) m);
    }



    @ForceInline
    @Override
    final
    void intoMemorySegment0(MemorySegment ms, long offset, VectorMask<Short> m) {
        super.intoMemorySegment0Template(ShortMask256.class, ms, offset, (ShortMask256) m);
    }

    @ForceInline
    @Override
    final
    void intoCharArray0(char[] a, int offset, VectorMask<Short> m) {
        super.intoCharArray0Template(ShortMask256.class, a, offset, (ShortMask256) m);
    }

    // End of specialized low-level memory operations.

    // ================================================

}

