/*
 * Copyright (c) 2020, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */
package jdk.internal.jextract.impl;

import javax.tools.JavaFileObject;
import java.lang.constant.ClassDesc;
import java.util.List;
import java.util.function.Consumer;

/**
 * Superclass for .java source generator classes.
 */
abstract class ClassSourceBuilder extends JavaSourceBuilder {

    enum Kind {
        CLASS("class"),
        INTERFACE("interface");

        final String kindName;

        Kind(String kindName) {
            this.kindName = kindName;
        }
    }

    final Kind kind;
    final ClassDesc desc;
    protected final JavaSourceBuilder enclosing;

    // code buffer
    private StringBuilder sb = new StringBuilder();
    // current line alignment (number of 4-spaces)
    private int align;

    ClassSourceBuilder(JavaSourceBuilder enclosing, Kind kind, String name) {
        this.enclosing = enclosing;
        this.align = (enclosing instanceof ClassSourceBuilder) ?
            ((ClassSourceBuilder) enclosing).align : 0;
        this.kind = kind;
        this.desc = ClassDesc.of(enclosing.packageName(), enclosing.uniqueNestedClassName(name));
    }

    boolean isNested() {
        return enclosing instanceof ClassSourceBuilder;
    }

    String className() {
        return desc.displayName();
    }

    String fullName() {
        return isNested() ?
                ((ClassSourceBuilder)enclosing).className() + "." + className() :
                className();
    }

    @Override
    public final String packageName() {
        return desc.packageName();
    }

    String superClass() {
        return null;
    }

    String mods() {
        return (!isNested() || kind == Kind.INTERFACE) ?
                    "public " : "public static ";
    }

    void classBegin() {
        if (isNested()) {
            incrAlign();
        }
        emitPackagePrefix();
        emitImportSection();

        indent();
        append(mods());
        append(kind.kindName + " " + className());
        if (superClass() != null) {
            append(" extends ");
            append(superClass());
        }
        append(" {\n\n");
    }

    JavaSourceBuilder classEnd() {
        indent();
        append("}\n\n");
        if (isNested()) {
            decrAlign();
            ((ClassSourceBuilder)enclosing).append(build());
            sb = null;
        }
        return enclosing;
    }

    @Override
    public List<JavaFileObject> toFiles() {
        if (isNested()) {
            throw new UnsupportedOperationException("Nested builder!");
        }
        String res = build();
        sb = null;
        return List.of(Utils.fileFromString(packageName(), className(), res));
    }

    // Internal generation helpers (used by other builders)

    void append(String s) {
        sb.append(s);
    }

    void append(char c) {
        sb.append(c);
    }

    void append(boolean b) {
        sb.append(b);
    }

    void append(long l) {
        sb.append(l);
    }

    void indent() {
        for (int i = 0; i < align; i++) {
            append("    ");
        }
    }

    void incrAlign() {
        align++;
    }

    void decrAlign() {
        align--;
    }

    String build() {
        String s = sb.toString();
        return s;
    }

    // is the name enclosed enclosed by a class of the same name?
    boolean isEnclosedBySameName(String name) {
        return className().equals(name) ||
                (isNested() && enclosing.isEnclosedBySameName(name));
    }

    protected void emitPackagePrefix() {
        if (!isNested()) {
            assert packageName().indexOf('/') == -1 : "package name invalid: " + packageName();
            append("// Generated by jextract\n\n");
            if (!packageName().isEmpty()) {
                append("package ");
                append(packageName());
                append(";\n\n");
            }
        }
    }

    protected void emitImportSection() {
        if (!isNested()) {
            append("import java.lang.invoke.MethodHandle;\n");
            append("import java.lang.invoke.VarHandle;\n");
            append("import java.nio.ByteOrder;\n");
            append("import jdk.incubator.foreign.*;\n");
            append("import static jdk.incubator.foreign.CLinker.*;\n");
        }
    }

    protected void emitGetter(String mods, Class<?> type, String name, String access, boolean nullCheck, String symbolName) {
        incrAlign();
        indent();
        append(mods + " " + type.getSimpleName() + " " +name + "() {\n");
        incrAlign();
        indent();
        append("return ");
        if (nullCheck) {
            append("RuntimeHelper.requireNonNull(");
        }
        append(access);
        if (nullCheck) {
            append(",\"");
            append(symbolName);
            append("\")");
        }
        append(";\n");
        decrAlign();
        indent();
        append("}\n");
        decrAlign();
    }

    protected void emitGetter(String mods, Class<?> type, String name, String access) {
        emitGetter(mods, type, name, access, false, null);
    }

    ToplevelBuilder toplevel() {
        JavaSourceBuilder encl = enclosing;
        while (encl instanceof ClassSourceBuilder) {
            encl = ((ClassSourceBuilder) encl).enclosing;
        }
        return (ToplevelBuilder)encl;
    }

    @Override
    protected void emitWithConstantClass(Consumer<ConstantBuilder> constantConsumer) {
        enclosing.emitWithConstantClass(constantConsumer);
    }
}
