/*
 * Copyright (c) 2015, 2025, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */
package java.lang.invoke;

import jdk.internal.value.ValueClass;
import jdk.internal.util.Preconditions;
import jdk.internal.vm.annotation.ForceInline;

import java.lang.invoke.VarHandle.VarHandleDesc;
import java.lang.reflect.Field;
import java.util.Objects;
import java.util.Optional;

import static java.lang.invoke.MethodHandleStatics.UNSAFE;

// -- This file was mechanically generated: Do not edit! -- //

final class VarHandleFlatValues {

    static sealed class FieldInstanceReadOnly extends VarHandle {
        final long fieldOffset;
        final Class<?> receiverType;
        final Class<?> fieldType;
        final boolean nullRestricted;
        final int layout; // Unsafe.fieldLayout

        FieldInstanceReadOnly(Class<?> receiverType, long fieldOffset, Class<?> fieldType, boolean nullRestricted, int layout) {
            this(receiverType, fieldOffset, fieldType, nullRestricted, layout, FieldInstanceReadOnly.FORM, false);
        }

        protected FieldInstanceReadOnly(Class<?> receiverType, long fieldOffset, Class<?> fieldType, boolean nullRestricted, int layout,
                                        VarForm form, boolean exact) {
            super(form, exact);
            this.fieldOffset = fieldOffset;
            this.receiverType = receiverType;
            this.fieldType = fieldType;
            this.nullRestricted = nullRestricted;
            this.layout = layout;
        }

        @Override
        public FieldInstanceReadOnly withInvokeExactBehavior() {
            return hasInvokeExactBehavior()
                ? this
                : new FieldInstanceReadOnly(receiverType, fieldOffset, fieldType, nullRestricted, layout, vform, true);
        }

        @Override
        public FieldInstanceReadOnly withInvokeBehavior() {
            return !hasInvokeExactBehavior()
                ? this
                : new FieldInstanceReadOnly(receiverType, fieldOffset, fieldType, nullRestricted, layout, vform, false);
        }

        @Override
        final MethodType accessModeTypeUncached(AccessType at) {
            return at.accessModeType(receiverType, fieldType);
        }

        @Override
        public Optional<VarHandleDesc> describeConstable() {
            var receiverTypeRef = receiverType.describeConstable();
            var fieldTypeRef = fieldType.describeConstable();
            if (!receiverTypeRef.isPresent() || !fieldTypeRef.isPresent())
                return Optional.empty();

            // Reflect on this VarHandle to extract the field name
            String name = VarHandles.getFieldFromReceiverAndOffset(
                receiverType, fieldOffset, fieldType).getName();
            return Optional.of(VarHandleDesc.ofField(receiverTypeRef.get(), name, fieldTypeRef.get()));
        }

        @ForceInline
        static Object get(VarHandle ob, Object holder) {
            FieldInstanceReadOnly handle = (FieldInstanceReadOnly)ob;
            Object value = UNSAFE.getFlatValue(Objects.requireNonNull(handle.receiverType.cast(holder)),
                                 handle.fieldOffset, handle.layout, handle.fieldType);
            return value;
        }

        @ForceInline
        static Object getVolatile(VarHandle ob, Object holder) {
            FieldInstanceReadOnly handle = (FieldInstanceReadOnly)ob;
            Object value = UNSAFE.getFlatValueVolatile(Objects.requireNonNull(handle.receiverType.cast(holder)),
                                 handle.fieldOffset, handle.layout, handle.fieldType);
            return value;
        }

        @ForceInline
        static Object getOpaque(VarHandle ob, Object holder) {
            FieldInstanceReadOnly handle = (FieldInstanceReadOnly)ob;
            Object value = UNSAFE.getFlatValueOpaque(Objects.requireNonNull(handle.receiverType.cast(holder)),
                                 handle.fieldOffset, handle.layout, handle.fieldType);
            return value;
        }

        @ForceInline
        static Object getAcquire(VarHandle ob, Object holder) {
            FieldInstanceReadOnly handle = (FieldInstanceReadOnly)ob;
            Object value = UNSAFE.getFlatValueAcquire(Objects.requireNonNull(handle.receiverType.cast(holder)),
                                 handle.fieldOffset, handle.layout, handle.fieldType);
            return value;
        }

        static final VarForm FORM = new VarForm(FieldInstanceReadOnly.class, Object.class, Object.class);
    }

    static final class FieldInstanceReadWrite extends FieldInstanceReadOnly {
        FieldInstanceReadWrite(Class<?> receiverType, long fieldOffset, Class<?> fieldType, boolean nullRestricted, int layout) {
            this(receiverType, fieldOffset, fieldType, nullRestricted, layout, false);
        }

        private FieldInstanceReadWrite(Class<?> receiverType, long fieldOffset, Class<?> fieldType, boolean nullRestricted, int layout,
                                       boolean exact) {
            super(receiverType, fieldOffset, fieldType, nullRestricted, layout, FieldInstanceReadWrite.FORM, exact);
        }

        @Override
        public FieldInstanceReadWrite withInvokeExactBehavior() {
            return hasInvokeExactBehavior()
                ? this
                : new FieldInstanceReadWrite(receiverType, fieldOffset, fieldType, nullRestricted, layout, true);
        }

        @Override
        public FieldInstanceReadWrite withInvokeBehavior() {
            return !hasInvokeExactBehavior()
                ? this
                : new FieldInstanceReadWrite(receiverType, fieldOffset, fieldType, nullRestricted, layout, false);
        }

        @ForceInline
        static Object checkCast(FieldInstanceReadWrite handle, Object value) {
            return handle.fieldType.cast(value);
        }

        @ForceInline
        static void set(VarHandle ob, Object holder, Object value) {
            FieldInstanceReadWrite handle = (FieldInstanceReadWrite)ob;
            UNSAFE.putFlatValue(Objects.requireNonNull(handle.receiverType.cast(holder)),
                             handle.fieldOffset, handle.layout, handle.fieldType,
                             checkCast(handle, value));
        }

        @ForceInline
        static void setVolatile(VarHandle ob, Object holder, Object value) {
            FieldInstanceReadWrite handle = (FieldInstanceReadWrite)ob;
            UNSAFE.putFlatValueVolatile(Objects.requireNonNull(handle.receiverType.cast(holder)),
                                     handle.fieldOffset, handle.layout, handle.fieldType,
                                     checkCast(handle, value));
        }

        @ForceInline
        static void setOpaque(VarHandle ob, Object holder, Object value) {
            FieldInstanceReadWrite handle = (FieldInstanceReadWrite)ob;
            UNSAFE.putFlatValueOpaque(Objects.requireNonNull(handle.receiverType.cast(holder)),
                                   handle.fieldOffset, handle.layout, handle.fieldType,
                                   checkCast(handle, value));
        }

        @ForceInline
        static void setRelease(VarHandle ob, Object holder, Object value) {
            FieldInstanceReadWrite handle = (FieldInstanceReadWrite)ob;
            UNSAFE.putFlatValueRelease(Objects.requireNonNull(handle.receiverType.cast(holder)),
                                    handle.fieldOffset, handle.layout, handle.fieldType,
                                    checkCast(handle, value));
        }

        @ForceInline
        static boolean compareAndSet(VarHandle ob, Object holder, Object expected, Object value) {
            FieldInstanceReadWrite handle = (FieldInstanceReadWrite)ob;
            return UNSAFE.compareAndSetFlatValue(Objects.requireNonNull(handle.receiverType.cast(holder)),
                                               handle.fieldOffset, handle.layout, handle.fieldType,
                                               checkCast(handle, expected),
                                               checkCast(handle, value));
        }

        @ForceInline
        static Object compareAndExchange(VarHandle ob, Object holder, Object expected, Object value) {
            FieldInstanceReadWrite handle = (FieldInstanceReadWrite)ob;
            return UNSAFE.compareAndExchangeFlatValue(Objects.requireNonNull(handle.receiverType.cast(holder)),
                                               handle.fieldOffset, handle.layout, handle.fieldType,
                                               checkCast(handle, expected),
                                               checkCast(handle, value));
        }

        @ForceInline
        static Object compareAndExchangeAcquire(VarHandle ob, Object holder, Object expected, Object value) {
            FieldInstanceReadWrite handle = (FieldInstanceReadWrite)ob;
            return UNSAFE.compareAndExchangeFlatValueAcquire(Objects.requireNonNull(handle.receiverType.cast(holder)),
                                               handle.fieldOffset, handle.layout, handle.fieldType,
                                               checkCast(handle, expected),
                                               checkCast(handle, value));
        }

        @ForceInline
        static Object compareAndExchangeRelease(VarHandle ob, Object holder, Object expected, Object value) {
            FieldInstanceReadWrite handle = (FieldInstanceReadWrite)ob;
            return UNSAFE.compareAndExchangeFlatValueRelease(Objects.requireNonNull(handle.receiverType.cast(holder)),
                                               handle.fieldOffset, handle.layout, handle.fieldType,
                                               checkCast(handle, expected),
                                               checkCast(handle, value));
        }

        @ForceInline
        static boolean weakCompareAndSetPlain(VarHandle ob, Object holder, Object expected, Object value) {
            FieldInstanceReadWrite handle = (FieldInstanceReadWrite)ob;
            return UNSAFE.weakCompareAndSetFlatValuePlain(Objects.requireNonNull(handle.receiverType.cast(holder)),
                                               handle.fieldOffset, handle.layout, handle.fieldType,
                                               checkCast(handle, expected),
                                               checkCast(handle, value));
        }

        @ForceInline
        static boolean weakCompareAndSet(VarHandle ob, Object holder, Object expected, Object value) {
            FieldInstanceReadWrite handle = (FieldInstanceReadWrite)ob;
            return UNSAFE.weakCompareAndSetFlatValue(Objects.requireNonNull(handle.receiverType.cast(holder)),
                                               handle.fieldOffset, handle.layout, handle.fieldType,
                                               checkCast(handle, expected),
                                               checkCast(handle, value));
        }

        @ForceInline
        static boolean weakCompareAndSetAcquire(VarHandle ob, Object holder, Object expected, Object value) {
            FieldInstanceReadWrite handle = (FieldInstanceReadWrite)ob;
            return UNSAFE.weakCompareAndSetFlatValueAcquire(Objects.requireNonNull(handle.receiverType.cast(holder)),
                                               handle.fieldOffset, handle.layout, handle.fieldType,
                                               checkCast(handle, expected),
                                               checkCast(handle, value));
        }

        @ForceInline
        static boolean weakCompareAndSetRelease(VarHandle ob, Object holder, Object expected, Object value) {
            FieldInstanceReadWrite handle = (FieldInstanceReadWrite)ob;
            return UNSAFE.weakCompareAndSetFlatValueRelease(Objects.requireNonNull(handle.receiverType.cast(holder)),
                                               handle.fieldOffset, handle.layout, handle.fieldType,
                                               checkCast(handle, expected),
                                               checkCast(handle, value));
        }

        @ForceInline
        static Object getAndSet(VarHandle ob, Object holder, Object value) {
            FieldInstanceReadWrite handle = (FieldInstanceReadWrite)ob;
            return UNSAFE.getAndSetFlatValue(Objects.requireNonNull(handle.receiverType.cast(holder)),
                                          handle.fieldOffset, handle.layout, handle.fieldType,
                                          checkCast(handle, value));
        }

        @ForceInline
        static Object getAndSetAcquire(VarHandle ob, Object holder, Object value) {
            FieldInstanceReadWrite handle = (FieldInstanceReadWrite)ob;
            return UNSAFE.getAndSetFlatValueAcquire(Objects.requireNonNull(handle.receiverType.cast(holder)),
                                          handle.fieldOffset, handle.layout, handle.fieldType,
                                          checkCast(handle, value));
        }

        @ForceInline
        static Object getAndSetRelease(VarHandle ob, Object holder, Object value) {
            FieldInstanceReadWrite handle = (FieldInstanceReadWrite)ob;
            return UNSAFE.getAndSetFlatValueRelease(Objects.requireNonNull(handle.receiverType.cast(holder)),
                                          handle.fieldOffset, handle.layout, handle.fieldType,
                                          checkCast(handle, value));
        }

        static final VarForm FORM = new VarForm(FieldInstanceReadWrite.class, Object.class, Object.class);
    }


}
