/*
 * Copyright (c) 2015, 2025, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */
package java.lang.invoke;

import jdk.internal.value.ValueClass;
import jdk.internal.util.Preconditions;
import jdk.internal.vm.annotation.ForceInline;

import java.lang.invoke.VarHandle.VarHandleDesc;
import java.lang.reflect.Field;
import java.util.Objects;
import java.util.Optional;

import static java.lang.invoke.MethodHandleStatics.UNSAFE;

// -- This file was mechanically generated: Do not edit! -- //

final class VarHandleNonAtomicReferences {

    static sealed class FieldInstanceReadOnly extends VarHandle {
        final long fieldOffset;
        final Class<?> receiverType;
        final Class<?> fieldType;
        final boolean nullRestricted;

        FieldInstanceReadOnly(Class<?> receiverType, long fieldOffset, Class<?> fieldType, boolean nullRestricted) {
            this(receiverType, fieldOffset, fieldType, nullRestricted, FieldInstanceReadOnly.FORM, false);
        }

        protected FieldInstanceReadOnly(Class<?> receiverType, long fieldOffset, Class<?> fieldType, boolean nullRestricted,
                                        VarForm form, boolean exact) {
            super(form, exact);
            this.fieldOffset = fieldOffset;
            this.receiverType = receiverType;
            this.fieldType = fieldType;
            this.nullRestricted = nullRestricted;
        }

        @Override
        public FieldInstanceReadOnly withInvokeExactBehavior() {
            return hasInvokeExactBehavior()
                ? this
                : new FieldInstanceReadOnly(receiverType, fieldOffset, fieldType, nullRestricted, vform, true);
        }

        @Override
        public FieldInstanceReadOnly withInvokeBehavior() {
            return !hasInvokeExactBehavior()
                ? this
                : new FieldInstanceReadOnly(receiverType, fieldOffset, fieldType, nullRestricted, vform, false);
        }

        @Override
        final MethodType accessModeTypeUncached(AccessType at) {
            return at.accessModeType(receiverType, fieldType);
        }

        @Override
        public Optional<VarHandleDesc> describeConstable() {
            var receiverTypeRef = receiverType.describeConstable();
            var fieldTypeRef = fieldType.describeConstable();
            if (!receiverTypeRef.isPresent() || !fieldTypeRef.isPresent())
                return Optional.empty();

            // Reflect on this VarHandle to extract the field name
            String name = VarHandles.getFieldFromReceiverAndOffset(
                receiverType, fieldOffset, fieldType).getName();
            return Optional.of(VarHandleDesc.ofField(receiverTypeRef.get(), name, fieldTypeRef.get()));
        }

        @ForceInline
        static Object get(VarHandle ob, Object holder) {
            FieldInstanceReadOnly handle = (FieldInstanceReadOnly)ob;
            Object value = UNSAFE.getReference(Objects.requireNonNull(handle.receiverType.cast(holder)),
                                 handle.fieldOffset);
            if (value == null && handle.nullRestricted) {
                throw new NullPointerException("Uninitialized null-restricted field");
            }
            return value;
        }


        static final VarForm FORM = new VarForm(FieldInstanceReadOnly.class, Object.class, Object.class);
    }

    static final class FieldInstanceReadWrite extends FieldInstanceReadOnly {
        FieldInstanceReadWrite(Class<?> receiverType, long fieldOffset, Class<?> fieldType, boolean nullRestricted) {
            this(receiverType, fieldOffset, fieldType, nullRestricted, false);
        }

        private FieldInstanceReadWrite(Class<?> receiverType, long fieldOffset, Class<?> fieldType, boolean nullRestricted,
                                       boolean exact) {
            super(receiverType, fieldOffset, fieldType, nullRestricted, FieldInstanceReadWrite.FORM, exact);
        }

        @Override
        public FieldInstanceReadWrite withInvokeExactBehavior() {
            return hasInvokeExactBehavior()
                ? this
                : new FieldInstanceReadWrite(receiverType, fieldOffset, fieldType, nullRestricted, true);
        }

        @Override
        public FieldInstanceReadWrite withInvokeBehavior() {
            return !hasInvokeExactBehavior()
                ? this
                : new FieldInstanceReadWrite(receiverType, fieldOffset, fieldType, nullRestricted, false);
        }

        @ForceInline
        static Object checkCast(FieldInstanceReadWrite handle, Object value) {
            if (value == null && handle.nullRestricted)
                throw new NullPointerException("Uninitialized null-restricted field");
            return handle.fieldType.cast(value);
        }

        @ForceInline
        static void set(VarHandle ob, Object holder, Object value) {
            FieldInstanceReadWrite handle = (FieldInstanceReadWrite)ob;
            UNSAFE.putReference(Objects.requireNonNull(handle.receiverType.cast(holder)),
                             handle.fieldOffset,
                             checkCast(handle, value));
        }


        static final VarForm FORM = new VarForm(FieldInstanceReadWrite.class, Object.class, Object.class);
    }

    static sealed class FieldStaticReadOnly extends VarHandle {
        final Class<?> declaringClass;
        final Object base;
        final long fieldOffset;
        final Class<?> fieldType;
        final boolean nullRestricted;

        FieldStaticReadOnly(Class<?> declaringClass, Object base, long fieldOffset, Class<?> fieldType, boolean nullRestricted) {
            this(declaringClass, base, fieldOffset, fieldType, nullRestricted, FieldStaticReadOnly.FORM, false);
        }

        protected FieldStaticReadOnly(Class<?> declaringClass, Object base, long fieldOffset, Class<?> fieldType, boolean nullRestricted,
                                      VarForm form, boolean exact) {
            super(form, exact);
            this.declaringClass = declaringClass;
            this.base = base;
            this.fieldOffset = fieldOffset;
            this.fieldType = fieldType;
            this.nullRestricted = nullRestricted;
        }

        @Override
        public FieldStaticReadOnly withInvokeExactBehavior() {
            return hasInvokeExactBehavior()
                ? this
                : new FieldStaticReadOnly(declaringClass, base, fieldOffset, fieldType, nullRestricted, vform, true);
        }

        @Override
        public FieldStaticReadOnly withInvokeBehavior() {
            return !hasInvokeExactBehavior()
                ? this
                : new FieldStaticReadOnly(declaringClass, base, fieldOffset, fieldType, nullRestricted, vform, false);
        }

        @Override
        public Optional<VarHandleDesc> describeConstable() {
            var fieldTypeRef = fieldType.describeConstable();
            if (!fieldTypeRef.isPresent())
                return Optional.empty();

            // Reflect on this VarHandle to extract the field name
            var staticField = VarHandles.getStaticFieldFromBaseAndOffset(
                declaringClass, fieldOffset, fieldType);
            var declaringTypeRef = declaringClass.describeConstable();
            if (!declaringTypeRef.isPresent())
                return Optional.empty();
            return Optional.of(VarHandleDesc.ofStaticField(declaringTypeRef.get(), staticField.getName(), fieldTypeRef.get()));
        }

        @Override
        final MethodType accessModeTypeUncached(AccessType at) {
            return at.accessModeType(null, fieldType);
        }

        @ForceInline
        static Object get(VarHandle ob) {
            FieldStaticReadOnly handle = (FieldStaticReadOnly) ob.target();
            Object value = UNSAFE.getReference(handle.base,
                                 handle.fieldOffset);
            if (value == null && handle.nullRestricted) {
                throw new NullPointerException("Uninitialized null-restricted field");
            }
            return value;
        }


        static final VarForm FORM = new VarForm(FieldStaticReadOnly.class, null, Object.class);
    }

    static final class FieldStaticReadWrite extends FieldStaticReadOnly {

        FieldStaticReadWrite(Class<?> declaringClass, Object base, long fieldOffset, Class<?> fieldType, boolean nullRestricted) {
            this(declaringClass, base, fieldOffset, fieldType, nullRestricted, false);
        }

        private FieldStaticReadWrite(Class<?> declaringClass, Object base, long fieldOffset, Class<?> fieldType, boolean nullRestricted,
                                     boolean exact) {
            super(declaringClass, base, fieldOffset, fieldType, nullRestricted, FieldStaticReadWrite.FORM, exact);
        }

        @Override
        public FieldStaticReadWrite withInvokeExactBehavior() {
            return hasInvokeExactBehavior()
                ? this
                : new FieldStaticReadWrite(declaringClass, base, fieldOffset, fieldType, nullRestricted, true);
        }

        @Override
        public FieldStaticReadWrite withInvokeBehavior() {
            return !hasInvokeExactBehavior()
                ? this
                : new FieldStaticReadWrite(declaringClass, base, fieldOffset, fieldType, nullRestricted, false);
        }

        @ForceInline
        static Object checkCast(FieldStaticReadWrite handle, Object value) {
            return handle.fieldType.cast(value);
        }

        @ForceInline
        static void set(VarHandle ob, Object value) {
            FieldStaticReadWrite handle = (FieldStaticReadWrite) ob.target();
            UNSAFE.putReference(handle.base,
                             handle.fieldOffset,
                             checkCast(handle, value));
        }


        static final VarForm FORM = new VarForm(FieldStaticReadWrite.class, null, Object.class);
    }

}
