/*
 * Copyright (c) 2015, 2025, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */
package java.lang.invoke;

import jdk.internal.util.Preconditions;
import jdk.internal.vm.annotation.ForceInline;

import java.lang.invoke.VarHandle.VarHandleDesc;
import java.lang.reflect.Field;
import java.util.Objects;
import java.util.Optional;

import static java.lang.invoke.MethodHandleStatics.UNSAFE;

// -- This file was mechanically generated: Do not edit! -- //

final class VarHandleDoubles {

    static sealed class FieldInstanceReadOnly extends VarHandle {
        final long fieldOffset;
        final Class<?> receiverType;

        FieldInstanceReadOnly(Class<?> receiverType, long fieldOffset) {
            this(receiverType, fieldOffset, FieldInstanceReadOnly.FORM, false);
        }

        protected FieldInstanceReadOnly(Class<?> receiverType, long fieldOffset,
                                        VarForm form, boolean exact) {
            super(form, exact);
            this.fieldOffset = fieldOffset;
            this.receiverType = receiverType;
        }

        @Override
        public FieldInstanceReadOnly withInvokeExactBehavior() {
            return hasInvokeExactBehavior()
                ? this
                : new FieldInstanceReadOnly(receiverType, fieldOffset, vform, true);
        }

        @Override
        public FieldInstanceReadOnly withInvokeBehavior() {
            return !hasInvokeExactBehavior()
                ? this
                : new FieldInstanceReadOnly(receiverType, fieldOffset, vform, false);
        }

        @Override
        final MethodType accessModeTypeUncached(AccessType at) {
            return at.accessModeType(receiverType, double.class);
        }

        @Override
        public Optional<VarHandleDesc> describeConstable() {
            var receiverTypeRef = receiverType.describeConstable();
            var fieldTypeRef = double.class.describeConstable();
            if (!receiverTypeRef.isPresent() || !fieldTypeRef.isPresent())
                return Optional.empty();

            // Reflect on this VarHandle to extract the field name
            String name = VarHandles.getFieldFromReceiverAndOffset(
                receiverType, fieldOffset, double.class).getName();
            return Optional.of(VarHandleDesc.ofField(receiverTypeRef.get(), name, fieldTypeRef.get()));
        }

        @ForceInline
        static double get(VarHandle ob, Object holder) {
            FieldInstanceReadOnly handle = (FieldInstanceReadOnly)ob;
            double value = UNSAFE.getDouble(Objects.requireNonNull(handle.receiverType.cast(holder)),
                                 handle.fieldOffset);
            return value;
        }

        @ForceInline
        static double getVolatile(VarHandle ob, Object holder) {
            FieldInstanceReadOnly handle = (FieldInstanceReadOnly)ob;
            double value = UNSAFE.getDoubleVolatile(Objects.requireNonNull(handle.receiverType.cast(holder)),
                                 handle.fieldOffset);
            return value;
        }

        @ForceInline
        static double getOpaque(VarHandle ob, Object holder) {
            FieldInstanceReadOnly handle = (FieldInstanceReadOnly)ob;
            double value = UNSAFE.getDoubleOpaque(Objects.requireNonNull(handle.receiverType.cast(holder)),
                                 handle.fieldOffset);
            return value;
        }

        @ForceInline
        static double getAcquire(VarHandle ob, Object holder) {
            FieldInstanceReadOnly handle = (FieldInstanceReadOnly)ob;
            double value = UNSAFE.getDoubleAcquire(Objects.requireNonNull(handle.receiverType.cast(holder)),
                                 handle.fieldOffset);
            return value;
        }

        static final VarForm FORM = new VarForm(FieldInstanceReadOnly.class, Object.class, double.class);
    }

    static final class FieldInstanceReadWrite extends FieldInstanceReadOnly {
        FieldInstanceReadWrite(Class<?> receiverType, long fieldOffset) {
            this(receiverType, fieldOffset, false);
        }

        private FieldInstanceReadWrite(Class<?> receiverType, long fieldOffset,
                                       boolean exact) {
            super(receiverType, fieldOffset, FieldInstanceReadWrite.FORM, exact);
        }

        @Override
        public FieldInstanceReadWrite withInvokeExactBehavior() {
            return hasInvokeExactBehavior()
                ? this
                : new FieldInstanceReadWrite(receiverType, fieldOffset, true);
        }

        @Override
        public FieldInstanceReadWrite withInvokeBehavior() {
            return !hasInvokeExactBehavior()
                ? this
                : new FieldInstanceReadWrite(receiverType, fieldOffset, false);
        }


        @ForceInline
        static void set(VarHandle ob, Object holder, double value) {
            FieldInstanceReadWrite handle = (FieldInstanceReadWrite)ob;
            UNSAFE.putDouble(Objects.requireNonNull(handle.receiverType.cast(holder)),
                             handle.fieldOffset,
                             value);
        }

        @ForceInline
        static void setVolatile(VarHandle ob, Object holder, double value) {
            FieldInstanceReadWrite handle = (FieldInstanceReadWrite)ob;
            UNSAFE.putDoubleVolatile(Objects.requireNonNull(handle.receiverType.cast(holder)),
                                     handle.fieldOffset,
                                     value);
        }

        @ForceInline
        static void setOpaque(VarHandle ob, Object holder, double value) {
            FieldInstanceReadWrite handle = (FieldInstanceReadWrite)ob;
            UNSAFE.putDoubleOpaque(Objects.requireNonNull(handle.receiverType.cast(holder)),
                                   handle.fieldOffset,
                                   value);
        }

        @ForceInline
        static void setRelease(VarHandle ob, Object holder, double value) {
            FieldInstanceReadWrite handle = (FieldInstanceReadWrite)ob;
            UNSAFE.putDoubleRelease(Objects.requireNonNull(handle.receiverType.cast(holder)),
                                    handle.fieldOffset,
                                    value);
        }

        @ForceInline
        static boolean compareAndSet(VarHandle ob, Object holder, double expected, double value) {
            FieldInstanceReadWrite handle = (FieldInstanceReadWrite)ob;
            return UNSAFE.compareAndSetDouble(Objects.requireNonNull(handle.receiverType.cast(holder)),
                                               handle.fieldOffset,
                                               expected,
                                               value);
        }

        @ForceInline
        static double compareAndExchange(VarHandle ob, Object holder, double expected, double value) {
            FieldInstanceReadWrite handle = (FieldInstanceReadWrite)ob;
            return UNSAFE.compareAndExchangeDouble(Objects.requireNonNull(handle.receiverType.cast(holder)),
                                               handle.fieldOffset,
                                               expected,
                                               value);
        }

        @ForceInline
        static double compareAndExchangeAcquire(VarHandle ob, Object holder, double expected, double value) {
            FieldInstanceReadWrite handle = (FieldInstanceReadWrite)ob;
            return UNSAFE.compareAndExchangeDoubleAcquire(Objects.requireNonNull(handle.receiverType.cast(holder)),
                                               handle.fieldOffset,
                                               expected,
                                               value);
        }

        @ForceInline
        static double compareAndExchangeRelease(VarHandle ob, Object holder, double expected, double value) {
            FieldInstanceReadWrite handle = (FieldInstanceReadWrite)ob;
            return UNSAFE.compareAndExchangeDoubleRelease(Objects.requireNonNull(handle.receiverType.cast(holder)),
                                               handle.fieldOffset,
                                               expected,
                                               value);
        }

        @ForceInline
        static boolean weakCompareAndSetPlain(VarHandle ob, Object holder, double expected, double value) {
            FieldInstanceReadWrite handle = (FieldInstanceReadWrite)ob;
            return UNSAFE.weakCompareAndSetDoublePlain(Objects.requireNonNull(handle.receiverType.cast(holder)),
                                               handle.fieldOffset,
                                               expected,
                                               value);
        }

        @ForceInline
        static boolean weakCompareAndSet(VarHandle ob, Object holder, double expected, double value) {
            FieldInstanceReadWrite handle = (FieldInstanceReadWrite)ob;
            return UNSAFE.weakCompareAndSetDouble(Objects.requireNonNull(handle.receiverType.cast(holder)),
                                               handle.fieldOffset,
                                               expected,
                                               value);
        }

        @ForceInline
        static boolean weakCompareAndSetAcquire(VarHandle ob, Object holder, double expected, double value) {
            FieldInstanceReadWrite handle = (FieldInstanceReadWrite)ob;
            return UNSAFE.weakCompareAndSetDoubleAcquire(Objects.requireNonNull(handle.receiverType.cast(holder)),
                                               handle.fieldOffset,
                                               expected,
                                               value);
        }

        @ForceInline
        static boolean weakCompareAndSetRelease(VarHandle ob, Object holder, double expected, double value) {
            FieldInstanceReadWrite handle = (FieldInstanceReadWrite)ob;
            return UNSAFE.weakCompareAndSetDoubleRelease(Objects.requireNonNull(handle.receiverType.cast(holder)),
                                               handle.fieldOffset,
                                               expected,
                                               value);
        }

        @ForceInline
        static double getAndSet(VarHandle ob, Object holder, double value) {
            FieldInstanceReadWrite handle = (FieldInstanceReadWrite)ob;
            return UNSAFE.getAndSetDouble(Objects.requireNonNull(handle.receiverType.cast(holder)),
                                          handle.fieldOffset,
                                          value);
        }

        @ForceInline
        static double getAndSetAcquire(VarHandle ob, Object holder, double value) {
            FieldInstanceReadWrite handle = (FieldInstanceReadWrite)ob;
            return UNSAFE.getAndSetDoubleAcquire(Objects.requireNonNull(handle.receiverType.cast(holder)),
                                          handle.fieldOffset,
                                          value);
        }

        @ForceInline
        static double getAndSetRelease(VarHandle ob, Object holder, double value) {
            FieldInstanceReadWrite handle = (FieldInstanceReadWrite)ob;
            return UNSAFE.getAndSetDoubleRelease(Objects.requireNonNull(handle.receiverType.cast(holder)),
                                          handle.fieldOffset,
                                          value);
        }

        @ForceInline
        static double getAndAdd(VarHandle ob, Object holder, double value) {
            FieldInstanceReadWrite handle = (FieldInstanceReadWrite)ob;
            return UNSAFE.getAndAddDouble(Objects.requireNonNull(handle.receiverType.cast(holder)),
                                       handle.fieldOffset,
                                       value);
        }

        @ForceInline
        static double getAndAddAcquire(VarHandle ob, Object holder, double value) {
            FieldInstanceReadWrite handle = (FieldInstanceReadWrite)ob;
            return UNSAFE.getAndAddDoubleAcquire(Objects.requireNonNull(handle.receiverType.cast(holder)),
                                       handle.fieldOffset,
                                       value);
        }

        @ForceInline
        static double getAndAddRelease(VarHandle ob, Object holder, double value) {
            FieldInstanceReadWrite handle = (FieldInstanceReadWrite)ob;
            return UNSAFE.getAndAddDoubleRelease(Objects.requireNonNull(handle.receiverType.cast(holder)),
                                       handle.fieldOffset,
                                       value);
        }


        static final VarForm FORM = new VarForm(FieldInstanceReadWrite.class, Object.class, double.class);
    }

    static sealed class FieldStaticReadOnly extends VarHandle {
        final Class<?> declaringClass;
        final Object base;
        final long fieldOffset;

        FieldStaticReadOnly(Class<?> declaringClass, Object base, long fieldOffset) {
            this(declaringClass, base, fieldOffset, FieldStaticReadOnly.FORM, false);
        }

        protected FieldStaticReadOnly(Class<?> declaringClass, Object base, long fieldOffset,
                                      VarForm form, boolean exact) {
            super(form, exact);
            this.declaringClass = declaringClass;
            this.base = base;
            this.fieldOffset = fieldOffset;
        }

        @Override
        public FieldStaticReadOnly withInvokeExactBehavior() {
            return hasInvokeExactBehavior()
                ? this
                : new FieldStaticReadOnly(declaringClass, base, fieldOffset, vform, true);
        }

        @Override
        public FieldStaticReadOnly withInvokeBehavior() {
            return !hasInvokeExactBehavior()
                ? this
                : new FieldStaticReadOnly(declaringClass, base, fieldOffset, vform, false);
        }

        @Override
        public Optional<VarHandleDesc> describeConstable() {
            var fieldTypeRef = double.class.describeConstable();
            if (!fieldTypeRef.isPresent())
                return Optional.empty();

            // Reflect on this VarHandle to extract the field name
            var staticField = VarHandles.getStaticFieldFromBaseAndOffset(
                declaringClass, fieldOffset, double.class);
            var declaringTypeRef = declaringClass.describeConstable();
            if (!declaringTypeRef.isPresent())
                return Optional.empty();
            return Optional.of(VarHandleDesc.ofStaticField(declaringTypeRef.get(), staticField.getName(), fieldTypeRef.get()));
        }

        @Override
        final MethodType accessModeTypeUncached(AccessType at) {
            return at.accessModeType(null, double.class);
        }

        @ForceInline
        static double get(VarHandle ob) {
            FieldStaticReadOnly handle = (FieldStaticReadOnly) ob.target();
            double value = UNSAFE.getDouble(handle.base,
                                 handle.fieldOffset);
            return value;
        }

        @ForceInline
        static double getVolatile(VarHandle ob) {
            FieldStaticReadOnly handle = (FieldStaticReadOnly) ob.target();
            double value = UNSAFE.getDoubleVolatile(handle.base,
                                 handle.fieldOffset);
            return value;
        }

        @ForceInline
        static double getOpaque(VarHandle ob) {
            FieldStaticReadOnly handle = (FieldStaticReadOnly) ob.target();
            double value = UNSAFE.getDoubleOpaque(handle.base,
                                 handle.fieldOffset);
            return value;
        }

        @ForceInline
        static double getAcquire(VarHandle ob) {
            FieldStaticReadOnly handle = (FieldStaticReadOnly) ob.target();
            double value = UNSAFE.getDoubleAcquire(handle.base,
                                 handle.fieldOffset);
            return value;
        }

        static final VarForm FORM = new VarForm(FieldStaticReadOnly.class, null, double.class);
    }

    static final class FieldStaticReadWrite extends FieldStaticReadOnly {

        FieldStaticReadWrite(Class<?> declaringClass, Object base, long fieldOffset) {
            this(declaringClass, base, fieldOffset, false);
        }

        private FieldStaticReadWrite(Class<?> declaringClass, Object base, long fieldOffset,
                                     boolean exact) {
            super(declaringClass, base, fieldOffset, FieldStaticReadWrite.FORM, exact);
        }

        @Override
        public FieldStaticReadWrite withInvokeExactBehavior() {
            return hasInvokeExactBehavior()
                ? this
                : new FieldStaticReadWrite(declaringClass, base, fieldOffset, true);
        }

        @Override
        public FieldStaticReadWrite withInvokeBehavior() {
            return !hasInvokeExactBehavior()
                ? this
                : new FieldStaticReadWrite(declaringClass, base, fieldOffset, false);
        }


        @ForceInline
        static void set(VarHandle ob, double value) {
            FieldStaticReadWrite handle = (FieldStaticReadWrite) ob.target();
            UNSAFE.putDouble(handle.base,
                             handle.fieldOffset,
                             value);
        }

        @ForceInline
        static void setVolatile(VarHandle ob, double value) {
            FieldStaticReadWrite handle = (FieldStaticReadWrite) ob.target();
            UNSAFE.putDoubleVolatile(handle.base,
                                     handle.fieldOffset,
                                     value);
        }

        @ForceInline
        static void setOpaque(VarHandle ob, double value) {
            FieldStaticReadWrite handle = (FieldStaticReadWrite) ob.target();
            UNSAFE.putDoubleOpaque(handle.base,
                                   handle.fieldOffset,
                                   value);
        }

        @ForceInline
        static void setRelease(VarHandle ob, double value) {
            FieldStaticReadWrite handle = (FieldStaticReadWrite) ob.target();
            UNSAFE.putDoubleRelease(handle.base,
                                    handle.fieldOffset,
                                    value);
        }

        @ForceInline
        static boolean compareAndSet(VarHandle ob, double expected, double value) {
            FieldStaticReadWrite handle = (FieldStaticReadWrite) ob.target();
            return UNSAFE.compareAndSetDouble(handle.base,
                                               handle.fieldOffset,
                                               expected,
                                               value);
        }


        @ForceInline
        static double compareAndExchange(VarHandle ob, double expected, double value) {
            FieldStaticReadWrite handle = (FieldStaticReadWrite) ob.target();
            return UNSAFE.compareAndExchangeDouble(handle.base,
                                               handle.fieldOffset,
                                               expected,
                                               value);
        }

        @ForceInline
        static double compareAndExchangeAcquire(VarHandle ob, double expected, double value) {
            FieldStaticReadWrite handle = (FieldStaticReadWrite) ob.target();
            return UNSAFE.compareAndExchangeDoubleAcquire(handle.base,
                                               handle.fieldOffset,
                                               expected,
                                               value);
        }

        @ForceInline
        static double compareAndExchangeRelease(VarHandle ob, double expected, double value) {
            FieldStaticReadWrite handle = (FieldStaticReadWrite) ob.target();
            return UNSAFE.compareAndExchangeDoubleRelease(handle.base,
                                               handle.fieldOffset,
                                               expected,
                                               value);
        }

        @ForceInline
        static boolean weakCompareAndSetPlain(VarHandle ob, double expected, double value) {
            FieldStaticReadWrite handle = (FieldStaticReadWrite) ob.target();
            return UNSAFE.weakCompareAndSetDoublePlain(handle.base,
                                               handle.fieldOffset,
                                               expected,
                                               value);
        }

        @ForceInline
        static boolean weakCompareAndSet(VarHandle ob, double expected, double value) {
            FieldStaticReadWrite handle = (FieldStaticReadWrite) ob.target();
            return UNSAFE.weakCompareAndSetDouble(handle.base,
                                               handle.fieldOffset,
                                               expected,
                                               value);
        }

        @ForceInline
        static boolean weakCompareAndSetAcquire(VarHandle ob, double expected, double value) {
            FieldStaticReadWrite handle = (FieldStaticReadWrite) ob.target();
            return UNSAFE.weakCompareAndSetDoubleAcquire(handle.base,
                                               handle.fieldOffset,
                                               expected,
                                               value);
        }

        @ForceInline
        static boolean weakCompareAndSetRelease(VarHandle ob, double expected, double value) {
            FieldStaticReadWrite handle = (FieldStaticReadWrite) ob.target();
            return UNSAFE.weakCompareAndSetDoubleRelease(handle.base,
                                               handle.fieldOffset,
                                               expected,
                                               value);
        }

        @ForceInline
        static double getAndSet(VarHandle ob, double value) {
            FieldStaticReadWrite handle = (FieldStaticReadWrite) ob.target();
            return UNSAFE.getAndSetDouble(handle.base,
                                          handle.fieldOffset,
                                          value);
        }

        @ForceInline
        static double getAndSetAcquire(VarHandle ob, double value) {
            FieldStaticReadWrite handle = (FieldStaticReadWrite) ob.target();
            return UNSAFE.getAndSetDoubleAcquire(handle.base,
                                          handle.fieldOffset,
                                          value);
        }

        @ForceInline
        static double getAndSetRelease(VarHandle ob, double value) {
            FieldStaticReadWrite handle = (FieldStaticReadWrite) ob.target();
            return UNSAFE.getAndSetDoubleRelease(handle.base,
                                          handle.fieldOffset,
                                          value);
        }

        @ForceInline
        static double getAndAdd(VarHandle ob, double value) {
            FieldStaticReadWrite handle = (FieldStaticReadWrite) ob.target();
            return UNSAFE.getAndAddDouble(handle.base,
                                       handle.fieldOffset,
                                       value);
        }

        @ForceInline
        static double getAndAddAcquire(VarHandle ob, double value) {
            FieldStaticReadWrite handle = (FieldStaticReadWrite) ob.target();
            return UNSAFE.getAndAddDoubleAcquire(handle.base,
                                       handle.fieldOffset,
                                       value);
        }

        @ForceInline
        static double getAndAddRelease(VarHandle ob, double value) {
            FieldStaticReadWrite handle = (FieldStaticReadWrite) ob.target();
            return UNSAFE.getAndAddDoubleRelease(handle.base,
                                       handle.fieldOffset,
                                       value);
        }

        static final VarForm FORM = new VarForm(FieldStaticReadWrite.class, null, double.class);
    }

    static final class Array extends VarHandle {
        static final long BASE = UNSAFE.arrayBaseOffset(double[].class);
        static final int SHIFT = Integer.numberOfTrailingZeros(UNSAFE.arrayIndexScale(double[].class));

        private Array(boolean exact) {
            super(Array.FORM, exact);
        }

        @Override
        public Array withInvokeExactBehavior() {
            return hasInvokeExactBehavior()
                ? this
                : EXACT_INSTANCE;
        }

        @Override
        public Array withInvokeBehavior() {
            return !hasInvokeExactBehavior()
                ? this
                : NON_EXACT_INSTANCE;
        }

        @Override
        public Optional<VarHandleDesc> describeConstable() {
            return Optional.of(VarHandleDesc.ofArray(double[].class.describeConstable().orElseThrow()));
        }

        @Override
        final MethodType accessModeTypeUncached(AccessType at) {
            return at.accessModeType(double[].class, double.class, int.class);
        }

        @ForceInline
        static double get(VarHandle ob, Object oarray, int index) {
            double[] array = (double[]) oarray;
            return array[index];
        }

        @ForceInline
        static void set(VarHandle ob, Object oarray, int index, double value) {
            double[] array = (double[]) oarray;
            array[index] = value;
        }

        @ForceInline
        static double getVolatile(VarHandle ob, Object oarray, int index) {
            double[] array = (double[]) oarray;
            return UNSAFE.getDoubleVolatile(array,
                    (((long) Preconditions.checkIndex(index, array.length, Preconditions.AIOOBE_FORMATTER)) << SHIFT) + BASE);
        }

        @ForceInline
        static void setVolatile(VarHandle ob, Object oarray, int index, double value) {
            double[] array = (double[]) oarray;
            UNSAFE.putDoubleVolatile(array,
                    (((long) Preconditions.checkIndex(index, array.length, Preconditions.AIOOBE_FORMATTER)) << SHIFT) + BASE,
                    value);
        }

        @ForceInline
        static double getOpaque(VarHandle ob, Object oarray, int index) {
            double[] array = (double[]) oarray;
            return UNSAFE.getDoubleOpaque(array,
                    (((long) Preconditions.checkIndex(index, array.length, Preconditions.AIOOBE_FORMATTER)) << SHIFT) + BASE);
        }

        @ForceInline
        static void setOpaque(VarHandle ob, Object oarray, int index, double value) {
            double[] array = (double[]) oarray;
            UNSAFE.putDoubleOpaque(array,
                    (((long) Preconditions.checkIndex(index, array.length, Preconditions.AIOOBE_FORMATTER)) << SHIFT) + BASE,
                    value);
        }

        @ForceInline
        static double getAcquire(VarHandle ob, Object oarray, int index) {
            double[] array = (double[]) oarray;
            return UNSAFE.getDoubleAcquire(array,
                    (((long) Preconditions.checkIndex(index, array.length, Preconditions.AIOOBE_FORMATTER)) << SHIFT) + BASE);
        }

        @ForceInline
        static void setRelease(VarHandle ob, Object oarray, int index, double value) {
            double[] array = (double[]) oarray;
            UNSAFE.putDoubleRelease(array,
                    (((long) Preconditions.checkIndex(index, array.length, Preconditions.AIOOBE_FORMATTER)) << SHIFT) + BASE,
                    value);
        }

        @ForceInline
        static boolean compareAndSet(VarHandle ob, Object oarray, int index, double expected, double value) {
            double[] array = (double[]) oarray;
            return UNSAFE.compareAndSetDouble(array,
                    (((long) Preconditions.checkIndex(index, array.length, Preconditions.AIOOBE_FORMATTER)) << SHIFT) + BASE,
                    expected,
                    value);
        }

        @ForceInline
        static double compareAndExchange(VarHandle ob, Object oarray, int index, double expected, double value) {
            double[] array = (double[]) oarray;
            return UNSAFE.compareAndExchangeDouble(array,
                    (((long) Preconditions.checkIndex(index, array.length, Preconditions.AIOOBE_FORMATTER)) << SHIFT) + BASE,
                    expected,
                    value);
        }

        @ForceInline
        static double compareAndExchangeAcquire(VarHandle ob, Object oarray, int index, double expected, double value) {
            double[] array = (double[]) oarray;
            return UNSAFE.compareAndExchangeDoubleAcquire(array,
                    (((long) Preconditions.checkIndex(index, array.length, Preconditions.AIOOBE_FORMATTER)) << SHIFT) + BASE,
                    expected,
                    value);
        }

        @ForceInline
        static double compareAndExchangeRelease(VarHandle ob, Object oarray, int index, double expected, double value) {
            double[] array = (double[]) oarray;
            return UNSAFE.compareAndExchangeDoubleRelease(array,
                    (((long) Preconditions.checkIndex(index, array.length, Preconditions.AIOOBE_FORMATTER)) << SHIFT) + BASE,
                    expected,
                    value);
        }

        @ForceInline
        static boolean weakCompareAndSetPlain(VarHandle ob, Object oarray, int index, double expected, double value) {
            double[] array = (double[]) oarray;
            return UNSAFE.weakCompareAndSetDoublePlain(array,
                    (((long) Preconditions.checkIndex(index, array.length, Preconditions.AIOOBE_FORMATTER)) << SHIFT) + BASE,
                    expected,
                    value);
        }

        @ForceInline
        static boolean weakCompareAndSet(VarHandle ob, Object oarray, int index, double expected, double value) {
            double[] array = (double[]) oarray;
            return UNSAFE.weakCompareAndSetDouble(array,
                    (((long) Preconditions.checkIndex(index, array.length, Preconditions.AIOOBE_FORMATTER)) << SHIFT) + BASE,
                    expected,
                    value);
        }

        @ForceInline
        static boolean weakCompareAndSetAcquire(VarHandle ob, Object oarray, int index, double expected, double value) {
            double[] array = (double[]) oarray;
            return UNSAFE.weakCompareAndSetDoubleAcquire(array,
                    (((long) Preconditions.checkIndex(index, array.length, Preconditions.AIOOBE_FORMATTER)) << SHIFT) + BASE,
                    expected,
                    value);
        }

        @ForceInline
        static boolean weakCompareAndSetRelease(VarHandle ob, Object oarray, int index, double expected, double value) {
            double[] array = (double[]) oarray;
            return UNSAFE.weakCompareAndSetDoubleRelease(array,
                    (((long) Preconditions.checkIndex(index, array.length, Preconditions.AIOOBE_FORMATTER)) << SHIFT) + BASE,
                    expected,
                    value);
        }

        @ForceInline
        static double getAndSet(VarHandle ob, Object oarray, int index, double value) {
            double[] array = (double[]) oarray;
            return UNSAFE.getAndSetDouble(array,
                    (((long) Preconditions.checkIndex(index, array.length, Preconditions.AIOOBE_FORMATTER)) << SHIFT) + BASE,
                    value);
        }

        @ForceInline
        static double getAndSetAcquire(VarHandle ob, Object oarray, int index, double value) {
            double[] array = (double[]) oarray;
            return UNSAFE.getAndSetDoubleAcquire(array,
                    (((long) Preconditions.checkIndex(index, array.length, Preconditions.AIOOBE_FORMATTER)) << SHIFT) + BASE,
                    value);
        }

        @ForceInline
        static double getAndSetRelease(VarHandle ob, Object oarray, int index, double value) {
            double[] array = (double[]) oarray;
            return UNSAFE.getAndSetDoubleRelease(array,
                    (((long) Preconditions.checkIndex(index, array.length, Preconditions.AIOOBE_FORMATTER)) << SHIFT) + BASE,
                    value);
        }

        @ForceInline
        static double getAndAdd(VarHandle ob, Object oarray, int index, double value) {
            double[] array = (double[]) oarray;
            return UNSAFE.getAndAddDouble(array,
                    (((long) Preconditions.checkIndex(index, array.length, Preconditions.AIOOBE_FORMATTER)) << SHIFT) + BASE,
                    value);
        }

        @ForceInline
        static double getAndAddAcquire(VarHandle ob, Object oarray, int index, double value) {
            double[] array = (double[]) oarray;
            return UNSAFE.getAndAddDoubleAcquire(array,
                    (((long) Preconditions.checkIndex(index, array.length, Preconditions.AIOOBE_FORMATTER)) << SHIFT) + BASE,
                    value);
        }

        @ForceInline
        static double getAndAddRelease(VarHandle ob, Object oarray, int index, double value) {
            double[] array = (double[]) oarray;
            return UNSAFE.getAndAddDoubleRelease(array,
                    (((long) Preconditions.checkIndex(index, array.length, Preconditions.AIOOBE_FORMATTER)) << SHIFT) + BASE,
                    value);
        }

        static final VarForm FORM = new VarForm(Array.class, double[].class, double.class, int.class);
        static final Array NON_EXACT_INSTANCE = new Array(false);
        static final Array EXACT_INSTANCE = new Array(true);
    }
}
