/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

/*
 *  $Id: CreateDomainCommand.java,v 1.12 2006/02/06 06:11:14 km105526 Exp $
 */

package com.sun.enterprise.cli.commands;

import com.sun.appserv.management.client.prefs.LoginInfo;
import com.sun.appserv.management.client.prefs.LoginInfoStore;
import com.sun.appserv.management.client.prefs.LoginInfoStoreFactory;
import com.sun.enterprise.cli.framework.CommandValidationException;
import com.sun.enterprise.cli.framework.CommandException;
import com.sun.enterprise.cli.framework.CLILogger;

import com.sun.enterprise.util.net.NetUtils;
import com.sun.enterprise.util.io.FileUtils;

import com.sun.enterprise.admin.servermgmt.DomainConfig;
import com.sun.enterprise.admin.servermgmt.DomainsManager;

// jdk imports
import java.io.File;

import java.util.Properties;
import java.util.StringTokenizer;



/**
 *  This is a local command that creates a new domain
 *  @version  $Revision: 1.12 $
 */
public class CreateDomainCommand extends BaseLifeCycleCommand
{
    // constant variables for create-domain options
    private static final String DOMAIN_PATH    = "path";    
    private static final String INSTANCE_PORT  = "instanceport";    
    private static final String DOCROOT        = "docroot";
    private static final String TEMPLATE       = "template";
    private static final String DOMAIN_PROPERTIES     = "domainproperties";
    private static final String CHECKPORTS_OPTION     = "checkports";
    private static final int DEFAULT_HTTPSSL_PORT           = 8181;
    private static final int DEFAULT_IIOPSSL_PORT           = 3820;
    private static final int DEFAULT_IIOPMUTUALAUTH_PORT    = 3920;
    private static final int DEFAULT_INSTANCE_PORT          = 8080;
    private static final int DEFAULT_JMS_PORT               = 7676;
    private static final String DEFAULT_JMS_USER            = "admin";
    private static final String DEFAULT_JMS_PASSWORD        = "admin";
    private static final int DEFAULT_IIOP_PORT              = 3700;
    private static final int DEFAULT_JMX_PORT              = 8686;                     

    public static String DOMAINDIR_OPTION = "domaindir";
    private static String SAVELOGIN_OPTION = "savelogin";

    private String domainName = null;
    private String adminUser = null;
    private String adminPassword = null;
    private String masterPassword = null;

    /** Creates new CreateDomainCommand */
    public CreateDomainCommand()
    {
    }

    /**
     *  An abstract method that validates the options 
     *  on the specification in the xml properties file
     *  @return true if successfull
     */
    public boolean validateOptions() throws CommandValidationException
    {
        return super.validateOptions();
    }

    
    /* validates adminpassword and masterpassword */
    public void validatePasswords() throws CommandValidationException
    {
        //verify adminpassword is greater than 8 characters	
        if (!isPasswordValid(adminPassword)) {
            throw new CommandValidationException(getLocalizedString("PasswordLimit",
                new Object[]{ADMIN_PASSWORD}));
        }
        
        //verify masterpassword is greater than 8 characters
        if (!isPasswordValid(masterPassword)) {
            throw new CommandValidationException(getLocalizedString("PasswordLimit",
                new Object[]{MASTER_PASSWORD}));
        }

        //verify that domainName is valid
        CLILogger.getInstance().printDebugMessage("domainName = " + domainName);
    }


    /**
     *  this methods returns the admin password and is used to get the admin password
     *  for create-domain. The password can be passed in on the command line 
     *  (--adminpassword), environment variable (AS_ADMIN_ADMINPASSWORD), 
     *  or in the password file (--passwordfile).
     *  first it checks if adminpassword option is specified on command line.
     *  if not, then it'll try to get AS_ADMIN_ADMINPASSWORD from the password file.
     *  if that still does not exist then get AS_ADMIN_PASSWORD from 
     *  ./asadminprefs file.
     *  if all else fails, then prompt the user for the password if interactive=true.
     *  @return admin password
     *  @throws CommandValidationException if could not get adminpassword option 
     */
    protected String getAdminPassword()     
        throws CommandValidationException, CommandException
    {
        //getPassword(optionName, allowedOnCommandLine, readPrefsFile, readPasswordOptionFromPrefs, readMasterPasswordFile, mgr, config,
        //promptUser, confirm, validate)
        return getPassword(ADMIN_PASSWORD, "AdminPasswordPrompt", "AdminPasswordConfirmationPrompt",
                            true, true, false, false, null, null,
                            true, true, true, true);
    }

    
    /**
     *  An abstract method that executes the command
     *  @throws CommandException
     */
    public void runCommand() 
        throws CommandException, CommandValidationException
    {
        validateOptions();
        setLoggerLevel();

        //verify for no-spaces in domaindir option on non-windows platform
        String domainDirValue = getOption(DOMAINDIR_OPTION);
        if ((domainDirValue != null) && !isWindows() && isSpaceInPath(domainDirValue))
            throw new CommandException(getLocalizedString("SpaceNotAllowedInPath", 
                                                            new Object[]{DOMAINDIR_OPTION}));
        //domain validation upfront (i.e. before we prompt)
        try {
            domainName = (String)operands.firstElement();      
            DomainsManager manager = getFeatureFactory().getDomainsManager();
            DomainConfig config = getDomainConfig(domainName); 
            manager.validateDomain(config, false);
        } catch (Exception e) {
            CLILogger.getInstance().printDetailMessage(e.getLocalizedMessage());
            throw new CommandException(getLocalizedString("CouldNotCreateDomain", 
                new Object[] {domainName}), e);
        }
        
        adminUser = getAdminUser();

        adminPassword = getAdminPassword();
        masterPassword = getMasterPassword(true);
        validatePasswords();
    
        try
        {
            //verify admin port is valid
            verifyPortIsValid(getOption(ADMIN_PORT));
            //instance option is entered then verify instance port is valid
            if (getOption(INSTANCE_PORT) != null)
                verifyPortIsValid(getOption(INSTANCE_PORT));
        
            //get domain properties from options or from option
            Properties domainProperties = getDomainProperties(getOption(DOMAIN_PROPERTIES));
            //we give priority to the --adminport option
            domainProperties.remove(DomainConfig.K_ADMIN_PORT);
            //saving the login information happens inside this method
            createTheDomain(getDomainsRoot(), domainProperties);
        }
        catch (Exception e)
        {
            CLILogger.getInstance().printDetailMessage(e.getLocalizedMessage());
            throw new CommandException(getLocalizedString("CouldNotCreateDomain", 
                new Object[] {domainName}), e);
        }
    }


    /**
       Verify that the port is valid.
       Port must be greater than 0 and less than 65535.
       This method will also check if the port is in used.
       @param portNum - the port number to verify
       @throws CommandException if Port is not valid
       @throws CommandValidationException is port number is not a numeric value.
       
     */      
    private void verifyPortIsValid(String portNum)
        throws CommandException, CommandValidationException
    {

        final int portToVerify = convertPortStr(portNum);

        if (portToVerify <= 0 || portToVerify > 65535)
        {
            throw new CommandException(getLocalizedString("InvalidPortRange", 
                                         new Object[] {portNum}));
        }
        if (getBooleanOption(CHECKPORTS_OPTION) && !NetUtils.isPortFree(portToVerify))
        {
            throw new CommandException(getLocalizedString("PortInUse", 
                                         new Object[] {(String) operands.firstElement(),
                                                       portNum}));
        }
        CLILogger.getInstance().printDebugMessage("Port =" + portToVerify);
    }
    

    /** 
     *  create the domain
     *  @param domainPath - domain path to insert in domainConfig
     *  @param domainProperties - properties to insert in domainConfig
     *  @throws CommandException if domain cannot be created
     */
    private void createTheDomain(final String domainPath, 
                                 Properties domainProperties)
        throws Exception
    {	
        Integer adminPort = new Integer(getOption(ADMIN_PORT));
            

	    //
	    // fix for bug# 4930684
	    // domain name is validated before the ports
	    //

	    String domainFilePath = (domainPath + File.separator + domainName);
	    if (FileUtils.safeGetCanonicalFile(new File(domainFilePath)).exists()) {
            	throw new CommandValidationException(
                	getLocalizedString("DomainExists", new Object[] {domainName}));
	    }
    
        final Integer instancePort = getPort(domainProperties, 
                                             DomainConfig.K_INSTANCE_PORT,
                                             getOption(INSTANCE_PORT),
                                             Integer.toString(DEFAULT_INSTANCE_PORT),
                                             "HTTP Instance");
	    
        final Integer jmsPort = getPort(domainProperties, 
                                        DomainConfig.K_JMS_PORT, null, 
                                        Integer.toString(DEFAULT_JMS_PORT), "JMS");
        
/*  Bug# 4859518
    The authentication of jms broker will fail if the jms user is not in the 
    user database referred by jms broker. Setting the jms user and password to 
    "admin" which is always present in the user db. Once the mechanism to update
    the user db with given jms user/password is in place we should set the
    user specified values for jms user and password.
    final String jmsUser = adminUser;
    final String jmsPassword = adminPassword;
*/
        final String jmsUser = DEFAULT_JMS_USER;
        final String jmsPassword =  DEFAULT_JMS_PASSWORD;
	    
        final Integer orbPort = getPort(domainProperties, 
                                        DomainConfig.K_ORB_LISTENER_PORT,
                                        null, Integer.toString(DEFAULT_IIOP_PORT), 
                                        "IIOP");
	    	    
        final Integer httpSSLPort = getPort(domainProperties, 
                                            DomainConfig.K_HTTP_SSL_PORT, null, 
                                            Integer.toString(DEFAULT_HTTPSSL_PORT), 
                                            "HTTP_SSL");

        final Integer iiopSSLPort = getPort(domainProperties, 
                                            DomainConfig.K_IIOP_SSL_PORT, null, 
                                            Integer.toString(DEFAULT_IIOPSSL_PORT), 
                                            "IIOP_SSL");

        final Integer iiopMutualAuthPort = getPort(domainProperties, 
                                                   DomainConfig.K_IIOP_MUTUALAUTH_PORT, null, 
                                                   Integer.toString(DEFAULT_IIOPMUTUALAUTH_PORT), 
                                                   "IIOP_MUTUALAUTH");

        //FIXTHIS: Currently the jmx admin port is used only for SE/EE where the DAS as
        //in addition to the http admin listener a jsr160 connector listener. This jmx connector
        //will be exposed to access the mbean API of the DAS. If we choose to expose the API
        //in PE, then this jsr160 listener will be added to the PE domain.xml and this will be 
        //needed. If we choose not to, then this should be moved into SE/EE specific code
        //(namely DomainsManager.createDomain).
        final Integer jmxPort = getPort(domainProperties, 
                                        DomainConfig.K_JMX_PORT, null, 
                                        Integer.toString(DEFAULT_JMX_PORT), 
                                        "JMX_ADMIN");
                
        Boolean saveMasterPassword = getSaveMasterPassword(masterPassword);        
        
        //System.out.println("adminPassword=" + adminPassword + " masterPassword=" + masterPassword + 
        //     " saveMasterPassword=" + saveMasterPassword);               
        
        DomainConfig domainConfig = new DomainConfig(domainName,
                                                     adminPort, domainPath, adminUser, 
                                                     adminPassword, 
                                                     masterPassword, 
                                                     saveMasterPassword, instancePort, jmsUser,
                                                     jmsPassword, jmsPort, orbPort, 
                                                     httpSSLPort, iiopSSLPort,
                                                     iiopMutualAuthPort, jmxPort, 
                                                     domainProperties);        
        if (getOption(TEMPLATE) != null) {
            domainConfig.put(DomainConfig.K_TEMPLATE_NAME, getOption(TEMPLATE));
        }

        domainConfig.put(DomainConfig.K_VALIDATE_PORTS, new Boolean(getBooleanOption(CHECKPORTS_OPTION)));     

        DomainsManager manager = getFeatureFactory().getDomainsManager();
        
        manager.createDomain(domainConfig);
        CLILogger.getInstance().printDetailMessage(getLocalizedString("DomainCreated",
                                                                      new Object[] {domainName}));
        checkAsadminPrefsFile();
        if (getBooleanOption(SAVELOGIN_OPTION))
            saveLogin(adminPort, adminUser, adminPassword, domainName);
    }


        /**
         *  This routine checks if the .asadminprefs file exists in the home
         *  directory.  If it exists, then a warning message is displayed.
         *  This message is to warn user that the remote commands will
         *  retrieve user and password from .asadminpref file if it is not
         *  entered on the command line, environment variable or passwordfile
         **/
    private void checkAsadminPrefsFile()
    {
        try {
            checkForFileExistence(System.getProperty("user.home"), ASADMINPREFS);
            CLILogger.getInstance().printMessage(getLocalizedString("AsadminPrefFileDetected", new Object[] {domainName}));
        }
        catch(CommandException ce) {
                //do nothing since we do not want to display a warning msg
                //if the file does not exist.
        }
    }



    /**
     *  get port from from the properties option or default or free port
     *  @param properties - properties from command line
     *  @param key - key for the type of port
     *  @param portStr -
     *  @param defaultPort - default port to use 
     *  @name  - of port
     *  @throws CommandValidationException if error in retrieve port
     */
    private Integer getPort(Properties  properties,
                            String      key,
                            String      portStr,
                            String      defaultPort,
                            String      name)
        throws CommandValidationException
    {
        int port = 0;
        boolean portNotSpecified = false;
        boolean invalidPortSpecified = false;
        boolean defaultPortUsed = false;
        if ((portStr != null) && !portStr.equals("")) 
        {
            port = convertPortStr(portStr);
            if ((port <= 0) || (port > 65535))
                invalidPortSpecified = true;
        }
        else if (properties != null) 
        {
            String property = properties.getProperty(key);
            if ((property != null) && !property.equals("")) {
                port = convertPortStr(property);
            }
            else {
                portNotSpecified = true;
            }
        }
        else {
            portNotSpecified = true;
        }
        if (portNotSpecified) {
            port = convertPortStr(defaultPort);
            defaultPortUsed = true;
        }
        if(getBooleanOption(CHECKPORTS_OPTION) && !NetUtils.isPortFree(port)) 
	{
            port = NetUtils.getFreePort();
                //don't understand why this is a printMessage not an Exception??
                //maybe there will always be a port ??
            if (portNotSpecified) {
                if (defaultPortUsed) {
                    CLILogger.getInstance().printDetailMessage(
                        getLocalizedString("DefaultPortInUse", 
                                           new Object[] {name, defaultPort, Integer.toString(port)}));
                }
                else {
                    CLILogger.getInstance().printDetailMessage(
                        getLocalizedString("PortNotSpecified", 
                                           new Object[] {name, new Integer(port)}));
                }
            }
            else if (invalidPortSpecified){
                CLILogger.getInstance().printDetailMessage(
                    getLocalizedString("InvalidPortRangeMsg", 
                                       new Object[] {name, new Integer(port)}));
            }
            else {
                CLILogger.getInstance().printDetailMessage(
                    getLocalizedString("PortInUseMsg", 
                                       new Object[] {name, new Integer(port)}));
            }
        }
        else if (defaultPortUsed) {
            CLILogger.getInstance().printDetailMessage(
                getLocalizedString("UsingDefaultPort", 
                                   new Object[] {name, Integer.toString(port)}));
        }
        
        if (properties != null) {
            properties.remove(key);
        }
        return new Integer(port);
    }


    /** 
     *  Get domain properties from options
     *  @param propertyValues from options
     *  @return domain properties 
     *  @throws CommandException if cannot get properties
     **/
    private Properties getDomainProperties(final String propertyValues)
        throws CommandException, CommandValidationException
    {
        Properties propertyList = new Properties();

        if (propertyValues == null) return propertyList;
        StringTokenizer st = new StringTokenizer(propertyValues, DELIMITER);
        while (st.hasMoreTokens())
        {
            String propertyString = st.nextToken();
            while (st.hasMoreTokens() && 
                   propertyString.endsWith(Character.toString(ESCAPE_CHAR)))
            {
                propertyString = propertyString.concat(st.nextToken());
            }
            final int index = propertyString.indexOf(Character.toString(EQUAL_SIGN));
            if (index == -1)
                throw new CommandValidationException(getLocalizedString("InvalidPropertySyntax"));
            final String propertyName = propertyString.substring(0, index);
            final String propertyValue = propertyString.substring(index+1);
            propertyList.put(propertyName, propertyValue);
        }
        CLILogger.getInstance().printDebugMessage("domain properties = " + propertyList);
        return propertyList;
    }

    /**
     * Converts the port string to port int
     * @param port - the port number
     * @return 
     * @throws CommandValidationExeption if port string is not numeric
     */
    private int convertPortStr(final String port) 
        throws CommandValidationException
    {
        try
        {
            return Integer.parseInt(port);
        }
        catch(Exception e)
        {
            throw new CommandValidationException(
                getLocalizedString("InvalidPortNumber", 
                                   new Object[] {port}));
        }
    }

    /** Saves the login information to the login store. Usually this is the file
     * ".asadminpass" in user's home directory.
    */
    private void saveLogin(final int port, final String user, final String password, final String dn)
    {
        final CLILogger logger = CLILogger.getInstance();
        try {
            //by definition, the host name will default to "localhost" and entry is overwritten
            final LoginInfoStore store = LoginInfoStoreFactory.getStore(null);
            final LoginInfo login      = new LoginInfo("localhost", port, user, password);
            if (store.exists(login.getHost(), login.getPort())) {
                //just let the user know that the user has chosen to overwrite the login information. This is non-interactive, on purpose
                final Object[] params = new Object[] {login.getHost(), ""+login.getPort()};
                final String msg = getLocalizedString("OverwriteLoginMsgCreateDomain", params);
                logger.printMessage(msg);
            }
            store.store(login, true);
            final Object[] params = new String[] {user, dn, store.getName()};
            final String msg = getLocalizedString("LoginInfoStoredCreateDomain", params);
            logger.printMessage(msg);
        }
        catch(final Exception e) {
            final Object[] params = new String[] {user, dn};
            final String msg = getLocalizedString("LoginInfoStoredCreateDomain", params);
            logger.printWarning(msg);
            if (logger.isDebug())
                logger.printExceptionStackTrace(e);
        }
    }
}
