/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */
 
/*
 * $Header: /cvs/glassfish/admin-core/mbeanapi/src/java/com/sun/appserv/management/util/jmx/ObjectNameQueryImpl.java,v 1.4 2006/03/09 20:30:32 llc Exp $
 * $Revision: 1.4 $
 * $Date: 2006/03/09 20:30:32 $
 */
package com.sun.appserv.management.util.jmx;

import java.util.Set;
import java.util.HashSet;
import java.util.regex.*;
import java.util.Hashtable;
import java.util.Enumeration;
import java.util.Iterator;


import javax.management.MBeanServerConnection;
import javax.management.ObjectName;
import javax.management.MalformedObjectNameException;

import com.sun.appserv.management.util.misc.EnumerationIterator;

public class ObjectNameQueryImpl implements ObjectNameQuery
{
		public
	ObjectNameQueryImpl()
	{
	}
	
	
	/**
		Return true if one (or more) of the properties match the regular expressions
		for both name and value.   Return false if no property/value combinations match.
		
		A null pattern matches anything.
	 */
		boolean
	match( Hashtable properties, Pattern propertyPattern, Pattern valuePattern )
	{
		final Iterator	keys	= new EnumerationIterator( properties.keys() );
		boolean	matches	= false;
		
		while ( keys.hasNext() )
		{
			final String	key	= (String)keys.next();
			
			if ( propertyPattern == null || propertyPattern.matcher( key ).matches() )
			{
				if ( valuePattern == null )
				{
					matches	= true;
					break;
				}

				// see if value matches
				final String	value	= (String)properties.get( key );
				
				if ( valuePattern.matcher( value ).matches() )
				{
					matches	= true;
					break;
				}
			}
		}
		
		return( matches );
	}
	
	/**
		Match all property/value expressions against the ObjectName.
		
		Return true if for each property/value regular expression pair there is at least one
		property within the ObjectName whose property name and value match their respective
		patterns.
		
		A null regex indicates "match anything".
	 */
		boolean
	matchAll(	ObjectName	name,
				Pattern []	propertyPatterns,
				Pattern []	valuePatterns )
	{
		boolean	matches	= true;
		
		final Hashtable	properties	= name.getKeyPropertyList();
		
		for( int i = 0; i < propertyPatterns.length; ++i )
		{
			if ( ! match( properties, propertyPatterns[ i ], valuePatterns[ i ] ) )
			{
				matches	= false;
				break;
			}
		}
		
		return( matches );
	}
	
	
	/**
		Match all property/value expressions against the ObjectName.
		
		Return true if there is at least one property/value regular expression pair that
		matches a property/value pair within the ObjectName.
		
		A null regex indicates "match anything".
	 */
		boolean
	matchAny(	ObjectName	name,
				Pattern []	propertyPatterns,
				Pattern []	valuePatterns )
	{
		boolean	matches	= false;
		
		final Hashtable	properties	= name.getKeyPropertyList();
		
		for( int i = 0; i < propertyPatterns.length; ++i )
		{
			if ( match( properties, propertyPatterns[ i ], valuePatterns[ i ] ) )
			{
				matches	= true;
				break;
			}
		}
		
		return( matches );
	}
	
	
	
		Pattern []
	createPatterns( final String [] patternStrings, int numItems )
	{
		final Pattern []	patterns	= new Pattern [ numItems ];
		
		if ( patternStrings == null )
		{
			for( int i = 0; i < numItems; ++i )
			{
				patterns[ i ]	= null;
			}
			
			return( patterns );
		}
			
		
		for( int i = 0; i < numItems; ++i )
		{
			// consider null to match anything
			
			if ( patternStrings[ i ] == null )
			{
				patterns[ i ]	= null;
			}
			else
			{
				patterns[ i ]	= Pattern.compile( patternStrings[ i ] );
			}
		}
		
		return( patterns );
	}
	
	private interface Matcher
	{
		boolean		match( ObjectName name, Pattern [] names, Pattern [] values );
	}
	
	private class MatchAnyMatcher implements Matcher
	{
		public MatchAnyMatcher() {}
		
			public boolean	
		match( ObjectName name, Pattern [] names, Pattern [] values )
		{
			return( matchAny( name, names, values ) );
		}
	}
	
	private class MatchAllMatcher implements Matcher
	{
		public MatchAllMatcher() {}
		
			public boolean	
		match( ObjectName name, Pattern [] names, Pattern [] values )
		{
			return( matchAll( name, names, values ) );
		}
	}
	

		Set<ObjectName>
	matchEither(
	    Matcher matcher,
	    Set<ObjectName> startingSet,
	    String [] regexNames,
	    String [] regexValues )
	{
		if ( regexNames == null && regexValues == null )
		{
			// both null => matches entire original set
			return( startingSet );
		}
		
		final Set<ObjectName> results	= new HashSet<ObjectName>();
		
		int	numMatches	= 0;
		if ( regexNames != null )
		{
			numMatches	= regexNames.length;
		}
		else
		{
			numMatches	= regexValues.length;
		}

		final Pattern [] namePatterns	= createPatterns( regexNames, numMatches );
		final Pattern [] valuePatterns	= createPatterns( regexValues, numMatches );
		
		for( final ObjectName name : startingSet )
		{
			if ( matcher.match( name, namePatterns, valuePatterns ) )
			{
				results.add( name );
			}
		}

		return( results );
	}

		public Set<ObjectName>
	matchAll( Set<ObjectName> startingSet, String [] regexNames, String [] regexValues )
	{
		return( matchEither( new MatchAllMatcher(), startingSet, regexNames, regexValues ) );
	}
	

				
		public Set<ObjectName>
	matchAny( Set<ObjectName> startingSet, String [] regexNames, String [] regexValues )
	{
		return( matchEither( new MatchAnyMatcher(), startingSet, regexNames, regexValues ) );
	}
}






