/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the "License").  You may not use this file except 
 * in compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * glassfish/bootstrap/legal/CDDLv1.0.txt or 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html. 
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * HEADER in each file and include the License file at 
 * glassfish/bootstrap/legal/CDDLv1.0.txt.  If applicable, 
 * add the following below this CDDL HEADER, with the 
 * fields enclosed by brackets "[]" replaced with your 
 * own identifying information: Portions Copyright [yyyy] 
 * [name of copyright owner]
 */

package com.sun.enterprise.tools.admingui.util;

import com.sun.enterprise.tools.guiframework.view.ViewDescriptorManager;
import com.sun.enterprise.tools.admingui.AdminGUIConstants;
import com.sun.enterprise.tools.guiframework.view.ViewXMLEntityResolver;

import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.net.URL;
import javax.servlet.ServletConfig;


public class PreloadXML extends Thread {

    /**
     *  Constructor.
     *
     *  @param	name	    The Thread name.
     *  @param	xmlFileName The name of the XML file to load.
     *  @param	config	    The ServletConfig.
     */
    public PreloadXML(String name, String xmlFileName, ServletConfig config) {
	super(name);
	setXMLFileName(xmlFileName);
	_config = config;
    }

    /**
     *	<p> This constructor attempt to figure out the XML file that should be
     *	    used.  It will be "xml/eeViewDescriptor.xml" or if not found:
     *	    "xml/viewDescriptor.xml".  The ThreadName will be "PreloadXML".</p>
     *
     *	@param	config	    The ServletConfig.
     */
    public PreloadXML(ServletConfig config) {
	// Init what we know...
	this("PreloadXML", null, config);

	// Calculate xmlFile name
	String xmlFile = "xml/eeViewDescriptor.xml";
	URL xmlURL = null;
	try {
	    xmlURL = getClass().getClassLoader().getResource(xmlFile);
	    if (xmlURL == null) {
		// Next try file system...
		xmlURL = new URL("file:///" + _config.getServletContext().getRealPath(xmlFile));
		xmlURL.openConnection().connect(); // Exception if not found
	    }
	} catch (Exception ex) {
	    // Fall back...
	    xmlFile = "xml/viewDescriptor.xml";
	}

	// We should now have the correct xmlFile...
	setXMLFileName(xmlFile);
    }

    /**
     *	<p> This method returns true if the ViewXML file appears to be loaded
     *	    already.</p>
     */
    public static boolean isAlreadyLoaded() {
	ViewDescriptorManager vdm = ViewDescriptorManager.getInstance();
	return (vdm.getViewDescriptorURL() != null);
    }

    /**
     *	<p> This method populates the VDM by deserializing its state.</p>
     */
    public void deserialize() {
	ViewDescriptorManager vdm = ViewDescriptorManager.getInstance();
	try {
	    vdm.deserialize(
		new ObjectInputStream(getClass().getClassLoader().
		    getResourceAsStream("viewXML.ser")));
	} catch (Exception ex) {
	    // This is an optimization, ignore if it fails
	    ex.printStackTrace();
	    return;
	}
    }

    /**
     *  <p> This method pre-loads the ViewXML structure.</p>
     */
    public void run() {
	init();
	deserialize();
    }

    /**
     *	Initialize stuff.
     */
    protected void init() {
	// The following is to pre-load the ViewXML descriptor stuff...
	URL xmlURL = null;
	try {
	    xmlURL = getClass().getClassLoader().getResource(_xmlFileName);
	} catch (Exception ex) {
	    // Ignore...
	}
	try {
	    ViewDescriptorManager vdm = ViewDescriptorManager.getInstance();
	    if (PreloadXML.isAlreadyLoaded()) {
		// Although this is not synchronizing, this thread is invoked
		// from the login page... and the LOGIN page must be accessed
		// before a page w/i the application.  So the chances of a
		// Threading issue are next to none. (Also, the login page
		// calls isAlreadyLoaded(), so it won't execute this Thread
		// more than once.)
		return;
	    }
	    if (xmlURL == null) {
		xmlURL = new URL("file:///" + _config.getServletContext().getRealPath(_xmlFileName));
	    }
	    vdm.setViewDescriptorURL(xmlURL);
	    vdm.setDTDURLBase("/xml/");  // Different servlet, hard code
	    vdm.setJSPRoot(AdminGUIConstants.DEFAULT_DISPLAY_URL_DIR);
	    vdm.setViewXMLEntityResolver(new ViewXMLEntityResolver());
	} catch (Exception ex) {
	    ex.printStackTrace();
	}
    }

    /**
     *	<p> This displays usage information for using this class from the
     *	    Commandline to generate a Serialization file.</p>
     */
    public static void printUsage() {
	System.out.println("You must specify the xml file name:\n");
	System.out.println("Usage:\n");
	System.out.println("java " + PreloadXML.class.getName()
		+ " <xml file name>\n");
    }

    /**
     *	<p> This <code>main</code> method allows this class to be used as a
     *	    utility for generating a Serialized version of the ViewDescriptor
     *	    tree.  The XML/DTD files must be in the ClassPath. The XML file
     *	    name must be supplied via the command line.</p>
     */
    public static void main(String args[]) {
	if (args.length < 1) {
	    printUsage();
	    return;
	}

	// Initialize stuff
	PreloadXML pre = new PreloadXML("na", args[0], (ServletConfig) null);
	pre.init();

	// Get the VDM and ensure it is populated
	ViewDescriptorManager vdm = ViewDescriptorManager.getInstance();
	vdm.getViewDescriptor("TopFrameset"); 

	// Serialize the VDM state
	try {
	    vdm.serialize(
		new ObjectOutputStream(new FileOutputStream("viewXML.ser")));
	} catch (Exception ex) {
	    ex.printStackTrace();
	    return;
	}

	// Let the user know that it worked!
	System.out.println("Success!");
    }

    public String getXMLFileName() {
	return _xmlFileName;
    }

    public void setXMLFileName(String name) {
	_xmlFileName = name;
    }

    private String	    _xmlFileName    = null;
    private ServletConfig   _config	    = null;
}
