/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.ejb.base.stats;

import java.lang.reflect.Method;
import java.util.HashMap;

import com.sun.ejb.spi.stats.EJBMethodStatsManager;

import com.sun.enterprise.admin.monitor.registry.MonitoringRegistry;
import com.sun.enterprise.admin.monitor.registry.MonitoringLevel;
import com.sun.enterprise.admin.monitor.registry.MonitoringRegistrationException;

import java.util.logging.*;
import com.sun.enterprise.log.Log;
import com.sun.logging.*;

/**
 * A class that manages all the monitored EJB methods
 *
 * @author Mahesh Kannan
 *
 */

public final class EJBMethodStatsManagerImpl
    implements EJBMethodStatsManager
{
    private static Logger _logger =
        LogDomains.getLogger(LogDomains.EJB_LOGGER);

    private MonitoringRegistry		registry;

    private String			appName;
    private String			modName;
    private String			ejbName;

    private Method[]			methods;
    private boolean			methodMonitorOn;

    private MethodMonitor[]		methodMonitors;
    private HashMap			methodMonitorMap;
    private Object			lock = new Object();
    private Object[]			logParams = null;

    EJBMethodStatsManagerImpl(MonitoringRegistry registry, 
	    String ejbName, String modName, String appName)
    {
	this.registry = registry;

	this.ejbName = ejbName;
	this.modName = modName;
	this.appName = appName;

	logParams = new Object[] {ejbName, modName, appName};
    }

    public final boolean isMethodMonitorOn() {
	return methodMonitorOn;
    }

    public final void preInvoke(Method method) {
	if (methodMonitorOn) {
	    MethodMonitor monitor = null;
	    synchronized (lock) {
		if (methodMonitorOn) {
		    monitor = (MethodMonitor) methodMonitorMap.get(method);
		}
	    }
	    if (monitor != null) {
		monitor.preInvoke();
	    }
	}
    }

    public final void postInvoke(Method method, Throwable th) {
	if (methodMonitorOn) {
	    MethodMonitor monitor = null;
	    synchronized (lock) {
		if (methodMonitorOn) {
		    monitor = (MethodMonitor) methodMonitorMap.get(method);
		}
	    }
	    if (monitor != null) {
		monitor.postInvoke(th);
	    }
	}
    }

    public MethodMonitor[]  getMethodMonitors() {
	return this.methodMonitors;
    }

    public void undeploy() {
	synchronized (lock) {
	    methodMonitorOn = false;
	}
	deregisterStats();

	methods = null;
	methodMonitors = null;
	methodMonitorMap = null;
	registry = null;
    }

    void registerMethods(Method[] methods) {
	this.methods = methods;
    }
    
    void setMethodMonitorOn(boolean monitorOn) {
	if (methods == null) {
	    _logger.log(Level.WARNING, "base.stats.method.nomethods", logParams);
	    return;
	}
	int size = methods.length;
	if (monitorOn == true) {
	    this.methodMonitors = new MethodMonitor[size];
	    HashMap map = new HashMap();
	    for (int i=0; i<size; i++) {
		methodMonitors[i] = new MethodMonitor(methods[i]);
		map.put(methods[i], methodMonitors[i]);

		EJBMethodStatsImpl impl =
		    new EJBMethodStatsImpl(methodMonitors[i]);
		try {
		    if (_logger.isLoggable(Level.FINE)) {
			_logger.log(Level.FINE, "Registering method: "
			    + methodMonitors[i].getMethodName()
			    + "; for " + appName + "; " + modName
			    + "; " + ejbName);
		    }
		    registry.registerEJBMethodStats(impl, 
			methodMonitors[i].getMethodName(),
			ejbName, modName, appName, null);
		    if (_logger.isLoggable(Level.FINE)) {
			_logger.log(Level.FINE, "Registered method: "
			    + methodMonitors[i].getMethodName()
			    + "; for " + appName + "; " + modName
			    + "; " + ejbName);
		    }
		} catch (MonitoringRegistrationException monRegEx) {
		    Object[] params = new Object[] {ejbName, modName,
			    appName, methodMonitors[i].getMethodName()};
		    _logger.log(Level.WARNING,
			    "base.stats.method.register.monreg.error", params);
		    _logger.log(Level.FINE, "", monRegEx);
		} catch (Exception ex) {
		    Object[] params = new Object[] {ejbName, modName,
			    appName, methodMonitors[i].getMethodName()};
		    _logger.log(Level.WARNING,
			    "base.stats.method.register.error", params);
		    _logger.log(Level.FINE, "", ex);
		}
	    }
	    this.methodMonitorMap = map;
	    synchronized (lock) {
		this.methodMonitorOn = true;
	    }
	} else {
	    synchronized (lock) {
		this.methodMonitorOn = false;
	    }
	    deregisterStats();

	    this.methodMonitorMap = null;
	    this.methodMonitors = null;

	}
    }

    void appendStats(StringBuffer sbuf) {
	if (methodMonitors != null) {
	    int size = methods.length;
	    for (int i=0; i<size; i++) {
		MethodMonitor monitor =
		    (MethodMonitor) methodMonitors[i];
		monitor.appendStats(sbuf);
	    }
	}
    }

    private void deregisterStats() {
	if (methodMonitors == null) {
	    return;
	}
	int size = methodMonitors.length;
	for (int i=0; i<size; i++) {
	    try {
		registry.unregisterEJBMethodStats(
		    methodMonitors[i].getMethodName(),
		     ejbName, modName, appName);
		if (_logger.isLoggable(Level.FINE)) {
		    _logger.log(Level.FINE, "Unregistered method: "
			    + methodMonitors[i].getMethodName()
			    + "; for " + appName + "; " + modName
			    + "; " + ejbName);
		}
	    } catch (MonitoringRegistrationException monRegEx) {
		Object[] params = new Object[] {ejbName, modName,
		    appName, methodMonitors[i].getMethodName()};
		_logger.log(Level.FINE,
			"base.stats.method.unregister.monreg.error", params);
		_logger.log(Level.FINE, "", monRegEx);
	    } catch (Exception ex) {
		Object[] params = new Object[] {ejbName, modName,
		    appName, methodMonitors[i].getMethodName()};
		_logger.log(Level.WARNING,
			"base.stats.method.unregister.error", params);
		_logger.log(Level.FINE, "", ex);
	    }
	}

	methodMonitors = null;
    }

}
