/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */
package com.sun.ejb.portable;

import java.io.*;
import javax.ejb.*;
import javax.rmi.PortableRemoteObject;

/**
 * A portable, Serializable implementation of EJBMetaData.
 * This class can potentially be instantiated in another vendor's container
 * so it must not refer to any non-portable RI-specific classes.
 *
 */

public final class EJBMetaDataImpl implements javax.ejb.EJBMetaData, Serializable
{
    private Class keyClass;
    private Class homeClass;
    private Class remoteClass;
    private boolean isSessionBean;
    private boolean isStatelessSessionBean;
    private HomeHandle homeHandle;

    // Dont serialize the EJBHome ref directly, use the HomeHandle
    transient private EJBHome ejbHomeStub;


    // this constructor is only called in the RI's EJB container
    public EJBMetaDataImpl(EJBHome ejbHomeStub, Class homeClass, 
		   Class remoteClass, Class keyClass, 
		   boolean isSessionBean, boolean isStatelessSessionBean)
    {
	this.ejbHomeStub = ejbHomeStub;
	this.homeHandle = new HomeHandleImpl(ejbHomeStub);
	this.keyClass  = keyClass;
	this.homeClass  = homeClass;
	this.remoteClass  = remoteClass;
	this.isSessionBean = isSessionBean;
	this.isStatelessSessionBean = isStatelessSessionBean;
    }


    /**
     * 
     */
    public Class getHomeInterfaceClass() 
    {
	return homeClass;
    }

    /**
     * 
     */
    public Class getRemoteInterfaceClass() 
    {
	return remoteClass;
    }

    /**
     *
     */
    public EJBHome getEJBHome() 
    {
	return ejbHomeStub;
    }

    /**
     *
     */
    public Class getPrimaryKeyClass() 
    {
	if ( keyClass == null ) {
	    // for SessionBeans there is no primary key
	    throw new RuntimeException("SessionBeans do not have a primary key");
	}
	return keyClass;
    }

    /**
     * 
     */
    public boolean isSession() 
    {
	return isSessionBean;
    }


    public boolean isStatelessSession()
    {
	return isStatelessSessionBean;
    }


    private void readObject(ObjectInputStream in)
	throws IOException, ClassNotFoundException
    {
	isSessionBean = in.readBoolean();
	isStatelessSessionBean = in.readBoolean();

	// Use thread context classloader to load home/remote/primarykey classes
	// See EJB2.0 spec section 18.4.4
	ClassLoader loader = Thread.currentThread().getContextClassLoader();
	remoteClass = loader.loadClass(in.readUTF());
	homeClass = loader.loadClass(in.readUTF());
	if ( !isSessionBean )
	    keyClass = loader.loadClass(in.readUTF());

	homeHandle = (HomeHandle)in.readObject();
	ejbHomeStub = homeHandle.getEJBHome();
	// narrow the home so that the application doesnt have to do
	// a narrow after EJBMetaData.getEJBHome().
	ejbHomeStub = (EJBHome)PortableRemoteObject.narrow(ejbHomeStub, homeClass);
    }

    private void writeObject(ObjectOutputStream out)
	throws IOException
    {
	out.writeBoolean(isSessionBean);
	out.writeBoolean(isStatelessSessionBean);

	// Write the String names of the Class objects, 
	// since Class objects cant be serialized unless the classes
	// they represent are Serializable.
	out.writeUTF(remoteClass.getName());
	out.writeUTF(homeClass.getName());
	if ( !isSessionBean )
	    out.writeUTF(keyClass.getName());

	out.writeObject(homeHandle);
    }   
}
