/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.enterprise.connectors.util;

import com.sun.enterprise.connectors.*;
import com.sun.enterprise.deployment.runtime.connector.*;
import com.sun.enterprise.deployment.*;
import com.sun.enterprise.connectors.util.SecurityMapUtils;
import com.sun.logging.*;
import java.util.logging.Logger;
import java.util.logging.Level;
import java.util.Set;
import com.sun.enterprise.connectors.authentication.RuntimeSecurityMap;
import com.sun.enterprise.connectors.authentication.ConnectorSecurityMap;
/**
 */ 

public class ConnectionPoolReconfigHelper {

    private static Logger _logger = LogDomains.getLogger( LogDomains.RSR_LOGGER );
    

    public enum ReconfigAction { RECREATE_POOL, UPDATE_MCF_AND_ATTRIBUTES, NO_OP };

    public static ReconfigAction compare( ConnectorConnectionPool oldPool, 
            ConnectorConnectionPool newPool, Set excludedProps ) 
	    throws ConnectorRuntimeException 
    {

        if ( isEqualConnectorConnectionPool( oldPool, newPool , excludedProps ) 
	    == ReconfigAction.NO_OP ) {
	
	    return ReconfigAction.UPDATE_MCF_AND_ATTRIBUTES;
	}
        
	return ReconfigAction.RECREATE_POOL;

    }

    /*
     * Compare the Original ConnectorConnectionPool with the passed one
     * If MCF properties are changed, indicate that pool recreation is
     * required 
     * We only check the MCF properties since a pool restart is required
     * for changes in MCF props. For pool specific properties we can get
     * away without restart
     * If the new pool and old pool have identical MCF properties returns 
     * true
     */
    private static ReconfigAction isEqualConnectorConnectionPool( ConnectorConnectionPool
            oldCcp, ConnectorConnectionPool newCcp, Set excludedProps )
    {
        boolean poolsEqual = true;
        
	//for all the following properties, we need to recreate pool if they
	//have changed
	if (newCcp.getTransactionSupport() != oldCcp.getTransactionSupport() ) {
	    return ReconfigAction.RECREATE_POOL;
	}

	if (newCcp.isAssociateWithThread() != oldCcp.isAssociateWithThread() ) {
	    return ReconfigAction.RECREATE_POOL;
	}

	if (newCcp.isLazyConnectionAssoc() != oldCcp.isLazyConnectionAssoc() ) {
	    return ReconfigAction.RECREATE_POOL;
	}

	
        ConnectorDescriptorInfo oldCdi = oldCcp.getConnectorDescriptorInfo();
        ConnectorDescriptorInfo newCdi = newCcp.getConnectorDescriptorInfo();

	if (! oldCdi.getResourceAdapterClassName().equals(
	        newCdi.getResourceAdapterClassName())  ) {
	    
	    logFine(
	        "isEqualConnectorConnectionPool: getResourceAdapterClassName:: " +
	         oldCdi.getResourceAdapterClassName() + " -- " +
		 newCdi.getResourceAdapterClassName()); 
	    return ReconfigAction.RECREATE_POOL;
	}

	if (! oldCdi.getConnectionDefinitionName().equals(
	        newCdi.getConnectionDefinitionName()) ) {
	    
	    logFine(
	        "isEqualConnectorConnectionPool: getConnectionDefinitionName:: "+
	        oldCdi.getConnectionDefinitionName() + " -- " +
		newCdi.getConnectionDefinitionName());
	   	
	    return ReconfigAction.RECREATE_POOL;
	}

    ConnectorSecurityMap[] newSecurityMaps = newCcp.getSecurityMaps();
    RuntimeSecurityMap newRuntimeSecurityMap = 
        SecurityMapUtils.processSecurityMaps(newSecurityMaps);
    ConnectorSecurityMap[] oldSecurityMaps = oldCcp.getSecurityMaps();
    RuntimeSecurityMap oldRuntimeSecurityMap = 
        SecurityMapUtils.processSecurityMaps(oldSecurityMaps);
    if (!( oldRuntimeSecurityMap.equals(newRuntimeSecurityMap) )){
    	logFine("isEqualConnectorConnectionPool: CCP.getSecurityMaps:: " +
    			"New set of Security Maps is not equal to the existing" +
    			" set of security Maps.");
    	return ReconfigAction.RECREATE_POOL;
    }
	
	Set newMCFConfigProps = newCdi.getMCFConfigProperties();
	Set oldMCFConfigProps = oldCdi.getMCFConfigProperties();
        return  oldCdi.compareMCFConfigProperties( newCdi, excludedProps );
    }
    
    private static void logFine( String msg ) {
        if (msg != null && _logger.isLoggable(Level.FINE)) {
	    _logger.fine( msg );
	}
    }
}
