/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.enterprise.deployment.backend;

import java.io.IOException;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.List;
import java.util.Properties;
import javax.enterprise.deploy.shared.ModuleType;

import com.sun.enterprise.deployment.Application;
import com.sun.enterprise.deployment.archivist.ApplicationArchivist;
import com.sun.enterprise.deployment.archivist.Archivist;
import com.sun.enterprise.deployment.archivist.ArchivistFactory;
import com.sun.enterprise.deployment.deploy.shared.AbstractArchive;
import com.sun.enterprise.deployment.interfaces.ClientJarMaker;
import com.sun.enterprise.deployment.RootDeploymentDescriptor;
import com.sun.enterprise.deployment.util.ModuleDescriptor;
import com.sun.enterprise.util.zip.ZipItem;

/**
 * This class is responsible for creating an appclient jar file that
 * will be used by the appclient container to run the appclients for
 * the deployed application.
 *
 * @author Jerome Dochez
 */
class ApplicationClientJarMaker implements ClientJarMaker {

    protected Properties props;

    /**
     * Default constructor for this stateless object
     * @param props are the implementation properties (if any)
     */
    public ApplicationClientJarMaker(Properties props) {
        this.props = props;
    }
    
    /**
     * creates the appclient container jar file
     * @param descriptor is the loaded module's deployment descriptor
     * @param source is the abstract archive for the source module deployed
     * @param target is the abstract archive for the desired appclient container jar file
     * @param stubs are the stubs generated by the deployment codegen
     * @param props is a properties collection to pass implementation parameters
     *
     * @throws IOException when the jar file creation fail
     */
    public void create(RootDeploymentDescriptor descriptor, AbstractArchive source,
        AbstractArchive target,ZipItem[] stubs, Properties props)
        throws IOException {
        create(descriptor, source, null, target, stubs, props);
    }

    /**
     * creates the appclient container jar file
     * @param descriptor is the loaded module's deployment descriptor
     * @param source is the abstract archive for the source module deployed
     * @param source is the abstract archive for the generated xml directory
     * @param target is the abstract archive for the desired appclient container jar file
     * @param stubs are the stubs generated by the deployment codegen
     * @param props is a properties collection to pass implementation parameters
     *
     * @throws IOException when the jar file creation fail
     */
    public void create(RootDeploymentDescriptor descriptor, AbstractArchive source,
        AbstractArchive source2, AbstractArchive target,ZipItem[] stubs, 
        Properties props) throws IOException {

        // in all cases we copy the stubs file in the target archive
        ClientJarMakerUtils.populateStubs(target, stubs);

        //@@@ this block will not be called if we use ModuleClientMaker
        if (!descriptor.isApplication()) {
            //copy over all content of the appclient
            ClientJarMakerUtils.populateModuleJar(source, source2, target);
            return;
        }
        
        Application app = Application.class.cast(descriptor);
        for (Iterator modules = app.getModules(); modules.hasNext();) {
            ModuleDescriptor md = ModuleDescriptor.class.cast(modules.next());

            //ignore the war and rar modules, include both appclient and ejb
            if (md.getModuleType().equals(ModuleType.WAR) 
                || md.getModuleType().equals(ModuleType.RAR)) {
                continue;
            }

            AbstractArchive subSource = source.getEmbeddedArchive(md.getArchiveUri());
            AbstractArchive subSource2 = null;
            if (source2 != null) {
                subSource2 = source2.getEmbeddedArchive(md.getArchiveUri());
            }
            AbstractArchive subTarget = target.getEmbeddedArchive(md.getArchiveUri());

            //copy over all content of the appclient
            ClientJarMakerUtils.populateModuleJar(subSource, subSource2, subTarget);
                
            target.closeEntry(subTarget);
            source.closeEntry(subSource);
            if (source2 != null) {
                source2.closeEntry(subSource2);
            }

            //copy over the alternative deployment descriptors
            if (md.getAlternateDescriptor() != null) {
                String ddPath = md.getAlternateDescriptor();
                String runtimeDDPath = "sun-" + ddPath;
                if (source2 != null) {
                    ClientJarMakerUtils.copy(source2, target, ddPath);
                    ClientJarMakerUtils.copy(source2, target, runtimeDDPath);
                } else {
                    ClientJarMakerUtils.copy(source, target, ddPath);
                    ClientJarMakerUtils.copy(source, target, runtimeDDPath);
                }
            }
        }

        // populate the library jars
        List<String> libraries = 
            ClientJarMakerUtils.getLibraryEntries(app, source);
        for (String library : libraries) {
            ClientJarMakerUtils.copy(source, target, library);
        }

        // deployment descriptors for the application
        ClientJarMakerUtils.copyDeploymentDescriptors(
            new ApplicationArchivist(), source, source2, target);
    }
}
