/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.enterprise.deployment.phasing;

import com.sun.enterprise.admin.common.exception.DeploymentException;
import com.sun.enterprise.admin.common.exception.ServerInstanceException;
import com.sun.enterprise.admin.common.MBeanServerFactory;
import com.sun.enterprise.admin.common.ObjectNames;
import com.sun.enterprise.admin.event.AdminEvent;
import com.sun.enterprise.admin.event.AdminEventCache;
import com.sun.enterprise.admin.event.AdminEventListenerException;
import com.sun.enterprise.admin.event.AdminEventMulticaster;
import com.sun.enterprise.admin.event.AdminEventResult;
import com.sun.enterprise.admin.event.EventBuilder;
import com.sun.enterprise.admin.event.EventContext;
import com.sun.enterprise.admin.event.EventStack;
import com.sun.enterprise.admin.event.ModuleDeployEvent;
import com.sun.enterprise.admin.server.core.AdminService;
import com.sun.enterprise.admin.server.core.DeploymentNotificationHelper;
import com.sun.enterprise.admin.target.TargetType;
import com.sun.enterprise.admin.util.HostAndPort;
import com.sun.enterprise.config.ConfigBean;
import com.sun.enterprise.config.ConfigBeansFactory;
import com.sun.enterprise.config.ConfigContext;
import com.sun.enterprise.config.ConfigException;
import com.sun.enterprise.config.serverbeans.ApplicationHelper;
import com.sun.enterprise.config.serverbeans.ClusterHelper;
import com.sun.enterprise.config.serverbeans.Config;
import com.sun.enterprise.config.serverbeans.HttpService;
import com.sun.enterprise.config.serverbeans.PropertyResolver;
import com.sun.enterprise.config.serverbeans.ServerHelper;
import com.sun.enterprise.config.serverbeans.ServerTags;
import com.sun.enterprise.config.serverbeans.ServerXPathHelper;
import com.sun.enterprise.config.serverbeans.VirtualServer;
import com.sun.enterprise.deployment.Application;
import com.sun.enterprise.deployment.archivist.Archivist;
import com.sun.enterprise.deployment.archivist.ArchivistFactory;
import com.sun.enterprise.deployment.backend.AppDD;
import com.sun.enterprise.deployment.backend.ClientJarMakerRegistry;
import com.sun.enterprise.deployment.backend.DeployableObjectType;
import com.sun.enterprise.deployment.backend.DeploymentLogger;
import com.sun.enterprise.deployment.backend.DeploymentRequest;
import com.sun.enterprise.deployment.backend.DeploymentRequestRegistry;
import com.sun.enterprise.deployment.backend.IASDeploymentException;
import com.sun.enterprise.deployment.interfaces.DeploymentImplConstants;
import com.sun.enterprise.deployment.pluggable.DeploymentFactory;
import com.sun.enterprise.deployment.util.DeploymentProperties;
import com.sun.enterprise.deployment.util.ModuleDescriptor;
import com.sun.enterprise.deployment.WebBundleDescriptor;
import com.sun.enterprise.instance.AppsManager;
import com.sun.enterprise.instance.WebModulesManager;
import com.sun.enterprise.instance.EjbModulesManager;
import com.sun.enterprise.instance.ConnectorModulesManager;
import com.sun.enterprise.instance.AppclientModulesManager;
import com.sun.enterprise.instance.BaseManager;
import com.sun.enterprise.instance.InstanceEnvironment;
import com.sun.enterprise.server.ApplicationServer;
import com.sun.enterprise.server.Constants;
import com.sun.enterprise.util.i18n.StringManager;
import com.sun.enterprise.util.io.FileUtils;
import com.sun.enterprise.util.RelativePathResolver;
import com.sun.enterprise.util.StringUtils;
import com.sun.enterprise.util.SystemPropertyConstants;
import com.sun.enterprise.util.FileUtil;
import com.sun.enterprise.resource.Resource;
import com.sun.enterprise.resource.ResourcesXMLParser;

import java.io.File;
import java.io.FilenameFilter;
import java.util.Enumeration;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.Properties;
import java.util.ArrayList;
import java.util.Map;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;
import java.util.StringTokenizer;
import java.util.Set;
import java.util.HashSet;
import java.util.HashMap;
import java.net.URL;
import java.net.URLClassLoader;
import java.net.InetAddress;
import java.net.UnknownHostException;
import javax.enterprise.deploy.shared.ModuleType;
import javax.management.Attribute;
import javax.management.AttributeList;
import javax.management.InstanceNotFoundException;
import javax.management.MBeanException;
import javax.management.MBeanServer;
import javax.management.ObjectName;

public class DeploymentServiceUtils {

    /** Deployment Logger object for this class */
    public static final Logger sLogger = DeploymentLogger.get();

    /** string manager */
    private static StringManager localStrings =
        StringManager.getManager( DeploymentServiceUtils.class );

    private static final Object[] emptyParams = new Object[]{};
    private static final String[] emptySignature = new String[]{};

    /**
     * @return the path for the client jar file of a deployed application
     */
    public static String getClientJarPath(String moduleID) {
        
        // let's ensure first that our client jar is ready.
        ClientJarMakerRegistry registry = ClientJarMakerRegistry.getInstance();
        
        if (registry.isRegistered(moduleID)) {
            
            // let's wait until it is finished.
            registry.waitForCompletion(moduleID);
        }
        
        return moduleID + DeploymentImplConstants.ClientJarSuffix;
        
    }

    /**
     * This method is used by StartPhase, StopPhases to multicast start/stop
     * events to the listeners. To multicast to a particular server the managedServerInstance
     * representing that particular server is used
     * @param eventType APPLICATION_DEPLOYED/APPLICATION_UNDEPLOYED/APPLICATION_REDEPLOYED
     *                  MODULE_DEPLOYED/MODULE_REDEPLOYED/MODULE_UNDEPLOYED
     * @param entityName app/module name
     * @param targetName server to which event has to be sent
     */
    static boolean multicastEvent(int eventType, String entityName, 
				  String targetName) throws IASDeploymentException {
        return multicastEvent(eventType, entityName, null, targetName);
    }

    static boolean multicastEvent(int eventType, String entityName, 
				  String moduleType, String targetName) 
	throws IASDeploymentException {
        return multicastEvent(eventType,entityName,moduleType, false, targetName);
    }
    
    /**
     * This method is used by StartPhase, StopPhases to multicast start/stop
     * events to the listeners. To multicast to a particular server the managedServerInstance
     * representing that particular server is used
     * @param eventType APPLICATION_DEPLOYED/APPLICATION_UNDEPLOYED/APPLICATION_REDEPLOYED
     *                  MODULE_DEPLOYED/MODULE_REDEPLOYED/MODULE_UNDEPLOYED
     * @param entityName app/module name
     * @param moduleType ejb/web/connector
     * @param targetName server to which event has to be sent
     */
    static boolean multicastEvent(int eventType, String entityName, 
				  String moduleType, boolean cascade, String targetName) throws IASDeploymentException {
        return multicastEvent(eventType, entityName, moduleType, cascade, false, targetName) ;
    }

    static boolean multicastEvent(int eventType, String entityName,
                                  String moduleType, boolean cascade, boolean forceDeploy, String targetName) throws IASDeploymentException {
        return multicastEvent(eventType, entityName, moduleType, cascade, forceDeploy, Constants.LOAD_UNSET, targetName) ;
    }
    
    public static boolean multicastEvent(int eventType, String entityName, 
				  String moduleType, boolean cascade, boolean forceDeploy, int loadUnloadAction, String targetName) 
	throws IASDeploymentException {

        // Flush config changes before sending the event. The events API
        // requires that config be saved prior to sending event.
        try {
            ConfigContext config =  getConfigContext();
            if (config != null && config.isChanged()) {
                config.flush();
            } else {
                // Server is initializing. Can not flush config changes.
            }
        } catch (ConfigException ce) {
            throw new IASDeploymentException(ce);
        }
	try {
	    return DeploymentNotificationHelper.multicastEvent(eventType, entityName,
                                  moduleType, cascade, forceDeploy, loadUnloadAction, targetName);
	} catch (Throwable t) {
        IASDeploymentException iasEx = new IASDeploymentException(t.getMessage());
        iasEx.initCause(t);
        throw iasEx;
        }
        // unreachable
    }

    /**
     * Get module type string using the DeployableObjectType. This string is used
     * while multicasting events to remote servers
     * @param moduleType deployableObjectType of the module
     * @return String 
     * DeployableObjectType.APP = null
     * DeployableObjectType.EJB = "ejb"
     * DeployableObjectType.WEB = "web"
     * DeployableObjectType.CONNECTOR = "connector"
     */
    public static String getModuleTypeString(DeployableObjectType moduleType)
    {
        String moduleTypeString = null;
        if (moduleType.equals(DeployableObjectType.EJB)) 
        {
            moduleTypeString = ModuleDeployEvent.TYPE_EJBMODULE;
        }
        else if(moduleType.equals(DeployableObjectType.WEB))
        {
            moduleTypeString = ModuleDeployEvent.TYPE_WEBMODULE;
        }
        else if(moduleType.equals(DeployableObjectType.CONN))
        {
            moduleTypeString = ModuleDeployEvent.TYPE_CONNECTOR;
        } 
        else if (moduleType.equals(DeployableObjectType.CAR)) 
        {
            moduleTypeString = ModuleDeployEvent.TYPE_APPCLIENT;
        }
        return moduleTypeString;
    }

    public static BaseManager getInstanceManager(
        DeployableObjectType moduleType) {
        try {
            InstanceEnvironment insEnv = 
                new InstanceEnvironment(getInstanceName());
            if (moduleType.equals(DeployableObjectType.APP)) {
                return new AppsManager(insEnv);
            } else if (moduleType.equals(DeployableObjectType.EJB)) {
                return new EjbModulesManager(insEnv);
            } else if(moduleType.equals(DeployableObjectType.WEB)) {
                return new WebModulesManager(insEnv);
            } else if(moduleType.equals(DeployableObjectType.CONN)) {
                return new ConnectorModulesManager(insEnv);
            } else if (moduleType.equals(DeployableObjectType.CAR)) {
                return new AppclientModulesManager(insEnv);
            } else {
                // invalid module type
                return null;
            }
        } catch (Exception e) {
            return null;
        }
    }


    /** This method is for j2ee application with embedded web modules.
     *  First it tries to get application from AppsManager of DAS,
     *  if it's not already loaded there, then we load from disk.
     *  Since this method might read from disk, it could be time-consuming.
     *  The context roots in sun-application.xml will override the ones
     *  defined in application.xml.
     *  The method returns String[0] when there is no web module in the
     *  application.
     *                            
     *  @param name of the j2ee app registered in the config
     *  @return the context roots of the embedded web modules in j2ee 
     *          application
     */ 
    public static String[] getContextRootsForEmbeddedWebApp(String appName)
        throws IASDeploymentException {
        try {
            AppsManager appsManager =
                new AppsManager(new InstanceEnvironment(getInstanceName()));
            Application application =
                appsManager.getRegisteredDescriptor(appName);
            // if the application already loaded on DAS
            if (application != null) {
                ArrayList contextRoots = new ArrayList();
                for (Iterator itr = application.getWebBundleDescriptors().iterator(); itr.hasNext();) {
                    WebBundleDescriptor wbd = (WebBundleDescriptor) itr.next();
                    contextRoots.add(wbd.getContextRoot());
                }
                return (String[])contextRoots.toArray(new String[contextRoots.size()]);
            // if not, read from disk
            } else {
                // load from generated/xml dir first
                // print a warning if generated/xml dir is not there
                // and load from original dir (upgrade scenario)
                String xmlDir = appsManager.getGeneratedXMLLocation(appName);
                if (!FileUtils.safeIsDirectory(xmlDir)) {
                    String appDir = appsManager.getLocation(appName);
                    // log a warning message in the server log
                    sLogger.log(Level.WARNING, "core.no_xmldir",
                        new Object[]{xmlDir, appDir});
                    xmlDir = appDir;
                }
                AppDD appDD = new AppDD(new File(xmlDir));
                return appDD.getContextRoots();
            }
        } catch (Exception  e) {
            if (e instanceof IASDeploymentException) {
                throw (IASDeploymentException)e;
            } else {
                throw new IASDeploymentException(e);
            }
        }
    }

    public static List<String> getTargetNamesFromTargetString (
        String targetString) {
        if (targetString == null) {
            return null;
        }
        StringTokenizer st = new StringTokenizer(targetString);
        List<String> targetNames = new ArrayList<String>();
        while (st.hasMoreTokens()) {
            targetNames.add(st.nextToken()); 
        }
        return targetNames;
    }

    public static void setResourceOptionsInRequest (DeploymentRequest req, 
        DeploymentProperties props) {
        req.setResourceAction(props.getResourceAction());
        req.setResourceTargetList(props.getResourceTargetList());
    }

    public static FilenameFilter getFilenameFilter(final String fileName){
        FilenameFilter filter = new FilenameFilter() {
            public boolean accept(File dir, String name) {
                if ( (new File(dir, name)).isDirectory()) {
                    return true;
                } else if (name !=null && name.equals(fileName)){
                    return true;
                } else {
                    return false;
                }
            }
        };
        return filter;
    }

    public static List<Resource> getResources(String appId, 
        DeployableObjectType moduleType) {
        try {
            // retrieve all the sun-resources.xml under the 
            // generated/xml directory
            String directoryToLook = 
                getInstanceManager(moduleType).getGeneratedXMLLocation(appId); 
            Set listOfFiles = FileUtil.getAllFilesUnder(
                new File(directoryToLook), 
                getFilenameFilter(SUN_RESOURCE_XML), true);

            // get the expected paths
            ArrayList<String> expectedXMLPaths = new ArrayList<String>();
            Application app = getInstanceManager(
                moduleType).getRegisteredDescriptor(appId);

            if (app != null) {
                if (app.isVirtual()) {
                    // for standalone module, it's under the root META-INF
                    expectedXMLPaths.add(SUN_RESOURCE_XML_PATH);
                } else {
                    // for application
                    // first add the one under app root
                    expectedXMLPaths.add(SUN_RESOURCE_XML_PATH);
                    // then add the ones under sub module root
                    for (Iterator itr = app.getModules();itr.hasNext();) {
                        ModuleDescriptor aModule =
                            (ModuleDescriptor) itr.next();
                        String moduleUri = FileUtils.makeFriendlyFileName(
                            aModule.getArchiveUri());
                        String subModulePath = moduleUri +
                           File.separator + SUN_RESOURCE_XML_PATH;
                        expectedXMLPaths.add(subModulePath);
                    }
                }
            } else {
                // the default
                expectedXMLPaths.add(SUN_RESOURCE_XML_PATH);
            }
                                                                                
            // process all the sun-resources.xml and add them
            // to a hashmap for further processing
            HashMap<String, List<Resource>> resourcesListMap  = 
                new HashMap<String, List<Resource>>();
               
            for (Iterator fIter = listOfFiles.iterator(); fIter.hasNext(); ) {
                File file = (File)fIter.next();
                String filePath = file.getPath();
                String xmlFilePath =
                    (new File(directoryToLook, filePath)).getAbsolutePath();
                if (expectedXMLPaths.contains(filePath)) {
                    ResourcesXMLParser allResources =
                        new ResourcesXMLParser(xmlFilePath);
                    resourcesListMap.put(filePath, 
                        allResources.getResourcesList());
                } else {
                    sLogger.log(Level.WARNING,
                        "enterprise.deployment.ignore.sun.resources.xml",
                         xmlFilePath);
                }
            }

            // this will be called until the duplicate/conflict
            // logic is ready
            // return processResourcesListMap(resourcesListMap);
       
            return simpleProcessResourcesListMap(resourcesListMap);

        } catch (Exception e) {
            sLogger.log(Level.WARNING, 
                "enterprise.deployment.error.get.resources", 
                e.getMessage());
            return new ArrayList<Resource>();
        }
    }

    // this method just adds all the resources from all the lists
    // doesn't do anything with conflicts or duplicates
    private static List<Resource> simpleProcessResourcesListMap(
        HashMap<String, List<Resource>> resourcesListMap) {
        List<Resource> globalList = new ArrayList<Resource>();
        for (Iterator<List<Resource>> mapIter = 
            resourcesListMap.values().iterator(); mapIter.hasNext();) {
            globalList.addAll(mapIter.next());
        }
        return globalList;
    }

    // this method will go over all the resources list 
    // inside the archive and resolve for conflicts, duplicates 
    private static List<Resource> processResourcesListMap(
        HashMap<String, List<Resource>> resourcesListMap) {

        // whether we have top level sun-resources.xml
        // we need to process it separately
        boolean hasTopLevel = false;

        // the final resolved list to return
        List <Resource> globalResourcesList = new ArrayList<Resource>();

        // we need to resolve the duplicates and conflicts 
        // process the submodule level first and top level last 

        // process for submodule level first 
        // rule here: who appears later, who wins
        for (Iterator<String> mapIter = resourcesListMap.keySet().iterator(); 
            mapIter.hasNext();){
            String path = mapIter.next();
            if (path.equals(SUN_RESOURCE_XML_PATH)) {
                hasTopLevel = true;   
                continue;
            } else {
                processResourceListForXMLPath(globalResourcesList, 
                    resourcesListMap, path);
            } 
        }
      
        // now process the top level 
        if (hasTopLevel) {
            processResourceListForXMLPath(globalResourcesList,
                resourcesListMap, SUN_RESOURCE_XML_PATH);
        }
        
        return globalResourcesList;
    }

    private static void processResourceListForXMLPath(
        List<Resource> globalResourcesList, 
        HashMap<String, List<Resource>> resourcesListMap, String xmlPath) {
        List<Resource> resourcesList = resourcesListMap.get(xmlPath);
        for (Iterator<Resource> resIter = resourcesList.iterator(); 
            resIter.hasNext();) {
            Resource newResource = resIter.next();
            // add the resource to the global list
            // after checking with the current elements in the list
            for (Iterator<Resource> globalResIter = 
                globalResourcesList.iterator(); globalResIter.hasNext();) {
                Resource resourceInList = globalResIter.next();
                if (newResource.equals(resourceInList) ||
                    newResource.isAConflict(resourceInList)) {
                    globalResourcesList.remove(resourceInList);
                    break;
                }
            }
            globalResourcesList.add(newResource);
        }
    }

    //We must convert the absolute path back into a path of the
    //form ${com.sun.aas.instanceRoot}/applications/... so that
    //domain.xml is non-installation location specific.
    public static String getLocation(File appDir)
    {
        if (appDir != null) {
            return (RelativePathResolver.unresolvePath(
                        appDir.getAbsolutePath(),
                        new String[] {SystemPropertyConstants.INSTANCE_ROOT_PROPERTY,
                        SystemPropertyConstants.INSTALL_ROOT_PROPERTY}));
        } else {
            return null;
        }
    }

    // The following methods are related to get/set attributes of a deployed
    // app/module

    // Retrieving the location attribute
    public static String getLocation(String appId, DeployableObjectType type) 
        throws IASDeploymentException {
        try {
            MBeanServer mbs = MBeanServerFactory.getMBeanServer();
            String getOperationName = getGetOperationName(type);
            ObjectName appsConfigMBean = new ObjectName(appsConfigMBeanName);
            ObjectName componentON = (ObjectName)mbs.invoke(appsConfigMBean, getOperationName, new Object[]{appId},new String[]{"java.lang.String"});
            String location = (String)mbs.getAttribute(componentON, 
                ServerTags.LOCATION);
            return (new RelativePathResolver()).resolve(location);
        } catch (Exception e) {
            throw new IASDeploymentException(e);
        }
    }

    // Retrieving the enabled attribute
    public static boolean isEnabled(String appId, DeployableObjectType type) 
        throws IASDeploymentException {
        try {
            MBeanServer mbs = MBeanServerFactory.getMBeanServer();
            String getOperationName = getGetOperationName(type);
            ObjectName appsConfigMBean = new ObjectName(appsConfigMBeanName);
            ObjectName componentON = (ObjectName)mbs.invoke(appsConfigMBean, getOperationName, new Object[]{appId},new String[]{"java.lang.String"});

            String enabled = (String)mbs.getAttribute(componentON, 
                ServerTags.ENABLED);
            return Boolean.valueOf(enabled).booleanValue();
        } catch (Exception e) {
            throw new IASDeploymentException(e);
        }
    }

    // Retrieving the directory-deployed attribute
    public static boolean isDirectoryDeployed(String appId, 
        DeployableObjectType type) throws IASDeploymentException {
        try {
            MBeanServer mbs = MBeanServerFactory.getMBeanServer();
            String getOperationName = getGetOperationName(type);
            ObjectName appsConfigMBean = new ObjectName(appsConfigMBeanName);
            ObjectName componentON = (ObjectName)mbs.invoke(appsConfigMBean, getOperationName, new Object[]{appId},new String[]{"java.lang.String"});

            String directoryDeployed = (String)mbs.getAttribute(componentON,
                ServerTags.DIRECTORY_DEPLOYED);
            return Boolean.valueOf(directoryDeployed).booleanValue();
        } catch (Exception e) {
            throw new IASDeploymentException(e);
        }
    }

    // Retrieving the object-type attribute
    public static String getObjectType(String appId, DeployableObjectType type) 
        throws IASDeploymentException {
        try {
            // appclient module does not have object type
            if (type.isCAR()) {
                return null;
            }
            MBeanServer mbs = MBeanServerFactory.getMBeanServer();
            String getOperationName = getGetOperationName(type);
            ObjectName appsConfigMBean = new ObjectName(appsConfigMBeanName);
            ObjectName componentON = (ObjectName)mbs.invoke(appsConfigMBean, getOperationName, new Object[]{appId},new String[]{"java.lang.String"});

            String objectType = 
                (String)mbs.getAttribute(componentON, ServerTags.OBJECT_TYPE);
            return objectType;
        } catch (Exception e) {
            throw new IASDeploymentException(e);
        }
    }

    // Set the context-root attribute
    public static void setContextRoot(String appId, DeployableObjectType type, 
				      String contextRoot) throws IASDeploymentException {
        try {
            MBeanServer mbs = MBeanServerFactory.getMBeanServer();
            String getOperationName = getGetOperationName(type);
            ObjectName appsConfigMBean = new ObjectName(appsConfigMBeanName);
            ObjectName componentON = (ObjectName)mbs.invoke(appsConfigMBean, getOperationName, new Object[]{appId},new String[]{"java.lang.String"});

       
            if (contextRoot != null && contextRoot.length() > 0 ) {
                Attribute contextRootAttr = new Attribute(ServerTags.CONTEXT_ROOT, 
                    contextRoot);
                mbs.setAttribute(componentON, contextRootAttr);
            }
        } catch (Exception e) {
            throw new IASDeploymentException(e);
        }
    }

    // Retrieving the object-type attribute
    public static boolean isSystem(String appId, DeployableObjectType type) 
        throws IASDeploymentException{
        String objectType = getObjectType(appId, type);
        if (objectType != null) {
            return objectType.startsWith(SYSTEM_PREFIX);
        } else {
            return false;
        }
    }

    private static AttributeList populateAttributeList(
        DeploymentRequest request) throws IASDeploymentException {
        AttributeList attrList = new AttributeList();

        String name = request.getName();
        // name attribute
        if (name != null && name.length() > 0 ) {
            Attribute nameAttr = new Attribute(ServerTags.NAME, name);
            attrList.add(nameAttr);
        }

        // location attribute
        String location = getLocation(request.getDeployedDirectory());
        if (location != null && location.length() > 0 ) {
            Attribute locationAttr = new Attribute(ServerTags.LOCATION, location);
            attrList.add(locationAttr);
        }

        // description attribute
        String description = request.getDescription();
        if (description != null && description.length() > 0 ) {
            Attribute descriptionAttr =
                new Attribute(ServerTags.DESCRIPTION, description);
            attrList.add(descriptionAttr);
        }

        // enable attribute does not apply to app client module
        if (! request.getType().isCAR()) {
            String enabled = String.valueOf(request.isStartOnDeploy());
            if (enabled != null && enabled.length() > 0 ) {
                Attribute enableAttr = new Attribute(ServerTags.ENABLED, enabled);
                attrList.add(enableAttr);
            }
        }

        // java-web-start-enabled attribute
        // applies to application and appclient module
        if (request.getType().isAPP() || request.getType().isCAR()) {
            String jwsEnabled = 
                String.valueOf(request.isJavaWebStartEnabled());
            if (jwsEnabled != null && jwsEnabled.length() > 0 ) {
                Attribute jwsEnableAttr = new Attribute(
                    ServerTags.JAVA_WEB_START_ENABLED, jwsEnabled);
                attrList.add(jwsEnableAttr);
            }
        }

        // libraries attribute
        // applies to application, ejb, web module
        if (request.getType().isAPP() || request.getType().isEJB() || 
            request.getType().isWEB()) {
            String libraries = request.getLibraries();
            if (libraries != null && libraries.length() > 0 ) {
                Attribute librariesAttr = new Attribute(
                    ServerTags.LIBRARIES, libraries);
                attrList.add(librariesAttr);
            }
        }

        // context-root attribute only apply to web module
        if (request.getType().isWEB()) {
            String contextRoot = request.getContextRoot();
            if (contextRoot != null && contextRoot.length() > 0 ) {
                Attribute contextRootAttr =
                    new Attribute(ServerTags.CONTEXT_ROOT, contextRoot);
                attrList.add(contextRootAttr);
            }
        }

        // availability-enabled attribute does not apply to app client 
        // or connector module
        if (! request.getType().isCAR() && ! request.getType().isCONN()) {
            String availEnabled = String.valueOf(request.isAvailabilityEnabled());
            if (availEnabled!= null && availEnabled.length() > 0 ) {
                Attribute availEnabledAttr = 
                    new Attribute(ServerTags.AVAILABILITY_ENABLED, availEnabled);
                attrList.add(availEnabledAttr);
            }
        }

        // directory-deployed attribute
        String dirDeployed = String.valueOf(request.isDirectory());
        if (dirDeployed != null &&  dirDeployed.length() > 0 ) {
            Attribute  dirDeployedAttr =
                new Attribute(ServerTags.DIRECTORY_DEPLOYED, dirDeployed);
            attrList.add(dirDeployedAttr);
        }


        // other optional attributes like object-type
        Properties optionalAttributes = request.getOptionalAttributes();
        if (optionalAttributes != null) {
            // remove the virtual-server attribute which belongs
            // to application-ref
            // cloning is done so that the original Properties object
            // is not tampered with. The original Properties object is
            // referred to in a few other places.
            Properties mProps = (Properties) optionalAttributes.clone();
            try {
                mProps.remove(ServerTags.VIRTUAL_SERVERS);
            } catch(Exception re) {}

            Enumeration tags = mProps.keys();
            while(tags.hasMoreElements()) {
                String tag = (String)tags.nextElement();
                String value = mProps.getProperty(tag);
                if (tag != null & tag.length() > 0 &&
                    value != null && value.length() > 0) {
                    Attribute optionalAttr = new Attribute(tag, value);
                    attrList.add(optionalAttr);
                }
            }
        }

        return attrList;
    }

    // used when adding app/module to config
    public static void addToConfig(DeploymentRequest request) 
        throws IASDeploymentException {
        try {
            DeployableObjectType type = request.getType();
            MBeanServer mbs = MBeanServerFactory.getMBeanServer();
            ObjectName appsConfigMBean = new ObjectName(appsConfigMBeanName);

            String createOperationName = getCreateOperationName(type);

            AttributeList attrList = populateAttributeList(request);

            String[] signature = new String[] {AttributeList.class.getName()};
            Object[] params = new Object[] {attrList};
            mbs.invoke(appsConfigMBean, createOperationName, params, signature);
            getConfigContext().flush();
            getConfigContext().resetConfigChangeList();
        } catch (Exception e) {
            if (e instanceof IASDeploymentException) {
                throw (IASDeploymentException)e; 
            } else {
                throw new IASDeploymentException(e);
            }
        }
    }

    // used when updating config
    public static void updateConfig(DeploymentRequest request)
        throws IASDeploymentException { 
        try {
            String name = request.getName();
            ConfigBean appBean = 
                ApplicationHelper.findApplication(getConfigContext(), name);
            if (appBean != null) {
                for (Iterator itr = populateAttributeList(request).iterator();
                     itr.hasNext();) {
                    Attribute attr = (Attribute)itr.next();
                    appBean.setAttributeValue(attr.getName(), 
                                              (String)attr.getValue());
                }  
                getConfigContext().flush();
                getConfigContext().resetConfigChangeList();
            }
        } catch (Exception e) {
            if (e instanceof IASDeploymentException) {
                throw (IASDeploymentException)e; 
            } else {
                throw new IASDeploymentException(e);
            }
        }
    }

    // used when removing app/module to config
    public static void removeFromConfig(String name, DeployableObjectType type) 
        throws IASDeploymentException {
        try {
            MBeanServer mbs = MBeanServerFactory.getMBeanServer();
            String getOperationName = getGetOperationName(type);
            ObjectName appsConfigMBean = new ObjectName(appsConfigMBeanName);
            ObjectName componentON = (ObjectName)mbs.invoke(appsConfigMBean, getOperationName, new Object[]{name},new String[]{"java.lang.String"});
            if(componentON != null){
                String removeOperationName = getRemoveOperationName(type);
                mbs.invoke(appsConfigMBean, removeOperationName, new Object[]{name},new String[]{"java.lang.String"});
                getConfigContext().flush();
                getConfigContext().resetConfigChangeList();
            }
        } catch (Exception e) {
            throw new IASDeploymentException(e);
        }
    }

    public static boolean isRegistered(String appName, 
				       DeployableObjectType type) throws IASDeploymentException {
        try {
            MBeanServer mbs = MBeanServerFactory.getMBeanServer();
            String getOperationName = getGetOperationName(type);
            ObjectName appsConfigMBean = new ObjectName(appsConfigMBeanName);
            ObjectName componentON = null;

            try {
                componentON = (ObjectName)mbs.invoke(appsConfigMBean, getOperationName, new Object[]{appName}, new String[]{"java.lang.String"});
            } catch(Exception cn) {
            }

            if(componentON != null){
                return true;
            }
            return false;
        } catch (Exception e) {
            throw new IASDeploymentException(e);
        }
    }

    public static ConfigContext getConfigContext() {
        try {
            AdminService adminService = AdminService.getAdminService();
            if (adminService != null) {
                ConfigContext config =
                        adminService.getAdminContext().getAdminConfigContext();
                return config;
            } else {
               return null;
            }
        } catch (Exception e) {
            return null;
        }   
    }       

    public static String getInstanceName() {
         return ApplicationServer.getServerContext().getInstanceName();
    }

    public static String getDomainName()
        throws IASDeploymentException {
        try {
           AdminService adminService = AdminService.getAdminService();
            if (adminService != null) {
                String domainName =
                        adminService.getAdminContext().getDomainName();
                return domainName;
            } else {
               return null;
            }
        } catch (Exception e) {
            throw new IASDeploymentException(e);
        }
    }

            
    /**
     * This method checks if the app specified is of
     * a system application/module
     * @param appID name of the application
     * @param type object type of the application
     * @param action deploy/redeploy/undeploy/enable/disable
     * @throws IASDeploymentException if application/module is
     * of type system
     */
    public static void validate(String appID, DeployableObjectType type,
        String action) throws IASDeploymentException {
        boolean allowSystemAppModification =
            (Boolean.valueOf(System.getProperty(
                Constants.ALLOW_SYSAPP_DEPLOYMENT, "false")).booleanValue());

        if(allowSystemAppModification)
            return;
        if (isSystem(appID, type)) {
            String msg = localStrings.getString(
                "enterprise.deployment.phasing.component_is_system",
                new Object[]{ action, appID });
            throw new IASDeploymentException(msg);
        }
    }

    /**
     * This method checks if any of the virtual servers has the given web
     * module as default-web-module. If yes, it throws exception.
     * @param webModuleName the name of the web module.
     * @throws ConfigException if any of the virtual servers has this web
     * module as default-web-module.
     */
    public static void checkWebModuleReferences(String webModuleName)
    throws IASDeploymentException {
        ArrayList virtualServerIds = new ArrayList();

        try {
            Config config  = (Config) ConfigBeansFactory.getConfigBeanByXPath(
                getConfigContext(), ServerXPathHelper.XPATH_CONFIG);
            HttpService httpService = config.getHttpService();
            VirtualServer[] virtualServers = httpService.getVirtualServer();
            for (int j = 0; j < virtualServers.length; j++) {
                VirtualServer aServer   = virtualServers[j];
                String defWebModule     = aServer.getDefaultWebModule();
                if ((defWebModule != null) &&
                (defWebModule.equals(webModuleName))) {
                    virtualServerIds.add(aServer.getId());
                }
            }
        } catch (Exception e) { 
            throw new IASDeploymentException(e);
        }       
        if (!virtualServerIds.isEmpty()) {
            throw new IASDeploymentException(localStrings.getString(
                "enterprise.deployment.phasing.def_web_module_refs_exist",
                virtualServerIds.toString(), webModuleName));
        }
    }

    public static void checkAbort(String moduleID) 
        throws IASDeploymentException {
        DeploymentRequest request = 
            DeploymentRequestRegistry.getRegistry().getDeploymentRequest(
                moduleID);
        if (request != null && request.isAborted()) {
            DeploymentRequestRegistry.getRegistry().removeDeploymentRequest(
                moduleID);
            String msg = localStrings.getString(
                "enterprise.deployment.phasing.operation.abort",
                new Object[]{ moduleID });
            throw new IASDeploymentException(msg);
        }  
    }

   /**
     * This method returns the deployableObjectType of an archive by checking
     * the deployable descriptors in the archive
     * @param filePath absolute path to the archive
     * @return type DeployableObjectType
     */
    public static DeployableObjectType getTypeFromFile(String name, String filePath)
        throws DeploymentException {

        if(filePath == null)
            throw new DeploymentException("deploy file not specified");

        try {
            //@@@ special handling the deployment of .class POJO webservice
            //class.  
            if (filePath.endsWith(".class")) {
                // get the top directory from the moduleID, in the case of 
                // autodeploy, the moduleID will be the full class name (with package)
                // with . replaced with _
                StringTokenizer tk = new StringTokenizer(name, "_");
                File f = new File(filePath);
                for (int i=0;i<tk.countTokens();i++) {
                    f = f.getParentFile();
                }
                try {
                    // it's important to not cache this because the class can 
                    // change overtime... we need to load the last one.
                    URL[] urls = new URL[1];
                    urls[0] = f.toURL();
                    URLClassLoader cLoader = new URLClassLoader(urls, 
                                DeploymentServiceUtils.class.getClassLoader());
                    String className = name.replaceAll("_", ".");
                    Class clazz = cLoader.loadClass(className);
                    if (clazz!=null && clazz.getAnnotation(javax.ejb.Stateless.class)!=null) {
                        return DeployableObjectType.EJB;
                    } else {
                        return DeployableObjectType.WEB;
                    }                    
                } catch(Exception e) {
                    return DeployableObjectType.WEB;
                }
            }

            Archivist archivist = 
                ArchivistFactory.getArchivistForArchive(filePath);
            ModuleType moduleType = archivist.getModuleType();
            return getDeployableObjectType(moduleType);
        } catch(Exception  ex) {
            DeploymentException de = new DeploymentException(
                localStrings.getString(
                    "enterprise.deployment.unknown.application.type", filePath));
            de.initCause(ex);
            throw de;
        }
    }

    public static DeploymentTarget getAndValidateDeploymentTarget(String 
        targetName, String appName, boolean isDeleting) 
        throws IASDeploymentException {
        try {
            final DeploymentTarget target = DeploymentTargetFactory.getDeploymentTargetFactory().getTarget(getConfigContext(), getDomainName(), targetName);
            if (targetName == null) {
                // If the targetName passed in was null, we need to set it
                // to its default value.
                targetName = target.getTarget().getName();
            }
            if (target.getTarget().getType() == TargetType.SERVER ||
                target.getTarget().getType() == TargetType.DAS) {
                // make sure we have the reference before we delete it
                if (isDeleting) {
                    if (!ServerHelper.serverReferencesApplication(
                        getConfigContext(), targetName, appName)) {
                        throw new IASDeploymentException(localStrings.getString("serverApplicationRefDoesNotExist", targetName, appName));
                    }
                // make sure we don't have the reference already before 
                // we create it again
                } else {
                   if (ServerHelper.serverReferencesApplication(
                        getConfigContext(), targetName, appName)) {
                        throw new IASDeploymentException(localStrings.getString("serverApplicationRefAlreadyExists", appName, targetName));
                   }
                }
            } else if (target.getTarget().getType() == TargetType.CLUSTER){
               // make sure we have the reference before we delete it
                if (isDeleting) {
                    if (!ClusterHelper.clusterReferencesApplication(
                        getConfigContext(), targetName, appName)) {
                        throw new IASDeploymentException(localStrings.getString("clusterApplicationRefDoesNotExist", targetName, appName));
                    }
                // make sure we don't have the reference already before 
                // we create it again
                } else {
                    if (ClusterHelper.clusterReferencesApplication(
                        getConfigContext(), targetName, appName)) {
                        throw new IASDeploymentException(localStrings.getString("clusterApplicationRefAlreadyExists", appName, targetName));
                    }
                }
            }
         
            return target;
        } catch (IASDeploymentException ex) {
            throw (ex);
        } catch (Exception ex) {
            throw new IASDeploymentException(ex);
        }
    }

    /**
     * Set http listener host and port in deployment request. If the server
     * is not configured properly the defaults used are localhost:8080 for
     * clear and localhost:8181 for SSL.
     */
    public static void setHostAndPort(DeploymentRequest req)
            throws ServerInstanceException {

        String virtualServers = (String) req.getOptionalAttributes().get(ServerTags.VIRTUAL_SERVERS);
        if (virtualServers==null) {
            HostAndPort hap = getHostAndPort(false);
            if(hap != null) {
                req.setHttpHostName(getHostName(hap));
                req.setHttpPort(getPort(hap, false));
            }
            hap = getHostAndPort(true);
            if(hap != null) {
                req.setHttpsHostName(getHostName(hap));
                req.setHttpsPort(getPort(hap, true));
            }
        } else {
            StringTokenizer st = new StringTokenizer(virtualServers,",");
            if (st.hasMoreTokens()) {
                String aVirtualServer = st.nextToken();
                HostAndPort hap = getVirtualServerHostAndPort(aVirtualServer, false);
                if(hap != null) {
                    req.setHttpHostName(getHostName(hap));

                    req.setHttpPort(getPort(hap, false));
                }
                hap = getVirtualServerHostAndPort(aVirtualServer, true);
                if(hap != null) {
                    req.setHttpsHostName(getHostName(hap));
                    req.setHttpsPort(getPort(hap, true));
                }
            }
        }
    }

    private static String getHostName(HostAndPort hap) {
        String hostName = hap.getHost();
        if (hostName == null || hostName.trim().equals("")) {
            hostName = getDefaultHostName();
        }
        return hostName;
    }

    private static String getDefaultHostName() {
        String defaultHostName = "localhost";
        try {
            InetAddress host = InetAddress.getLocalHost();
            defaultHostName = host.getCanonicalHostName();
        } catch(UnknownHostException uhe) {
            sLogger.log(Level.FINEST, "mbean.get_local_host_error", uhe);
            sLogger.log(Level.INFO, "mbean.use_default_host");

       }
        return defaultHostName;
    }

    private static int getPort(HostAndPort hap, boolean securityEnabled) {
        int port = hap.getPort();
        if (port == 0) {
            port = getDefaultPort(securityEnabled);
        }
        return port;
    }

    private static int getDefaultPort(boolean securityEnabled) {
        int port = 0;
        if (securityEnabled) {
            port = 8181;
        } else {
            port = 8080;
        }
        sLogger.log(Level.INFO, "mbean.use_default_port", String.valueOf(port));
        return port;
    }

    /**
     * Get value of named attribute from attributes list. If an attribute with
     * specified name does not exist, the method returns null. If there are
     * than one attributes with same name then the method returns value of
     * first matching attribute.
     *
     * @param attrs list of attributes
     * @param attrName name of the attribute
     * @return value of the specified attrName or null if the attrName is
     *     not present in specified attrs
     */
    private static Object getNamedAttributeValue(AttributeList attrs,
            String attrName) {
        if (attrs == null || attrName == null) {
            return null;
        }
        Object value = null;
        Iterator iter = attrs.iterator();
        while (iter.hasNext()) {
           Attribute attr = (Attribute)iter.next();
           if (attrName.equals(attr.getName())) {
               value = attr.getValue();
               break;
           }
        }
        return value;
    }

    /**
     * @param securityEnabled
     * @throws ServerInstanceException
     */
    private static HostAndPort getHostAndPort(boolean securityEnabled) throws ServerInstanceException {
        HostAndPort hAndp = null;
        try {
            MBeanServer mbs = MBeanServerFactory.getMBeanServer();
            ObjectName objectName = new ObjectName(getDomainName()+":type=configs,category=config");
            String operationName1 = "getConfig";
            ObjectName[] configs = (ObjectName[])mbs.invoke(objectName,operationName1, emptyParams,emptySignature);
            String configName = (String)mbs.getAttribute(configs[0], "name");
            ObjectName httpService = new ObjectName(getDomainName()+":type=http-service,config="+configName+",category=config");
            String operationName2 = "getHttpListener";
            ObjectName[] httpListener = (ObjectName[])mbs.invoke(httpService, operationName2,emptyParams,emptySignature);

            String serverName = null;
            int port = 0;
            for (int i = 0; i < httpListener.length; i++) {
                AttributeList attrs = mbs.getAttributes(httpListener[i],
                        httpListenerAttrNames);
                Boolean bb = Boolean.valueOf((String)getNamedAttributeValue(
                        attrs, LISTENER_ENABLED));
                boolean enabled = ((bb == null) ? false : bb.booleanValue());
                if (!enabled) {
                    // Listener is not enabled
                    continue;
                }
                String vs = (String)getNamedAttributeValue(attrs, DEF_VS);
                if (ADMIN_VS.equals(vs)) {
                    // Listener is reserved for administration
                    continue;
                }
                bb = Boolean.valueOf((String)getNamedAttributeValue(
                        attrs, SEC_ENABLED));
                boolean sec_on = ((bb == null) ? false : bb.booleanValue());
                if (securityEnabled == sec_on) {
                    serverName = (String)getNamedAttributeValue(attrs,
                            SERVER_NAME);
                    if (serverName == null || serverName.trim().equals("")) {
                        serverName = getDefaultHostName();
                    }

                    String portStr = (String)getNamedAttributeValue(attrs,
                            PORT);
                    String redirPort = (String)getNamedAttributeValue(attrs,
                            REDIRECT_PORT);
                    if (redirPort != null && !redirPort.trim().equals("")) {
                        portStr = redirPort;
                    }
                    String resolvedPort =
                        new PropertyResolver(getConfigContext(),
                            getInstanceName()).resolve(portStr);
                    port = Integer.parseInt(resolvedPort);
                    break;
                }
            }
            hAndp = new HostAndPort(serverName, port);
        }
        catch (Exception e) {
            throw new ServerInstanceException(e.getLocalizedMessage());
        }
        return hAndp;
    }

    private static HostAndPort getVirtualServerHostAndPort(String vs, boolean securityEnabled)
        throws ServerInstanceException
    {
        String serverName = null;
        int port = 0;

        try {
            MBeanServer mbs = MBeanServerFactory.getMBeanServer();

            ObjectName objectName = new ObjectName(
            getDomainName()+":type=configs,category=config");
            String operationName1 = "getConfig";
            ObjectName[] configs = (
            ObjectName[])mbs.invoke(objectName,operationName1,
            emptyParams,emptySignature);
            String configName = (String)mbs.getAttribute(configs[0], "name");
            ObjectName httpService = new ObjectName(
            getDomainName()+":type=http-service,config="+configName+",category=config");

            // http listeners for the given config

            String operationName2 = "getHttpListener";
            ObjectName[] httpListener = (ObjectName[])mbs.invoke(httpService,
            operationName2,emptyParams,emptySignature);

            // virtual servers for the given config

            operationName2 = "getVirtualServer";
            ObjectName[] virtualServer = (ObjectName[])mbs.invoke(httpService,
            operationName2,emptyParams,emptySignature);

            // iterate for each of the config virtual server

            for (int i = 0; i < virtualServer.length; i++) {

                AttributeList vsAttrs = mbs.getAttributes(virtualServer[i], vsAttrNames);

                // virtual server id check
                //
                // if the virtual server obtained from application ref
                // does not match with the virtual server from config
                // then continue with next virtual server

                String id = (String)getNamedAttributeValue(vsAttrs, ID);
                if (! id.equals(vs)) {
                    continue;
                }

                // should we check for state, let us assume ON for PE

                // http listener
                //
                // Obtain the http listeners list from the virtual server
                // and iterate to match with the http listeners from config.
                // When a match is found get the host and port data

                String httpListeners = (String) getNamedAttributeValue(vsAttrs, "http-listeners");
                String vsHttpListener = null;
                List httpListenerList = StringUtils.parseStringList(httpListeners, " ,");
                ListIterator hlListIter = httpListenerList.listIterator();

                while(hlListIter.hasNext()) {

                    vsHttpListener = (String) hlListIter.next();

                    for (int j = 0; j < httpListener.length; j++) {

                        AttributeList attrs = mbs.getAttributes(httpListener[j],
                        httpListenerAttrNames);

                        // http listener id check
                        String listenerId = (String)getNamedAttributeValue(attrs, ID);
                        if (! listenerId.equals(vsHttpListener)) {
                            continue;
                        }

                        Boolean bb = Boolean.valueOf((String)getNamedAttributeValue(
                        attrs, LISTENER_ENABLED));
                        boolean enabled = ((bb == null) ? false : bb.booleanValue());
                        if (!enabled) {
                            // Listener is not enabled
                            continue;
                        }
                        bb = Boolean.valueOf((String)getNamedAttributeValue( attrs, SEC_ENABLED));
                        boolean sec_on = ((bb == null) ? false : bb.booleanValue());
                        if (securityEnabled == sec_on) {
                            serverName = (String)getNamedAttributeValue(attrs, SERVER_NAME);
                            if (serverName == null || serverName.trim().equals("")) {
                                serverName = getDefaultHostName();
                            }
                            String portStr = (String)getNamedAttributeValue(attrs, PORT);
                            String redirPort = (String)getNamedAttributeValue(attrs, REDIRECT_PORT);
                            if (redirPort != null && !redirPort.trim().equals("")) {
                                portStr = redirPort;
                            }
                            final String resolvedPort =
                                    new PropertyResolver(getConfigContext(),
                                        getInstanceName()).resolve(portStr);
                            port = Integer.parseInt(resolvedPort);
                            return new HostAndPort(serverName, port);
                        }
                    }
                }
            }
        } catch (Exception e) {
            throw new ServerInstanceException(e.getLocalizedMessage());
        }
        return null;
    }

    public static DeploymentTarget getDeploymentTarget(String targetName) 
        throws IASDeploymentException {
        try {
            final DeploymentTarget target = DeploymentTargetFactory.getDeploymentTargetFactory().getTarget(getConfigContext(), getDomainName(), targetName);
            return target;
        } catch (IASDeploymentException ex) {
            throw (ex);
        } catch (Exception ex) {
            throw new IASDeploymentException(ex);
        }
    }


    public static void checkAppReferencesBeforeUndeployFromDomain(
        String appName) throws IASDeploymentException {
        try {
            // make sure no one reference the app before we undeploy it
            if (ApplicationHelper.isApplicationReferenced(
                getConfigContext(), appName)) {
                throw new IASDeploymentException(localStrings.getString("applicationIsReferenced", appName, ApplicationHelper.getApplicationReferenceesAsString(getConfigContext(), appName)));
            }
        } catch (IASDeploymentException ex) {
            throw (ex);
        } catch (Exception ex) {
            throw new IASDeploymentException(ex);
        }
    }

    /**     
     * Returns the type of the registered component.
     * If the component is not registered throws a DeploymentException
     * @param name name of the component
     * @return type DeployableObjectType of the registered component(app/module)
     */     
    public static DeployableObjectType getRegisteredType(String name) 
        throws DeploymentException {
        try{
            for(int i = 0; i< deployableObjectTypes.length; i++)
            {
                if (isRegistered(name, deployableObjectTypes[i])) {
                    return deployableObjectTypes[i];
                }
            }
        }catch(Exception e){
            throw new DeploymentException(e.getMessage());
        }
        String msg = localStrings.getString(
            "enterprise.deployment.component.not.registered", name);
        throw new DeploymentException(msg);
    }

    static String getCreateOperationName(DeployableObjectType type)
        throws IASDeploymentException {
        if (type.isAPP()) {
            return CREATE_J2EE_APPLICATION;
        }
        else if(type.isEJB()) {
            return CREATE_EJB_MODULE;
        }
        else if(type.isWEB()) {
            return CREATE_WEB_MODULE;
        }
        else if(type.isCONN()) {
            return CREATE_CONNECTOR_MODULE;
        }
        else if(type.isCAR()) {
            return CREATE_APPCLIENT_MODULE;
        }
        else return null;
    }

    static String getGetOperationName(DeployableObjectType type) 
        throws IASDeploymentException {
        if (type.isAPP()) {
            return GET_J2EE_APPLICATION;
        }
        else if (type.isEJB()) {
            return GET_EJB_MODULE;
        }
        else if (type.isWEB()) {
            return GET_WEB_MODULE;
        }
        else if (type.isCONN()) {
            return GET_CONNECTOR_MODULE;
        }
        else if (type.isCAR()) {
            return GET_APPCLIENT_MODULE;
        }
        else return null;
    }  

    static String getRemoveOperationName(DeployableObjectType type) 
        throws IASDeploymentException {
	if(type.isAPP()) {
	    return REMOVE_J2EE_APPLICATION;
	}
	else if(type.isEJB()) {
	    return REMOVE_EJB_MODULE;
	}
	else if(type.isWEB()) {
	    return REMOVE_WEB_MODULE;
	}
	else if(type.isCONN()) {
	    return REMOVE_CONNECTOR_MODULE;
	}
	else if(type.isCAR()) {
	    return REMOVE_APPCLIENT_MODULE;
	}
	else return null;
    }

    /**
     * This method should be removed once we remove the use of 
     * DeployableObjectType.
     */
    public static DeployableObjectType getDeployableObjectType(ModuleType type) {
        if (ModuleType.EAR.equals(type)) {
            return DeployableObjectType.APP;
        } else if (ModuleType.EJB.equals(type)) {
            return DeployableObjectType.EJB;
        } else if (ModuleType.WAR.equals(type)) {
            return DeployableObjectType.WEB;
        } else if (ModuleType.CAR.equals(type)) {
            return DeployableObjectType.CAR;
        } else if (ModuleType.RAR.equals(type)) {
            return DeployableObjectType.CONN;
        }
        return null;
    }

    /** string constants */
    private static final String CREATE_J2EE_APPLICATION =
        "createJ2eeApplication";
    private static final String CREATE_EJB_MODULE =
        "createEjbModule";
    private static final String CREATE_WEB_MODULE =
        "createWebModule";
    private static final String CREATE_CONNECTOR_MODULE =
        "createConnectorModule";
    private static final String CREATE_APPCLIENT_MODULE =
        "createAppclientModule";
    private static final String GET_J2EE_APPLICATION =
        "getJ2eeApplicationByName";
    private static final String GET_EJB_MODULE =
        "getEjbModuleByName";
    private static final String GET_WEB_MODULE =
        "getWebModuleByName";
    private static final String GET_CONNECTOR_MODULE =
        "getConnectorModuleByName";
    private static final String GET_APPCLIENT_MODULE =
        "getAppclientModuleByName";
    private static final String REMOVE_J2EE_APPLICATION = 
        "removeJ2eeApplicationByName";
    private static final String REMOVE_EJB_MODULE = 
        "removeEjbModuleByName";
    private static final String REMOVE_WEB_MODULE = 
        "removeWebModuleByName";
    private static final String REMOVE_CONNECTOR_MODULE = 
        "removeConnectorModuleByName";
    private static final String REMOVE_APPCLIENT_MODULE = 
        "removeAppclientModuleByName";

    private static final String appsConfigMBeanName = 
        "com.sun.appserv:type=applications,category=config";
    private static final String SYSTEM_PREFIX = "system-";
 
    private static final String ADMIN_VS = "__asadmin";

     // Attribue names for http listeners
    private static final String PORT = "port";
    private static final String DEF_VS = "default-virtual-server";
    private static final String SERVER_NAME = "server-name";
    private static final String REDIRECT_PORT = "redirect-port";
    private static final String SEC_ENABLED = "security-enabled";
    private static final String LISTENER_ENABLED = "enabled";
    private static final String OBJECT_TYPE = "object-type";

    // Attribute names for virtual server
    private static final String HOSTS = "hosts";
    private static final String HTTP_LISTENERS = "http_listeners";
    private static final String DEFAULT_WEB_MODULE = "default_web_module";
    private static final String STATE = "state";
    private static final String ID = "id";

    // resouces constants
    private static final String SUN_RESOURCE_XML = "sun-resources.xml";
    private static final String SUN_RESOURCE_XML_PATH = 
        "META-INF" + File.separator + "sun-resources.xml";

    private static final DeployableObjectType[] deployableObjectTypes =
        new DeployableObjectType[] { DeployableObjectType.APP,
        DeployableObjectType.EJB, DeployableObjectType.WEB,
        DeployableObjectType.CONN, DeployableObjectType.CAR };

    private static String[] httpListenerAttrNames = {LISTENER_ENABLED,
            DEF_VS, SERVER_NAME, REDIRECT_PORT, PORT, SEC_ENABLED, ID };

    private static String[] vsAttrNames = {HOSTS, HTTP_LISTENERS,
            DEFAULT_WEB_MODULE, STATE, ID};

}
