/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */
package com.sun.enterprise.security;

import com.sun.enterprise.security.SecurityUtil;
import com.sun.enterprise.server.pluggable.SecuritySupport;

import java.io.IOException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.util.Enumeration;
import javax.net.ssl.KeyManagerFactory;
import javax.net.ssl.TrustManagerFactory;
import javax.net.ssl.KeyManager;
import javax.net.ssl.TrustManager;
import javax.net.ssl.X509KeyManager;

import org.apache.tomcat.util.net.jsse.JSSE14SocketFactory;
import org.apache.tomcat.util.net.jsse.JSSEKeyManager;

/**
 * NSS Socket Factory.
 * 
 * @author Jean-Francois Arcand
 */
public class NSSSocketFactory  extends JSSE14SocketFactory {

    final public static String INTERNAL_TOKEN = "NSS Certificate DB";
    
    public NSSSocketFactory() {
        super();        
    }
       
    
    /*
     * Gets the SSL server's keystore.
     */
    protected KeyStore getKeystore(String type, String pass) throws IOException {
        String keyAlias = (String)attributes.get("keyAlias");
        String token = getTokenFromKeyAlias(keyAlias);        
        SecuritySupport secSupp = SecurityUtil.getSecuritySupport();
        KeyStore ks = secSupp.getKeyStore(token);
        if (ks==null) {
            throw new IOException("keystore not found for token " + token);
        }
        return ks;
    }
    
    
    /*
     * Gets the SSL server's truststore. JDK 1.5 provider has 
     * issues in loading some of the NSStrust certs. In is case, we have our native 
     * code to load all trust certs and put it into a keystore. That is why we 
     * will have more than one keyStores even in flat file NSS. In General, 
     * we even cannot assume there is only one keystores. In case of hardware 
     * accelerators, there will be multiple (one for earch HW).  
     */
    protected KeyStore getTrustStore(String keystoreType) throws IOException {
        try {
            return SSLUtils.mergingTrustStores(
                           SecurityUtil.getSecuritySupport().getTrustStores());
        } catch (Exception ex) {
            throw new IOException(ex.getMessage());
        }
    }

    /**
     * Gets the initialized key managers.
     */
    protected KeyManager[] getKeyManagers(String keystoreType,
                                          String algorithm,
                                          String keyAlias)
                throws Exception {
        KeyManager[] kms = null;
        SecuritySupport secSupp = SecurityUtil.getSecuritySupport();
        String token=getTokenFromKeyAlias(keyAlias);
        String certAlias = getCertAliasFromKeyAlias(keyAlias);
        String keystorePass = secSupp.getKeyStorePassword(token);
        KeyStore ks = secSupp.getKeyStore(token);
        if (ks==null) {
            throw new IOException("keystore not found for token " + token);
        }
        KeyManagerFactory kmf = KeyManagerFactory.getInstance(algorithm);
        kmf.init(ks, keystorePass.toCharArray());
        kms = kmf.getKeyManagers();
        for(int i=0; certAlias!=null && i<kms.length; i++) {
            kms[i] = new JSSEKeyManager((X509KeyManager)kms[i], certAlias);
        }
        return kms;
    }
    
    private static String getTokenFromKeyAlias(String keyAlias) {
        String token = null;
        if (keyAlias!=null) {
            int idx = keyAlias.indexOf(':');
            if (idx != -1) {
                token = keyAlias.substring(0, idx);
            }
        }        
        if (token==null) {
            token = INTERNAL_TOKEN;
        } else {
            token = token.trim();
        }
        return token;
    }

    /**
     * @param keyAlias format is "token:certAlias" or "certAlias"
     *
     * in Appserver design, the "token" name part serves two purposes
     * (1) identify the token in NSS DB
     *     e.g. ./modutil -list -dbdir /export/sonia/appserver/domains/domain1/config/
     * (2) "token:certAlias" WHOLE string is the cert alias in NSS
     *     for example ("nobody@test" is the token name):
     *     ./certutil -L  -h nobody@test  -d /export/sonia/appserver/domains/domain1/config
     *     Enter Password or Pin for "nobody@test":
     *     nobody@test:mps                                              u,u,u
     *     nobody@test:J2EESQECA                                        u,u,u
     *     nobody@test:AppServer1                                       u,u,u
     *     nobody@test:Server-Cert                                      u,u,u
     *
     * JDK5 KeyStore of type "SunPKCS11" identifies cert by "certAlias" part of "token:certAlias"
     */
    private static String getCertAliasFromKeyAlias(String keyAlias) {
        String certAlias = null;
        if (keyAlias!=null) {
            int idx = keyAlias.indexOf(':');
            if (idx == -1) {
                certAlias = keyAlias;
            } else {
                idx++;
                if (idx < keyAlias.length()-1 ) {
                    certAlias = keyAlias.substring(idx);
                }
            } 
        }
        if (certAlias!=null)
            certAlias = certAlias.trim();
        return certAlias;
    }

}
