/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the "License").  You may not use this file except 
 * in compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * glassfish/bootstrap/legal/CDDLv1.0.txt or 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html. 
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * HEADER in each file and include the License file at 
 * glassfish/bootstrap/legal/CDDLv1.0.txt.  If applicable, 
 * add the following below this CDDL HEADER, with the 
 * fields enclosed by brackets "[]" replaced with your 
 * own identifying information: Portions Copyright [yyyy] 
 * [name of copyright owner]
 */
// Copyright (c) 1998, 2005, Oracle. All rights reserved.  
package oracle.toplink.essentials.internal.descriptors;

import java.lang.reflect.*;
import oracle.toplink.essentials.exceptions.*;
import oracle.toplink.essentials.internal.helper.*;
import oracle.toplink.essentials.mappings.AttributeAccessor;
import oracle.toplink.essentials.internal.security.*;

/**
 * <p><b>Purpose</b>: A wrapper class for handling cases when the domain object attributes are
 * to be accessed thru the accessor methods. This could happen if the variables are not defined
 * public in the domain object.
 *
 * @author Sati
 * @since TOPLink/Java 1.0
 */
public class MethodAttributeAccessor extends AttributeAccessor {
    protected String setMethodName;
    protected String getMethodName;
    protected transient Method setMethod;
    protected transient Method getMethod;

    /**
     * Return the return type of the method accessor.
     */
    public Class getAttributeClass() {
        if (getGetMethod() == null) {
            return null;
        }

        return getGetMethodReturnType();
    }

    /**
     * Gets the value of an instance variable in the object.
     */
    public Object getAttributeValueFromObject(Object anObject) throws DescriptorException {
        try {
            return PrivilegedAccessController.invokeMethod(getGetMethod(), anObject, (Object[])null);
        } catch (IllegalArgumentException exception) {
            throw DescriptorException.illegalArgumentWhileGettingValueThruMethodAccessor(getGetMethodName(), anObject.getClass().getName(), exception);
        } catch (IllegalAccessException exception) {
            throw DescriptorException.illegalAccessWhileGettingValueThruMethodAccessor(getGetMethodName(), anObject.getClass().getName(), exception);
        } catch (InvocationTargetException exception) {
            throw DescriptorException.targetInvocationWhileGettingValueThruMethodAccessor(getGetMethodName(), anObject.getClass().getName(), exception);
        } catch (NullPointerException exception) {
            // Some JVM's throw this exception for some very odd reason
            throw DescriptorException.nullPointerWhileGettingValueThruMethodAccessor(getGetMethodName(), anObject.getClass().getName(), exception);
        }
    }

    /**
     * Return the accessor method for the attribute accessor.
     */
    protected Method getGetMethod() {
        return getMethod;
    }

    /**
     * Return the name of theh accessor method for the attribute accessor.
     */
    public String getGetMethodName() {
        return getMethodName;
    }

    public Class getGetMethodReturnType() {
        return PrivilegedAccessController.getMethodReturnType(getGetMethod());
    }

    /**
     * Return the set method for the attribute accessor.
     */
    protected Method getSetMethod() {
        return setMethod;
    }

    /**
     * Return the name of the set method for the attribute accessor.
     */
    public String getSetMethodName() {
        return setMethodName;
    }

    public Class getSetMethodParameterType() {
        return PrivilegedAccessController.getMethodParameterTypes(getSetMethod())[0];
    }

    /**
     * Set get and set method after creating these methods by using
     * get and set method names
     */
    public void initializeAttributes(Class theJavaClass) throws DescriptorException {
        if (getAttributeName() == null) {
            throw DescriptorException.attributeNameNotSpecified();
        }
        try {
            setGetMethod(Helper.getDeclaredMethod(theJavaClass, getGetMethodName(), (Class[])null));

            // The parameter type for the set method must always be the return type of the get method.
            Class[] parameterTypes = new Class[1];
            parameterTypes[0] = getGetMethodReturnType();
            setSetMethod(Helper.getDeclaredMethod(theJavaClass, getSetMethodName(), parameterTypes));
        } catch (NoSuchMethodException ex) {
            DescriptorException descriptorException = DescriptorException.noSuchMethodWhileInitializingAttributesInMethodAccessor(getSetMethodName(), getGetMethodName(), theJavaClass.getName());
            descriptorException.setInternalException(ex);
            throw descriptorException;
        } catch (SecurityException exception) {
            DescriptorException descriptorException = DescriptorException.securityWhileInitializingAttributesInMethodAccessor(getSetMethodName(), getGetMethodName(), theJavaClass.getName());
            descriptorException.setInternalException(exception);
            throw descriptorException;
        }
    }

    public boolean isMethodAttributeAccessor() {
        return true;
    }

    /**
     * Sets the value of the instance variable in the object to the value.
     */
    public void setAttributeValueInObject(Object domainObject, Object attributeValue) throws DescriptorException {
        Object[] parameters = new Object[1];
        parameters[0] = attributeValue;
        try {
            PrivilegedAccessController.invokeMethod(getSetMethod(), domainObject, parameters);
        } catch (IllegalAccessException exception) {
            throw DescriptorException.illegalAccessWhileSettingValueThruMethodAccessor(getSetMethodName(), attributeValue, exception);
        } catch (IllegalArgumentException exception) {
            // TODO: This code should be removed, it should not be required and may cause unwanted sideeffects.
            // Allow XML change set to merge correctly since new value in XML change set is always String
            try {
                if (attributeValue instanceof String) {
                    Object newValue = ConversionManager.getDefaultManager().convertObject(attributeValue, getAttributeClass());
                    Object[] newParameters = new Object[1];
                    newParameters[0] = newValue;
                    PrivilegedAccessController.invokeMethod(getSetMethod(), domainObject, newParameters);
                    return;
                }
            } catch (Exception e) {
                // Do nothing and move on to throw the original exception
            }
            throw DescriptorException.illegalArgumentWhileSettingValueThruMethodAccessor(getSetMethodName(), attributeValue, exception);
        } catch (InvocationTargetException exception) {
            throw DescriptorException.targetInvocationWhileSettingValueThruMethodAccessor(getSetMethodName(), attributeValue, exception);
        } catch (NullPointerException exception) {
            try {
                // TODO: This code should be removed, it should not be required and may cause unwanted sideeffects.
                // cr 3737  If a null pointer was thrown because toplink attempted to set a null referece into a
                // primitive create a primitive of value 0 to set in the object.
                // Is this really the best place for this? is this not why we have null-value and conversion-manager?
                Class fieldClass = getSetMethodParameterType();

                //Found when fixing Bug2910086
                if (fieldClass.isPrimitive() && (attributeValue == null)) {
                    parameters[0] = ConversionManager.getDefaultManager().convertObject(new Integer(0), fieldClass);
                    PrivilegedAccessController.invokeMethod(getSetMethod(), domainObject, parameters);
                } else {
                    // Some JVM's throw this exception for some very odd reason
                    throw DescriptorException.nullPointerWhileSettingValueThruInstanceVariableAccessor(getAttributeName(), attributeValue, exception);
                }
            } catch (IllegalAccessException accessException) {
                throw DescriptorException.nullPointerWhileSettingValueThruInstanceVariableAccessor(getAttributeName(), attributeValue, exception);
            } catch (InvocationTargetException invocationException) {
                throw DescriptorException.nullPointerWhileSettingValueThruInstanceVariableAccessor(getAttributeName(), attributeValue, exception);
            }
        }
    }

    /**
     * Set the accessor method for the attribute accessor.
     */
    protected void setGetMethod(Method getMethod) {
        this.getMethod = getMethod;
    }

    /**
     * Set the name of the accessor method for the attribute accessor.
     */
    public void setGetMethodName(String getMethodName) {
        this.getMethodName = getMethodName;
    }

    /**
     * Set the set method for the attribute accessor.
     */
    protected void setSetMethod(Method setMethod) {
        this.setMethod = setMethod;
    }

    /**
     * Set the name of the set method for the attribute accessor.
     */
    public void setSetMethodName(String setMethodName) {
        this.setMethodName = setMethodName;
    }
}
