/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the "License").  You may not use this file except 
 * in compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * glassfish/bootstrap/legal/CDDLv1.0.txt or 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html. 
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * HEADER in each file and include the License file at 
 * glassfish/bootstrap/legal/CDDLv1.0.txt.  If applicable, 
 * add the following below this CDDL HEADER, with the 
 * fields enclosed by brackets "[]" replaced with your 
 * own identifying information: Portions Copyright [yyyy] 
 * [name of copyright owner]
 */
// Copyright (c) 1998, 2006, Oracle. All rights reserved.  
package oracle.toplink.essentials.internal.ejb.cmp3.base;

import java.lang.ref.WeakReference;
import java.util.Iterator;
import java.util.Vector;
import java.util.Map;
import oracle.toplink.essentials.threetier.ServerSession;
import oracle.toplink.essentials.tools.sessionmanagement.SessionManager;
import oracle.toplink.essentials.internal.localization.ExceptionLocalization;

/**
* <p>
* <b>Purpose</b>: Provides the implementation for the EntityManager Factory.
* <p>
* <b>Description</b>: This class will store a reference to the active ServerSession.  When a request
* is made for an EntityManager an new EntityManager is created with the ServerSession and returned.
* The primary consumer of these EntityManager is assumed to be either the Container.    There is
* one EntityManagerFactory per deployment.
* @see javax.persistence.EntityManager
* @see oracle.toplink.essentials.ejb.cmp3.EntityManager
* @see oracle.toplink.essentials.ejb.cmp3.EntityManagerFactory
*/

/*  @author  gyorke  
 *  @since   TopLink 10.1.3 EJB 3.0 Preview
 */

public abstract class EntityManagerFactoryImpl {
    // This stores a reference to the ServerSession for this deployement.
    protected ServerSession serverSession;
    protected Vector entityManagers;
    protected boolean isOpen = true;

    protected abstract EntityManagerImpl createEntityManagerImplInternal(Map properties, boolean extended);
    
    /**
     * Will return an instance of the Factory.  Should only be called by TopLink.
     * @param serverSession
     */
    public EntityManagerFactoryImpl(ServerSession serverSession){
        this.serverSession = serverSession;
        entityManagers = new Vector();
    }
    
    /**
     * INTERNAL:
     * Returns the ServerSession that the Factory will be using.
     */
    public ServerSession getServerSession(){
        return this.serverSession;
    }
    
    /**
	 * Closes this factory, releasing any resources that might be held by this factory. After
	 * invoking this method, all methods on the instance will throw an
	 * {@link IllegalStateException}, except for {@link #isOpen}, which will return
	 * <code>false</code>.
	 */
	public synchronized void close(){
        verifyOpen();
        isOpen = false;
        
        Iterator iterator = entityManagers.iterator();
        while (iterator.hasNext()) {
            EntityManagerImpl impl = (EntityManagerImpl)((WeakReference)iterator.next()).get();
            if(impl != null && impl.isOpen()) {
                impl.close();
            }
        }
        if(this.serverSession.isConnected()) {
            this.serverSession.logout();
        }
        SessionManager.getManager().getSessions().remove(serverSession.getName());
    }


	/**
	 * Indicates whether or not this factory is open. Returns <code>true</code> until a call
	 * to {@link #close} is made.
	 */
	public boolean isOpen(){
       return isOpen;
    }

    protected EntityManagerImpl createEntityManagerImpl(boolean extended) {
        return createEntityManagerImpl(null, extended);
    }

    protected synchronized EntityManagerImpl createEntityManagerImpl(Map properties, boolean extended) {
        verifyOpen();

        if (!this.serverSession.isConnected()) {
            this.serverSession.login();
        }
        EntityManagerImpl manager = createEntityManagerImplInternal(properties, extended);
        entityManagers.add(new WeakReference(manager));
        return manager;
    }

    protected void verifyOpen(){
        if (!isOpen){
            throw new IllegalStateException(ExceptionLocalization.buildMessage("operation_on_closed_entity_manager_factory"));
        }
    }    
}
