/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the "License").  You may not use this file except 
 * in compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * glassfish/bootstrap/legal/CDDLv1.0.txt or 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html. 
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * HEADER in each file and include the License file at 
 * glassfish/bootstrap/legal/CDDLv1.0.txt.  If applicable, 
 * add the following below this CDDL HEADER, with the 
 * fields enclosed by brackets "[]" replaced with your 
 * own identifying information: Portions Copyright [yyyy] 
 * [name of copyright owner]
 */
// Copyright (c) 1998, 2006, Oracle. All rights reserved.  
package oracle.toplink.essentials.internal.ejb.cmp3.xml;

import java.net.URL;

import java.util.Set;
import java.util.List;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.ArrayList;
import java.util.Collection;

import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.w3c.dom.Document;

import oracle.toplink.essentials.ejb.cmp3.EJBQuery;

import oracle.toplink.essentials.sessions.Project;
import oracle.toplink.essentials.sessions.Session;

import oracle.toplink.essentials.internal.helper.DatabaseField;
import oracle.toplink.essentials.internal.helper.DatabaseTable;

import oracle.toplink.essentials.descriptors.ClassDescriptor;
import oracle.toplink.essentials.internal.sessions.AbstractSession;

import oracle.toplink.essentials.mappings.OneToOneMapping;
import oracle.toplink.essentials.mappings.CollectionMapping;
import oracle.toplink.essentials.mappings.ManyToManyMapping;
import oracle.toplink.essentials.mappings.DirectToFieldMapping;
import oracle.toplink.essentials.mappings.AggregateObjectMapping;
import oracle.toplink.essentials.mappings.ForeignReferenceMapping;

import oracle.toplink.essentials.queryframework.FieldResult;
import oracle.toplink.essentials.queryframework.ColumnResult;
import oracle.toplink.essentials.queryframework.EntityResult;
import oracle.toplink.essentials.queryframework.SQLResultSetMapping;
import oracle.toplink.essentials.queryframework.EJBQLPlaceHolderQuery;

import oracle.toplink.essentials.internal.ejb.cmp3.EJBQueryImpl;
import oracle.toplink.essentials.internal.ejb.cmp3.metadata.MetadataHelper;
import oracle.toplink.essentials.internal.ejb.cmp3.metadata.MetadataLogger;
import oracle.toplink.essentials.internal.ejb.cmp3.metadata.MetadataAccessor;
import oracle.toplink.essentials.internal.ejb.cmp3.metadata.MetadataConstants;
import oracle.toplink.essentials.internal.ejb.cmp3.metadata.MetadataProcessor;
import oracle.toplink.essentials.internal.ejb.cmp3.metadata.MetadataDescriptor;
import oracle.toplink.essentials.internal.ejb.cmp3.metadata.MetadataJoinColumn;

import oracle.toplink.essentials.internal.ejb.cmp3.xml.sequencing.Id;
import oracle.toplink.essentials.internal.ejb.cmp3.xml.sequencing.TableGenerator;
import oracle.toplink.essentials.internal.ejb.cmp3.xml.sequencing.SequenceGenerator;
import oracle.toplink.essentials.internal.ejb.cmp3.xml.sequencing.SequencingProcessor;

/**
 * INTERNAL:
 * 
 * Process entity-mappings instance documents, with the end result being a 
 * TopLink project.  The project is handed to this class via TopLink session, 
 * and could possibly be partially/fully populated from project XML.  After 
 * processing, the project may/may not be fully complete;  if not, the 
 * EJBAnnotationsProcessor will complete the process. It is assumed here that 
 * an entity declaration does not have to be complete. Mappings, however, must 
 * be fully defined.
 * 
 * @author Dave McCann, Guy Pelletier
 * @since TopLink EJB 3.0 Reference Implementation
 */
public class EntityMappingsXMLProcessor extends MetadataProcessor {
    // Persistence unit default values, set on persistence-unit-default element.
    private String m_defaultAccess = "";
    private String m_defaultCatalog = "";
    private String m_defaultSchema = "";
    
    // Override values for persistence unit defaults - applies to a given 
    // instance document package, catalog and schema can be overridden in 
    // entity-mappings element, and access can be overridden in entity-mappings 
    // and entity elements.
    private String m_access;
    private String m_schema;
    private String m_catalog;
    private String m_package;
    
    private HashMap m_namedQueries;
    private HashMap m_namedNativeQueries;
    
    private Project m_project;
    private Document m_document;
    private XMLHelper m_xmlHelper;
    
    private XPathEngine m_xPathEngine;
    private SequencingProcessor m_sequencingProcessor;

    /**
     * INTERNAL:
     */    
    public EntityMappingsXMLProcessor() {
        this(true);
    }
    
    /**
     * INTERNAL:
     */
    public EntityMappingsXMLProcessor(boolean enableLazyForOneToOne) {
        m_enableLazyForOneToOne = enableLazyForOneToOne;
        m_namedQueries = new HashMap();
        m_namedNativeQueries = new HashMap();
    }

    /**
     * INTERNAL: 
     */
	public void addEntityListeners(Session session) {
        // no longer does anything ... 
	}
	
	/**
     * INTERNAL:
     * If node contains a child tag with name matching eventTag, the associated 
     * method name is stored in the listener for later processing.
     */
    protected void addEventMethodNameToListener(XMLListener listener, Node node, String eventTag) {
        Node result = m_xPathEngine.selectSingleNode(node, new String[] {eventTag});
        if (result != null) {
        	// process @method-name (required)
            listener.addEventMethodName(eventTag, m_xPathEngine.selectSingleNode(result, new String[] {XMLConstants.ATT_METHOD_NAME}).getNodeValue());
        }
    }
    
	/**
     * INTERNAL:
     * If node contains a child tag with name matching eventTag, the associated 
     * method name is stored in the map for later processing.
     */
    protected void addEventMethodNameToMap(HashMap methodMap, Node node, String eventTag) {
        Node result = m_xPathEngine.selectSingleNode(node, new String[] {eventTag});
        if (result != null) {
        	// process @method-name (required)
            methodMap.put(eventTag, m_xPathEngine.selectSingleNode(result, new String[] {XMLConstants.ATT_METHOD_NAME}).getNodeValue());
        }
    }
    
    /**
     * INTERNAL:
     * Add queries to a given session. The XMLDescriptorMetadata objects for 
     * any entities containing named-query or named-native-query definitions 
     * have been flagged and will be processed.  Queries at the entity-mapping 
     * level will be processed here as well.
     */
	public void addNamedQueriesToSession(Session session) {
		HashMap props, hints;
		String queryString;
		
        // for each named-query
        for (Iterator keys = m_namedQueries.keySet().iterator(); keys.hasNext();) {        
            String queryName = (String) keys.next();

            if (session.getQuery(queryName) != null) {
                getLogger().logWarningMessage(MetadataLogger.IGNORE_QUERY, queryName);
                continue;
            }

            try {
            	props = (HashMap) m_namedQueries.get(queryName);
                queryString = (String) props.get(XMLConstants.QUERY);
                hints = (HashMap) props.get(XMLConstants.QUERY_HINT);
                
                // add a new query to the TopLink Session
                session.addEjbqlPlaceHolderQuery(new EJBQLPlaceHolderQuery(queryName, queryString, hints));
            } catch (Exception exception) {
                getValidator().throwErrorProcessingNamedQueryElement(queryName, exception);
            }
        }
        
        // for each named-native-query
        for (Iterator keys = m_namedNativeQueries.keySet().iterator(); keys.hasNext();) {        
            String queryName = (String) keys.next();
            if (session.getQuery(queryName) != null) {
                getLogger().logWarningMessage(MetadataLogger.IGNORE_QUERY, queryName);
                continue;
            }

            try {
                props = (HashMap) m_namedNativeQueries.get(queryName);
            	queryString = (String) props.get(XMLConstants.QUERY);
                hints = (HashMap) props.get(XMLConstants.QUERY_HINT);

                // result-class (takes precidence)
                if (props.containsKey(XMLConstants.ATT_RESULT_CLASS)) {
                    Class resultClass = MetadataHelper.getClassForName(((String) props.get(XMLConstants.ATT_RESULT_CLASS)), m_loader);
                    if (resultClass != void.class) {
                        // add a new query to the TopLink Session
                        session.addQuery(queryName, EJBQueryImpl.buildSQLDatabaseQuery(resultClass, queryString, hints));
                        continue;
                    }
                }
                
                // result-set-mapping
                if (props.containsKey(XMLConstants.ATT_RESULT_SET_MAPPING)) {
                    // add a new query to the TopLink Session
                    session.addQuery(queryName, EJBQueryImpl.buildSQLDatabaseQuery((String) props.get(XMLConstants.ATT_RESULT_SET_MAPPING), queryString, hints));
                    continue;
                }
                
                // at this point we need to set default query values on the session
                session.addQuery(queryName, EJBQueryImpl.buildSQLDatabaseQuery(queryString, hints));
            } catch (Exception exception) {
                getValidator().throwErrorProcessingNamedQueryElement(queryName, exception);
            }
        }
	}

    /**
     * INTERNAL:
     * The class name of each node in the node list will be added to the 
     * provided collection.
     */
    protected static Set buildClassSetForNodeList(Document xmlDoc, String xPath, String defaultPkg) {
    	HashSet classNameSet = new HashSet();
        XPathEngine xPathEngine = XPathEngine.getInstance();
    	NodeList nodes = xPathEngine.selectNodes(xmlDoc, new String[] {XMLConstants.ENTITY_MAPPINGS, xPath});
    	int nodeCount = nodes.getLength();
        
        for (int i = 0; i < nodeCount; i++) {
            // process @class (required)
            classNameSet.add(XMLHelper.getFullyQualifiedClassName(xPathEngine.selectSingleNode(nodes.item(i), new String[] {XMLConstants.ATT_CLASS}).getNodeValue(), defaultPkg));
        }
        
        return classNameSet;
    }
    
    /**
     * INTERNAL:
     * Return a set of class names for each entity, embeddable, or 
     * mapped-superclass found in the xml descriptor instance document.
     */
    public static Set buildClassSet(URL xmlDocumentURL, ClassLoader loader) {
        XPathEngine xPathEngine = XPathEngine.getInstance();
        HashSet classSet = new HashSet();
        Document xmlDoc = XMLHelper.parseDocument(xmlDocumentURL, loader);
        String defaultPkg = "";
        Node node;

        // process package
        if ((node = xPathEngine.selectSingleNode(xmlDoc, new String[] {XMLConstants.ENTITY_MAPPINGS, XMLConstants.PACKAGE, XMLConstants.TEXT})) != null) {
            defaultPkg = node.getNodeValue();
        }

        // handle entities, embeddables and mapped-superclasses
        classSet.addAll(buildClassSetForNodeList(xmlDoc, XMLConstants.ENTITY, defaultPkg));
        classSet.addAll(buildClassSetForNodeList(xmlDoc, XMLConstants.EMBEDDABLE, defaultPkg));
        
        return classSet;
    }
    
    /**
     * INTERNAL:
     */
    protected XMLAccessor createAccessor(Node attributeNode, XMLDescriptorMetadata dmd) {
		// Process @name (required).
        String attributeName = m_xPathEngine.selectSingleNode(attributeNode, new String[] {XMLConstants.ATT_NAME}).getNodeValue();
		
        return new XMLAccessor(attributeName, this, dmd, attributeNode);
    }
    
    /**
     * INTERNAL:
     * Create and populate a SQLResultSetMapping.  This method assumes that 
     * 'node' is a valid sql-result-set-mapping node, i.e. it is non-null and 
     * contains child nodes.
     */
    protected SQLResultSetMapping createSQLResultSetMapping(Node node) {
        // process @name (required)
        SQLResultSetMapping mapping = new SQLResultSetMapping(m_xPathEngine.selectSingleNode(node, new String[] {XMLConstants.ATT_NAME}).getNodeValue());

        // process entity-results
        NodeList entityResultNodes = m_xPathEngine.selectNodes(node, new String[] {XMLConstants.ENTITY_RESULT});
        if (entityResultNodes != null) {
        	int nodeCount = entityResultNodes.getLength();
            
            for (int j = 0; j < nodeCount; j++) {
                Node erNode = entityResultNodes.item(j);

                // process @entity-class (required)
                Node result = m_xPathEngine.selectSingleNode(erNode, new String[] {XMLConstants.ATT_ENTITY_CLASS});
                Class entityClass = MetadataHelper.getClassForName(result.getNodeValue(), m_loader);
                EntityResult eResult = new EntityResult(entityClass.getName());
                
                // process @discriminator-column
                result = m_xPathEngine.selectSingleNode(erNode, new String[] {XMLConstants.ATT_DISCRIMINATOR_COLUMN});
                if (result != null) {
                    eResult.setDiscriminatorColumn(result.getNodeValue());
                }
                
                // process field-results
                NodeList fieldResultNodes = m_xPathEngine.selectNodes(erNode, new String[] {XMLConstants.FIELD_RESULT});
                if (fieldResultNodes != null) {
                	int fieldCount = fieldResultNodes.getLength();
                    
                    for (int k = 0; k < fieldCount; k++) {
                        Node frNode = fieldResultNodes.item(k);
                
                        // process @name (required)
                        String name = m_xPathEngine.selectSingleNode(frNode, new String[] {XMLConstants.ATT_NAME}).getNodeValue();
                        // process @column (required)
                        String column = m_xPathEngine.selectSingleNode(frNode, new String[] {XMLConstants.ATT_COLUMN}).getNodeValue();
                        
                        eResult.addFieldResult(new FieldResult(name, column));
                    }
                }
                
                mapping.addResult(eResult);
            }
        }
        
        // process column-results
        NodeList columnResultList = m_xPathEngine.selectNodes(node, new String[] {XMLConstants.COLUMN_RESULT});
        if (columnResultList != null) {
        	int columnCount = columnResultList.getLength();
            
            for (int j = 0; j < columnCount; j++) {
                // process @name (required)
                mapping.addResult(new ColumnResult(m_xPathEngine.selectSingleNode(columnResultList.item(j), new String[] {XMLConstants.ATT_NAME}).getNodeValue()));
            }
        }

        return mapping;
    }

    /**
     * INTERNAL:
	 * Lazily create a metadata/descriptor for any given class. 
     */
	public XMLDescriptorMetadata getMetadataDescriptor(Class cls) {
		XMLDescriptorMetadata dmd = (XMLDescriptorMetadata) getDescriptorMetadataMap().get(cls);
        
		if (dmd == null) {
            ClassDescriptor descriptorOnProject = MetadataHelper.findDescriptor(m_session.getProject(), cls);
            
            // We don't have a descriptor, check if there is a descriptor for 
            // this class on the project already.
            if (descriptorOnProject != null) {
                // Wrap the existing descriptor into a XMLDescriptorMetadata.
                dmd = new XMLDescriptorMetadata(descriptorOnProject, cls, m_xmlHelper);
            } else {
                // Create a new XMLDescriptorMetadata and add it to the project.
                dmd = new XMLDescriptorMetadata(cls, m_xmlHelper);
                m_session.getProject().addDescriptor(dmd.getDescriptor());
            }
            
            getDescriptorMetadataMap().put(cls, dmd);
        } else {
            // ensure we have the correct helper file ...
            dmd.setHelper(m_xmlHelper);
        }
        
        return dmd;
	}
	
	/**
     * INTERNAL:
     * Return the map containing descriptor metadata objects for any processed
     * entities.  The map is keyed on the entity java class.
     */
    public HashMap getDescriptorMetadataMap() {
        if (m_metadataDescriptors == null) {
        	m_metadataDescriptors = new HashMap();
        }
        
        return m_metadataDescriptors;
    }
    
    /**
     * INTERNAL:
     */
    protected Object[] getCascadeTypes(XMLAccessor accessor) {
    	HashSet cascadeVals = new HashSet();
        if (accessor.hasCascade()) {
        	NodeList cascadeNodes = accessor.getCascade();
        	int nodeCount = cascadeNodes.getLength();
            
			for (int i = 0; i < nodeCount; i++) {
				cascadeVals.add(cascadeNodes.item(i).getLocalName());
            }
        }
        
        return cascadeVals.toArray();
    }

    /**
     * INTERNAL:
     * Helper method to return a field name from a candidate field name and a 
     * default field name.
     */
	protected String getFieldName(String candidateName, String defaultName, String context, Class javaClass) {
		// Check if a candidate was specified otherwise use the default.
        if (candidateName != null && !candidateName.equals("")) {
            return candidateName;
        } else {
            // Log the defaulting field name based on the given context.
            getLogger().logConfigMessage(context, javaClass, defaultName);
            return defaultName;
        }
	}
    
	/**
	 * INTERNAL:
	 */
	protected XMLHelper getHelper() {
		return m_xmlHelper;
	}

    /**
     * INTERNAL:
     * Determine the appropriate type of hintValue based on the hintName, and return
     * the typed value.  Expressions and Calls will not be supported in XML. Warnings 
     * will be thrown for these and any other unsupported values.  
     * 
     * Hints - types:
     * 
     * 	fetchSize - Integer
     * 	referenceClass - class
     * 	cacheusage - Integer
     * 	refresh - Boolean
     * 	lockMode - Integer
     * 	expression - TopLink Expression (not supported)
     * 	timeout - Integer
     * 
     * @param hintName name of hint
     * @param hintValue value of hint
     * @param queryName used for logging warning messages
     * @param entityClass the ownign entity class - used for logging warnings
     * @return appropriate typed value, or null if the type cannot be determined
     */
    protected Object getTypedHintValue(String hintName, String hintValue, String queryName, String entityClass) {
        try {
            if (hintName.equals(EJBQuery.REFERENCE_CLASS)) {
                return MetadataHelper.getClassForName(hintValue, m_loader);
            } else if (hintName.equals(EJBQuery.CACHE_USAGE) || hintName.equals(EJBQuery.CASCADE_LEVEL)) {
                return Integer.valueOf(hintValue);
            } else if (hintName.equals(EJBQuery.REFRESH)) {
                return Boolean.valueOf(hintValue);
            } else if (hintName.equals(EJBQuery.LOCK_MODE)) {
                return Short.valueOf(hintValue);
            } else if (hintName.equals(EJBQuery.EXPRESSION) || hintName.equals(EJBQuery.CALL)) {
                // expression and call are not supported in XML
                getLogger().logWarningMessage(EntityMappingsXMLLogger.IGNORE_QUERY_HINT_UNSUPPORTED_TYPE, hintName, queryName, entityClass);
                return null;
            }
        } catch (ClassCastException exception) {
            // fall through, throw a warning, and return null
        }
        
        getLogger().logWarningMessage(EntityMappingsXMLLogger.IGNORE_QUERY_HINT_UNKNOWN_TYPE, hintName, queryName, entityClass);
        return null;
    }
	
    /** 
     * INTERNAL:
	 * Return the logger used by this processor.
	 */
	public EntityMappingsXMLLogger getLogger() {
		return (EntityMappingsXMLLogger) m_logger;
    }
    
    /**
     * INTERNAL:
	 * Method to return the primary key fields for the given descriptor
     * metadata.
     */
    protected List getPrimaryKeyFields(XMLDescriptorMetadata dmd) {
        List primaryKeyFields = dmd.getPrimaryKeyFields();
        
        // if this is an inheritance subclass, get the primary key fields from the parent
        if (primaryKeyFields.isEmpty() && dmd.isInheritanceSubclass()) {
            primaryKeyFields = getMetadataDescriptor(m_xmlHelper.locateRootEntity(dmd.getJavaClass())).getPrimaryKeyFields();
        }
        
        return primaryKeyFields;
    }
    
    /**
     * INTERNAL:
     * Returns an array primary key join column sub-nodes for a given node.
     */
    protected Object[] getPrimaryKeyJoinColumns(Node node) {
        Object[] pkJoinCols = {};
    	// look for primary-key-join-column(s) elements
        NodeList primaryKeyJoinColumns = m_xPathEngine.selectNodes(node, new String[] {XMLConstants.PK_JOIN_COLUMN});
        
        if (primaryKeyJoinColumns != null) {
        	pkJoinCols = new Object[primaryKeyJoinColumns.getLength()];
        	int columnCount = primaryKeyJoinColumns.getLength();
        	
            for (int i = 0; i < columnCount; i++) {
        		pkJoinCols[i] = primaryKeyJoinColumns.item(i);
        	}
        }
        
    	return pkJoinCols;
    }
    
    /**
     * INTERNAL:
     * Method to return the name of the join table.
     */
    protected DatabaseTable getRelationTable(XMLAccessor accessor) {
        DatabaseTable relationTable;
        Node joinTableNode = accessor.getJoinTableNode();
        
        if (joinTableNode != null) {
            String tableName = "";
            String catalogName = m_catalog;
            String schemaName = m_schema;

            Node attr;
            // process @name
            if ((attr = m_xPathEngine.selectSingleNode(joinTableNode, new String[] {XMLConstants.ATT_NAME})) != null) {
                tableName = attr.getNodeValue();
            }
        
            // process @schema
            if ((attr = m_xPathEngine.selectSingleNode(joinTableNode, new String[] {XMLConstants.ATT_SCHEMA})) != null) {
                schemaName = attr.getNodeValue();
            }
        
            // process @catalog
            if ((attr = m_xPathEngine.selectSingleNode(joinTableNode, new String[] {XMLConstants.ATT_CATALOG})) != null) {
                catalogName = attr.getNodeValue();
            }
            
    	    relationTable = buildJoinTable(tableName, catalogName, schemaName, accessor);
            
            // process unique-constraints
            processUniqueConstraints(getUniqueConstraints(joinTableNode), relationTable);
        } else {
    	    relationTable = buildJoinTable(accessor);
        }
        
        return relationTable;
    }
     
   /**
     * INTERNAL:
     * Process the table information contained within a table-generator type.
     */
    protected TableInfo getTableInfo(Node generatorNode) {
        // set default values
        String tableName = "";
        String catalogName = m_catalog;
        String schemaName = m_schema;
        Object[] uniqueConstraints = {};
        TableInfo tableInfo;

        if (generatorNode != null) {
            Node attr;
            // process @table
            if ((attr = m_xPathEngine.selectSingleNode(generatorNode, new String[] {XMLConstants.ATT_TABLE})) != null) {
                tableName = attr.getNodeValue();
            }
        
            // process @schema
            if ((attr = m_xPathEngine.selectSingleNode(generatorNode, new String[] {XMLConstants.ATT_SCHEMA})) != null) {
                schemaName = attr.getNodeValue();
            }
        
            // process @catalog
            if ((attr = m_xPathEngine.selectSingleNode(generatorNode, new String[] {XMLConstants.ATT_CATALOG})) != null) {
                catalogName = attr.getNodeValue();
            }
        
            // process unique-constraint
            uniqueConstraints = getUniqueConstraints(generatorNode);
            
            ArrayList constraintsList = new ArrayList();
            for (int i = 0; i < uniqueConstraints.length; i++) {
            	constraintsList.add(uniqueConstraints[i]);
            }

            tableInfo = new TableInfo(tableName, catalogName, schemaName, constraintsList);
        } else {
            tableInfo = new TableInfo();
            tableInfo.isSpecified(false);
        }
        
        return tableInfo;
    }
    
    /**
     * INTERNAL:
     * Initialize data members - this is done such that subsequent calls to the
     * processor don't require a new instance of the class.  A DOM is also 
     * constructed from the instance document indicated by xmlDocumentURL.
     * 
     * @param session TopLink session containing a TopLink project to which 
     * persistence information (contained within the provided persistence ORM 
     * instance document)
     * @param xmlDocumentURL the URL of the instance document to be processed
     * @param classLoader the class loader to be used
     */
    protected void init(AbstractSession session, URL xmlDocumentURL) {
        // initialize data members
        m_project = session.getProject();
        m_xPathEngine = XPathEngine.getInstance();
        m_session = session;
        m_sequencingProcessor = new SequencingProcessor();
        m_logger = new EntityMappingsXMLLogger(session);
        m_validator = new XMLValidator();
        m_xmlHelper = new XMLHelper(xmlDocumentURL, m_loader);
        m_relatedEntities = new HashSet();
        
        // set access, schema and catalog to persistence unit defaults
        m_access = m_defaultAccess;
        m_schema = m_defaultSchema;
        m_catalog = m_defaultCatalog;
        m_package = m_xmlHelper.getPackage();
        
        // create a DOM from the instance document
        m_document = m_xmlHelper.getDocument();
    }

    /**
     * INTERNAL:
     * Process the unique-constraints for a given table node.
     */
    protected Object[] getUniqueConstraints(Node tableNode) {
    	Object[] uniqueConstraints = {};
        
        NodeList constraints = m_xPathEngine.selectNodes(tableNode, new String[] {XMLConstants.UNIQUE_CONSTRAINTS});
        if (constraints != null) {
        	int constraintCount = constraints.getLength();
        	uniqueConstraints = new Object[constraintCount];
            
            // for each unique-constraint...
            for (int j = 0; j < constraintCount; j++) {
                Node constraintNode = constraints.item(j);
                // process column-name(s)
                NodeList columnNameNodeList = m_xPathEngine.selectNodes(constraintNode, new String[] {XMLConstants.COLUMN_NAME, XMLConstants.TEXT});
                if (columnNameNodeList != null) {
                    // for each column-name
                	int columnCount = columnNameNodeList.getLength();
                	String[] constraintArray = new String[columnCount];
                    
                    for (int k = 0; k < columnCount; k++) {
                        String columnName = columnNameNodeList.item(k).getNodeValue();
                        
                        if (columnName != null && !columnName.equals("")) {
                        	constraintArray[k] = columnName;
                        }
                    }
                    
                    uniqueConstraints[j] = constraintArray;
                }
            }
        }
        
        return uniqueConstraints;   
    }
    
    /** 
     * INTERNAL:
	 * Return the xml validator.
	 */
	protected XMLValidator getValidator() {
        return (XMLValidator) m_validator;
    }
    
    /**
     * INTERNAL:
     */
    protected void handlePotentialDefaultPrimaryKeyUsage(DatabaseField dbField, MetadataAccessor accessor, String defaultName, String context) {
    	// handle join column vs inverse join column
    	XMLDescriptorMetadata md;
    	XMLDescriptorMetadata referenceMd;
    	if (context.equals(MetadataLogger.SOURCE_PK_COLUMN) || context.equals(MetadataLogger.SOURCE_FK_COLUMN)) {
    		// join column
        	md = (XMLDescriptorMetadata) accessor.getMetadataDescriptor();
        	referenceMd = (XMLDescriptorMetadata) accessor.getReferenceMetadataDescriptor();
    	} else {
    		// inverse join column
        	md = (XMLDescriptorMetadata) accessor.getReferenceMetadataDescriptor();
        	referenceMd = (XMLDescriptorMetadata) accessor.getMetadataDescriptor();
    	}
    	
    	// Case 1:  straight up pk field name, i.e. defaultName == descriptor/reference 
    	// descriptor primary-key field name
    	if (defaultName.equals(md.getPrimaryKeyFieldName())) {
       		md.handleDefaultPrimaryKeyUsage(dbField);
    	} else if (defaultName.equals(referenceMd.getPrimaryKeyFieldName())) {
    		referenceMd.handleDefaultPrimaryKeyUsage(dbField);
    	} else {
        	// Case 2:  complex, i.e. defaultName == attributeName + "_" + reference 
        	// descriptor primary-key field name
    		referenceMd.handleComplexDefaultPrimaryKeyUsage(dbField);
    	}
    }

    /**
     * INTERNAL
     */
    protected void handlePotentialDefaultPrimaryKeyUsage(DatabaseField dbField, MetadataDescriptor md) {
    	((XMLDescriptorMetadata)md).handleDefaultPrimaryKeyUsage(dbField);
    }
    
    /**
     * INTERNAL:
     * Handle potential partial mapping definition in XML (i.e. target-entity 
     * class is not defined in the instance document)
     *  - if there is more than one join column, the mapping can be completed 
     *    w/o annotations, as they must be fully defined in the composite case
     *  - if there is one join column, and either the fk/pk field is empty, we 
     *    can't finish w/o annotations
     *  - if we need annotations to complete the mapping, make sure to store the 
     *    fk/pk field name (if defined in XML), otherwise the default will be 
     *    set during annotations processing
     */
    protected boolean handlePotentialPartialOneToOneRelationshipMapping(OneToOneMapping mapping, List joinColumns) {
        if (mapping.requiresCompletion()) {
            if (joinColumns.size() == 1) {
                MetadataJoinColumn mjc = (MetadataJoinColumn) joinColumns.get(0); 
                if (mjc.getForeignKeyField().getName().equals("")) {
	        		// At this point, we need the annotations processor to 
                    // default the fk and possibly the pk.
	        		// Store the pk - "" indicates that annotations processor 
                    // should default
        			mapping.setXMLPKNameAttribute(mjc.getPrimaryKeyField().getName());
	        		return true;
	        	}
	        
            	if (mjc.getPrimaryKeyField().getName().equals("")) {
	        		// At this point, we have a fk, but need the annotations 
                    // processor to default the pk.
	        		// Store the fk - "" indicates that annotations processor 
                    // should default
        			mapping.setXMLFKNameAttribute(mjc.getForeignKeyField().getName());
	        		return true;
	        	}
        	}
            
        	// At this point, we have enough information to complete the mapping.
        	mapping.setRequiresCompletion(false);
        }
        
        return false;
    }
    
    /**
     * INTERNAL:
     * Handle potential partial relationship mapping definition in XML. A single 
     * inverse join column (i.e. not a composite pk scenario) may not be fully 
     * defined, and require the annotations processor to provide default 
     * value(s).
     */
    protected boolean handlePotentialPartialManyToManyRelationshipMapping(ManyToManyMapping mapping, boolean isSource, List relationKeys) {
        if (!isSource && mapping.requiresCompletion()) {
            if (relationKeys.size() == 1) {
                MetadataJoinColumn mjc = (MetadataJoinColumn) relationKeys.get(0); 
                if (mjc.getForeignKeyField().getName().equals("")) {
                    // At this point, we need the annotations processor to 
                    // default the fk and possibly the pk.
                    // Store the pk - "" indicates that annotations processor 
                    // should default
                    mapping.setXMLPKNameAttribute(mjc.getPrimaryKeyField().getName());
                    return true;
                }
                
                if (mjc.getPrimaryKeyField().getName().equals("")) {
                    // At this point, we have a fk, but need the annotations 
                    // processor to default the pk. Store the fk - "" indicates 
                    // that annotations processor should default.
                    mapping.setXMLFKNameAttribute(mjc.getForeignKeyField().getName());
                    return true;
                }
            }
            
            // At this point, we have enough information to complete the mapping.
            mapping.setRequiresCompletion(false);
        }
        
        return false;
    }

    /**
     * INTERNAL:
     */
	protected boolean hasPrimaryKeyJoinColumn(Node node) {
		return m_xPathEngine.selectNodes(node, new String[] {XMLConstants.PK_JOIN_COLUMN}).getLength()>0;
	}    
	
    /**
     * INTERNAL:
     * Locate a node in the DOM tree for a given class.  
     * 
     * @param className the name of the class to search the DOM for
     * @param searchString the node name to search for, i.e. ENTITY or 
     * MAPPED_SUPERCLASS
     */
	protected Node locateNode(Class cls, String searchString) {
		return m_xmlHelper.locateNode(cls, searchString);
	}
    	
    /**
     * INTERNAL:
     * Process the attributes of an entity or embeddable.
     */
    protected void processAccessors(MetadataDescriptor dmd) {
    	processAttributes((XMLDescriptorMetadata) dmd, m_xmlHelper.locateNode(dmd.getJavaClass()));
    }
    
    /**
     * INTERNAL:
     * Satisfy the abstract method declaration in MetadataProcessor.
     */
    protected void processAssociationOverrides(MetadataAccessor accessor, AggregateObjectMapping mapping) {
        return;
    }
    
    /**
     * INTERNAL:
     * Process association-override info.
     */
    protected void processAssociationOverrides(XMLDescriptorMetadata dmd, Node entityNode) {
    	NodeList nodes = m_xPathEngine.selectNodes(entityNode, new String[] {XMLConstants.ASSOCIATION_OVERRIDE});
    	if (nodes != null) {
    		int nodeCount = nodes.getLength();
            
    		for (int i = 0; i < nodeCount; i++) {
    			Node overrideNode = nodes.item(i);
    			// process @name (required)
    			String name = m_xPathEngine.selectSingleNode(overrideNode, new String[] {XMLConstants.ATT_NAME}).getNodeValue();
    			
    	        // process join-columns
    	        NodeList jCols = m_xPathEngine.selectNodes(overrideNode, new String[] {XMLConstants.JOIN_COLUMN});
    	        if (jCols != null && jCols.getLength()>0) {
                    dmd.addAssociationOverride(name, new Object[] {jCols});
    	        }            	
    		}
    	}
    }
    
    /**
     * INTERNAL:
     * Process the attributes of an entity, embeddable or mapped-superclass.
     */
    protected void processAccessors(Class cls, XMLDescriptorMetadata dmd) {
    	processAttributes(dmd, m_xmlHelper.locateNode(cls));
    }

    /**
     * INTERNAL:
     */
    protected void processAttributeOverrides(MetadataAccessor metadataAccessor, AggregateObjectMapping mapping) {
    	XMLAccessor accessor = (XMLAccessor) metadataAccessor;
    	if (accessor.hasAttributeOverrides()) {
    		NodeList overrides = accessor.getAttributeOverrideNodes();
    		int overrideCount = overrides.getLength();
            
    		for (int i = 0; i < overrideCount; i++) {
    			Node overrideNode = overrides.item(i);
                // process @name (required)
                String overrideName = m_xPathEngine.selectSingleNode(overrideNode, new String[] {XMLConstants.ATT_NAME}).getNodeValue();
                // process column
                Node columnNode = m_xPathEngine.selectSingleNode(overrideNode, new String[] {XMLConstants.COLUMN});
                processAttributeOverride(overrideName, columnNode, accessor, mapping);
    		}
    	}
    }
    
    /**
     * INTERNAL:
	 * Process the attribute overrides for an entity that inherits from an 
     * abstract entity.  These overrides are at the entity level, and not the
     * attribute level.
	 */
    protected void processAttributeOverrides(XMLDescriptorMetadata dmd) {
    	processAttributeOverrides(dmd.getJavaClass(), dmd);
    }

    /**
     * INTERNAL:
	 * Process the attribute overrides for an entity that inherits from an 
     * abstract entity.  These overrides are at the entity level, and not the
     * attribute level.
	 */
    protected void processAttributeOverrides(Class cls, XMLDescriptorMetadata dmd) {
        Node entityNode = m_xmlHelper.locateNodeForEntity(cls);

        NodeList nodes = m_xPathEngine.selectNodes(entityNode, new String[] {XMLConstants.ATTRIBUTE_OVERRIDE});
        if (nodes != null) {
        	int nodeCount = nodes.getLength();
            
            for (int i = 0; i < nodeCount; i++) {
                Node node = nodes.item(i);
                
                // process @name (required)
                String attributeName = m_xPathEngine.selectSingleNode(node, new String[] {XMLConstants.ATT_NAME}).getNodeValue();
                
                // process column
                Node columnNode = m_xPathEngine.selectSingleNode(node, new String[] {XMLConstants.COLUMN});
                if (columnNode != null) {
                    DatabaseField dbField = processColumn(columnNode, attributeName, dmd, dmd.getJavaClass());
                    dmd.addAttributeOverride(attributeName, dbField);
                } 
            }
        }
    }
    
    /**
     * INTERNAL:
     * Method to init a collection mapping from a one-to-many. If we get this
     * far, it means we actually have a one-to-many specified, therefore, we
     * don't need to check against null.
     */
    protected void populateCollectionMapping(CollectionMapping mapping, Object oneToManyNode, MetadataAccessor metadataAccessor) {
    	//Node oneToMany = (Node) oneToManyNode;
    	XMLAccessor accessor = (XMLAccessor) metadataAccessor;
        
        populateCollectionMapping(mapping, 
                                  accessor, 
                                  accessor.getTargetEntity(), 
                                  getCascadeTypes(accessor),
                                  accessor.usesIndirection());
    }
    
    /**
     * INTERNAL:
     */
    protected void preProcessGeneratedValue(Object generatedValueNode, MetadataAccessor accessor, DatabaseField field, MetadataDescriptor dmd) {
        String strategy = MetadataConstants.AUTO;
        String generator = "";
    	Node result;

    	// process @strategy
        if ((result = m_xPathEngine.selectSingleNode((Node) generatedValueNode, new String[] {XMLConstants.ATT_STRATEGY})) != null) {
            strategy = result.getNodeValue();
        }
        
        // process @generator
        if ((result = m_xPathEngine.selectSingleNode((Node) generatedValueNode, new String[] {XMLConstants.ATT_GENERATOR})) != null) {
            generator = result.getNodeValue();
        }

        // pre-process the id for the sequencing
        m_sequencingProcessor.preProcessId(new Id(strategy, generator), field, dmd.getDescriptor());
    }

    /**
     * INTERNAL:
	 * Pre-process sequencing.  Gather the necessary info. The real sequencing 
     * setup is performed after all the sequencing-related info on all classes 
     * is preprocessed.
     * 
     * @param node either an entity or entity-mappings node
     */
    protected void preProcessSequenceGenerator(Node node) {
        int initialValue = 0;
        int allocationSize = 50;

        // process @name (required)
        String name = m_xPathEngine.selectSingleNode(node, new String[] {XMLConstants.ATT_NAME}).getNodeValue();
        
        // process @sequence-name
        String sequenceName = name;
        Node result = m_xPathEngine.selectSingleNode(node, new String[] {XMLConstants.ATT_SEQUENCE_NAME});
        if (result != null) {
            sequenceName = result.getNodeValue();
        }
        
        // process @initial-value
        if ((result = m_xPathEngine.selectSingleNode(node, new String[] {XMLConstants.ATT_INITIAL_VALUE})) != null) {
            initialValue = Integer.parseInt(result.getNodeValue());
        }
        
        // process @allocation-size
        if ((result = m_xPathEngine.selectSingleNode(node, new String[] {XMLConstants.ATT_ALLOCATION_SIZE})) != null) {
            allocationSize = Integer.parseInt(result.getNodeValue());
        }

        m_sequencingProcessor.preProcessSequenceGenerator(new SequenceGenerator(name, sequenceName, initialValue, allocationSize));
    }
    
    /**
     * INTERNAL:
     * Pre-process any sequencing. Gather the necessary info. The real 
     * sequencing setup is performed after all the sequencing-related info on 
     * all classes is pre-processed.
     */
    protected void preProcessSequencing(MetadataAccessor metadataAccessor) {
    	XMLAccessor accessor = (XMLAccessor) metadataAccessor;
    	Node generator;
    	// process table-generator
        if ((generator = accessor.getTableGeneratorNode()) != null) {
            preProcessTableGenerator(generator);
        }
        
        // process sequence-generator
        if ((generator = accessor.getSequenceGeneratorNode()) != null) {
            preProcessSequenceGenerator(generator);
        }
    } 
    
    /**
     * INTERNAL:
	 * Pre-process sequencing. Gather the necessary info. The real sequencing 
     * setup is performed after all the sequencing-related info on all classes 
     * is pre-processed.
     */
    protected void preProcessTableGenerator(Node node) {
        // set defaults
        String pkColumnName = "";
        String valueColumnName = "";
        String pkColumnValue = "";
        int initialValue = 0;
        int allocationSize = 50;
        Node result;

        // process @name (required)    
        String name = m_xPathEngine.selectSingleNode(node, new String[] {XMLConstants.ATT_NAME}).getNodeValue();
        
        // process @pk-column-name
        if ((result = m_xPathEngine.selectSingleNode(node, new String[] {XMLConstants.ATT_PK_COLUMN_NAME})) != null) {
            pkColumnName = result.getNodeValue();
        }
        
        // process @value-column-name
        if ((result = m_xPathEngine.selectSingleNode(node, new String[] {XMLConstants.ATT_VALUE_COLUMN_NAME})) != null) {
            valueColumnName = result.getNodeValue();
        }
        
        // process @pk-column-value
        if ((result = m_xPathEngine.selectSingleNode(node, new String[] {XMLConstants.ATT_PK_COLUMN_VALUE})) != null) {
            pkColumnValue = result.getNodeValue();
        }
        
        // process @initial-value
        if ((result = m_xPathEngine.selectSingleNode(node, new String[] {XMLConstants.ATT_INITIAL_VALUE})) != null) {
            initialValue = Integer.parseInt(result.getNodeValue());
        }
        
        // process @allocation-size
        if ((result = m_xPathEngine.selectSingleNode(node, new String[] {XMLConstants.ATT_ALLOCATION_SIZE})) != null) {
            allocationSize = Integer.parseInt(result.getNodeValue());
        }

        TableInfo tableInfo = getTableInfo(node);        
        m_sequencingProcessor.preProcessTableGenerator(new TableGenerator(name, pkColumnName, valueColumnName, pkColumnValue, tableInfo, initialValue, allocationSize));
    }
   
    /**
     * INTERNAL:
     * Process attributes.
     * 
     * @param dmd the XMLDescriptorMetadata object of the owning entity
     * @param node the DOM node representing the owning entity
     */
    protected void processAttributes(XMLDescriptorMetadata dmd, Node node) {
        NodeList attributeNodes = m_xPathEngine.selectNodes(node, new String[] {XMLConstants.ATTRIBUTES, XMLConstants.ALL_CHILDREN});
        if (attributeNodes != null) {
	        // for each attribute...
        	int nodeCount = attributeNodes.getLength();
            
	        for (int i = 0; i < nodeCount; i++) {
	        	processAccessor(createAccessor(attributeNodes.item(i), dmd));
	        }
        }
        
        dmd.setIsProcessed(true);
    }

    /**
     * INTERNAL:
     * Process a basic element.
     */
    protected void processBasic(MetadataAccessor metadataAccessor, DirectToFieldMapping mapping) {
    	XMLAccessor accessor = (XMLAccessor) metadataAccessor;
    	if (accessor.isBasic()) {
	    	// handle @fetch
	    	if (accessor.usesIndirection()) {
	            getLogger().logWarningMessage(MetadataLogger.IGNORE_BASIC_FETCH_LAZY, accessor);
	        }
            
	        // handle @optional
	    	mapping.setIsOptional(accessor.isOptional());
    	}
    }
    
    /**
     * INTERNAL:
     * Process the cascade type on a mapping.
     */
    protected void processCascadeType(MetadataAccessor accessor, Object[] cTypes, ForeignReferenceMapping mapping) {
        for (int i = 0; i < cTypes.length; i++) {
            setCascadeType((String) cTypes[i], mapping);
        }
        
        // apply the persistence unit default cascade-persist if necessary
        if (accessor.getMetadataDescriptor().isCascadePersistSet() && !mapping.isCascadePersist()) {
        	setCascadeType(XMLConstants.CASCADE_PERSIST, mapping);
        }
    }
    
    /**
     * INTERNAL:
     */
    protected DatabaseField processColumn(Object columnNode, Object annotatedElement, String attributeName, MetadataDescriptor dmd) {
    	if (columnNode != null) {
        	return processColumn((Node) columnNode, attributeName, (XMLDescriptorMetadata) dmd, annotatedElement);
    	}
    	
    	return processColumnDefaults(attributeName, dmd, annotatedElement);
    }
    
    /**
     * INTERNAL:
     */
    protected DatabaseField processColumn(MetadataAccessor metadataAccessor) {
    	XMLAccessor accessor = (XMLAccessor) metadataAccessor;
    	if (accessor.hasColumn()) {
    		return processColumn(accessor.getColumnNode(), accessor.getAttributeName(), accessor.getMetadataDescriptor(), accessor.getAnnotatedElement());
    	}
    	
    	return processColumnDefaults(accessor);
    }
    
    /**
     * INTERNAL:
     * Process a column in a DatabaseField and return it.
     */
    protected DatabaseField processColumn(Node columnNode, String attributeName, XMLDescriptorMetadata xmlDmd, Object element) {
        String fieldName = "";
        String columnDefinition = "";
        String tableName = "";
        boolean isUnique = false;
        boolean isNullable = true;
        boolean isInsertable = true;
        boolean isUpdatable = true;
        int length = 255;
        int precision = 0;
        int scale = 0;
        Node result;
        
        // process @name
        if ((result = m_xPathEngine.selectSingleNode(columnNode, new String[] {XMLConstants.ATT_NAME})) != null) {
            fieldName = result.getNodeValue();
        }
        
        // process @table
        if ((result = m_xPathEngine.selectSingleNode(columnNode, new String[] {XMLConstants.ATT_TABLE})) != null) {
            tableName = result.getNodeValue();
        }
        
        // process @insertable
        if ((result = m_xPathEngine.selectSingleNode(columnNode, new String[] {XMLConstants.ATT_INSERTABLE})) != null) {
            isInsertable = Boolean.parseBoolean(result.getNodeValue());
        }
        
        // process @updatable
        if ((result = m_xPathEngine.selectSingleNode(columnNode, new String[] {XMLConstants.ATT_UPDATABLE})) != null) {
            isUpdatable = Boolean.parseBoolean(result.getNodeValue());
        }
        
        // process @unique
        if ((result = m_xPathEngine.selectSingleNode(columnNode, new String[] {XMLConstants.ATT_UNIQUE})) != null) {
            isUnique = Boolean.parseBoolean(result.getNodeValue());
        }
        
        // process @nullable
        if ((result = m_xPathEngine.selectSingleNode(columnNode, new String[] {XMLConstants.ATT_NULLABLE})) != null) {
            isNullable = Boolean.parseBoolean(result.getNodeValue());
        }
        
        // process @column-definition
        if ((result = m_xPathEngine.selectSingleNode(columnNode, new String[] {XMLConstants.ATT_COLUMN_DEFINITION})) != null) {
            columnDefinition = result.getNodeValue();
        }
        
        // process @length
        if ((result = m_xPathEngine.selectSingleNode(columnNode, new String[] {XMLConstants.ATT_LENGTH})) != null) {
            length = Integer.parseInt(result.getNodeValue());
        }
        
        // process @precision
        if ((result = m_xPathEngine.selectSingleNode(columnNode, new String[] {XMLConstants.ATT_PRECISION})) != null) {
            precision = Integer.parseInt(result.getNodeValue());
        }
        
        // process @scale
        if ((result = m_xPathEngine.selectSingleNode(columnNode, new String[] {XMLConstants.ATT_SCALE})) != null) {
            scale = Integer.parseInt(result.getNodeValue());
        }

        return processColumn(attributeName, fieldName, columnDefinition, tableName, isUnique, isNullable, isInsertable, isUpdatable, length, precision, scale, element, xmlDmd);
	}
    
    /**
	 * INTERNAL:
	 * Set a default primary key for the given metadata descriptor.
	 */
	protected void processDefaultId(MetadataDescriptor metadataDescriptor) {
		// Flag the metadata descriptor to indicate default primary key field 
        // usage.
		metadataDescriptor.setDefaultPrimaryKey();
        
		DatabaseField pkField = new DatabaseField();
        pkField.setTableName(metadataDescriptor.getPrimaryTableName());
        
        // Flag the metadata descriptor if a default table is being used.
        metadataDescriptor.handlePotentialDefaultTableUsage(pkField);
        pkField.setName(XMLConstants.DEFAULT_PKFIELD_NAME);
    
        // Set the field as the primary key. This will be rectified during
        // annotation processing.
		metadataDescriptor.addPrimaryKeyField(pkField);
	}
    
    /**
     * INTERNAL
     * Process default entity listeners and add them to the descriptor metadata.
     */
    protected void processDefaultListeners(List defaultListeners, HashMap defaultListenerEventMethods, XMLDescriptorMetadata xmlDmd) {
        String entityClassName = xmlDmd.getJavaClassName();
        Iterator keys = defaultListeners.iterator();
        
	    while (keys.hasNext()) {
            String listenerClassName = (String) keys.next();
            XMLEntityListener listener = new XMLEntityListener(listenerClassName, entityClassName);
            listener.getEventMethodNames().putAll((HashMap) defaultListenerEventMethods.get(listenerClassName));
            xmlDmd.addDefaultEventListener(listener);
		}
    }
                
    /**
	 * INTERNAL:
	 * Create and return a default database table object for the given
     * metadata descriptor.
	 */
	protected void processDefaultTable(MetadataDescriptor md) {
		md.setDefaultPrimaryTable();
        super.processDefaultTable(md);
	}
	
    /**
     * INTERNAL:
     * Process discriminator-column information.
     */
    protected void processDiscriminatorColumn(Node entityNode, XMLDescriptorMetadata xmlDmd) {
        int length = 31;
        String columnName = "";
        String columnDefinition = "";
        String discriminatorType = MetadataConstants.STRING;
        
        Node result;
        if ((result = m_xPathEngine.selectSingleNode(entityNode, new String[] {XMLConstants.DISCRIMINATOR_COLUMN})) != null) {
            Node discriminatorResult;
                            
            // Process @discriminator-type
            if ((discriminatorResult = m_xPathEngine.selectSingleNode(result, new String[] {XMLConstants.ATT_DISCRIMINATOR_TYPE})) != null) {
               discriminatorType = discriminatorResult.getNodeValue();
            }
            
            // Process @name
            if ((discriminatorResult = m_xPathEngine.selectSingleNode(result, new String[] {XMLConstants.ATT_NAME})) != null) {
                columnName = discriminatorResult.getNodeValue();
            }
            
            // Process @column-definition
            if ((discriminatorResult = m_xPathEngine.selectSingleNode(result, new String[] {XMLConstants.ATT_COLUMN_DEFINITION})) != null) {
                columnDefinition = discriminatorResult.getNodeValue();
            }
            
            // Process @length
            if ((discriminatorResult = m_xPathEngine.selectSingleNode(result, new String[] {XMLConstants.ATT_LENGTH})) != null) {
                length = Integer.parseInt(discriminatorResult.getNodeValue());
            }
        }
        
        processDiscriminatorColumn(columnName, columnDefinition, length, discriminatorType, xmlDmd);
    }

    /**
     * INTERNAL:
     */
    protected void processDiscriminatorValue(MetadataDescriptor metadataDescriptor) {
        processDiscriminatorValue(m_xmlHelper.locateNodeForEntity(metadataDescriptor.getJavaClass()), (XMLDescriptorMetadata) metadataDescriptor);
	}
	
    /**
     * INTERNAL:
	 * Process a discriminator value to set the class indicator on the root 
     * descriptor of the inheritance hierarchy. 
     * 
     * If there is no discriminator value, the class indicator defaults to the 
     * class name.
	 */
    protected void processDiscriminatorValue(Node entityNode, XMLDescriptorMetadata xmlDmd) {
	    String dValue = null;
	    Node result = m_xPathEngine.selectSingleNode(entityNode, new String[] {XMLConstants.DISCRIMINATOR_VALUE, XMLConstants.TEXT});
        
	    if (result != null && !result.getNodeValue().equals("")) {
	        dValue = result.getNodeValue();
	    }

	    processDiscriminatorValue(dValue, xmlDmd);
	}
    
    /**
     * INTERNAL:
     */
    protected void processEnumerated(MetadataAccessor accessor, DirectToFieldMapping mapping) {
    	Node enumeratedNode = ((XMLAccessor) accessor).getEnumeratedNode();
        boolean isOrdinal = true;   // default
        String enumeratedType = enumeratedNode.getNodeValue();
       
        if (enumeratedType != null && !enumeratedType.equals("")) {
        	isOrdinal = enumeratedType.equals(XMLConstants.ORDINAL);
        }
        
    	processEnumerated(accessor, isOrdinal, mapping);
    }
    
    /**
     * INTERNAL:
     * Process any entity definitions in the entity-mappings instance document.
     */
    protected void processEntities() {
        NodeList nodes = m_xPathEngine.selectNodes(m_document, new String[] {XMLConstants.ENTITY_MAPPINGS, XMLConstants.ENTITY});
        if (nodes != null) {
            // for each entity in the instance doc...
        	int nodeCount = nodes.getLength();
            
            for (int i = 0; i < nodeCount; i++) {
                processEntityNode(nodes.item(i));
            }
        }
    }
	
	/**
     * INTERNAL:
     * Process an entity, embeddable or mapped-superclass.
     */
    protected void processEntityClass(Class cls) {
    	processEntityNode(m_xmlHelper.locateNode(cls));
    }
    
    /**
     * INTERNAL:
     * Process a given entity.
     * 
     * @param node the node in the DOM representing the entity being processed
     */
    protected void processEntityNode(Node node) {    
        // process @class (required)
    	String entityClassName = m_xmlHelper.getClassNameForNode(node);
    	Class entityClass = MetadataHelper.getClassForName(entityClassName, m_loader);
        XMLDescriptorMetadata xmlDmd = getMetadataDescriptor(entityClass);
        
        // avoid double-processing of entities
        if (! xmlDmd.isProcessed()) {
            xmlDmd.setIgnoreFlags();

            // process @name
            String alias = "";
            Node attr = m_xPathEngine.selectSingleNode(node, new String[] {XMLConstants.ATT_NAME});
            if (attr != null) {
                alias = attr.getNodeValue();
            }
        
            processEntity(alias, xmlDmd);

            // process @metadata-complete
            if (!xmlDmd.shouldIgnoreAnnotations()) {
                if ((attr = m_xPathEngine.selectSingleNode(node, new String[] {XMLConstants.ATT_METADATA_COMPLETE})) != null) {
                    xmlDmd.setShouldIgnoreAnnotations(Boolean.parseBoolean(attr.getNodeValue()));
                }
            }
		
            // process persistence unit default access override
            // process @access
            if ((attr = m_xPathEngine.selectSingleNode(node, new String[] {XMLConstants.ATT_ACCESS})) != null) {
                // use entity override value
                m_access = attr.getNodeValue();
            }
        
            // apply the access, catalog, schema, and package defaults to the descriptor metadata
            if (m_access.equals(XMLConstants.FIELD)) {
                xmlDmd.setIsXmlFieldAccess();
            } else if (m_access.equals(XMLConstants.PROPERTY)) {
                xmlDmd.setIsXmlPropertyAccess();
            }
        
            xmlDmd.setCatalog(m_catalog);
            xmlDmd.setSchema(m_schema);
            xmlDmd.setUsesPropertyAccess(m_access.equals(XMLConstants.PROPERTY));
        
            // process named-query
            processNamedQueries(node, new String[] {XMLConstants.NAMED_QUERY}, entityClassName);

            // process named-native-query
            processNamedNativeQueries(node, new String[] {XMLConstants.NAMED_NATIVE_QUERY}, entityClassName);
        
            // process sql-result-set-mappings
            processSQLResultSetMappings(node, new String[] {XMLConstants.SQL_RESULT_SET_MAPPING});

            Node result;
            // process table-generator
            if ((result = m_xPathEngine.selectSingleNode(node, new String[] {XMLConstants.TABLE_GENERATOR})) != null) {
                preProcessTableGenerator(result);
            }

            // process sequence-generator
            if ((result = m_xPathEngine.selectSingleNode(node, new String[] {XMLConstants.SEQUENCE_GENERATOR})) != null) {
                preProcessSequenceGenerator(result);
            }

            // process table
            processTable(xmlDmd, node);

            // process id-class (pkClass)
            processIdClass(xmlDmd);
        
            // process association overrides
            processAssociationOverrides(xmlDmd, node);
        
            // set our attribute overrides from the entity class before 
            // processing the accessors from the embeddable superclass
            processAttributeOverrides(xmlDmd);

            // process mapped-superclass
            processMappedSuperclasses(xmlDmd);

            // process attributes
            processAttributes(xmlDmd, node);
        
            // If this descriptor has a composite primary key, check that all 
            // our composite primary key attributes were validated. 
            if (xmlDmd.hasCompositePrimaryKey()) {
                if (xmlDmd.pkClassWasNotValidated()) {
                    getValidator().throwInvalidCompositePKSpecification(entityClass, xmlDmd.getPKClassName());
                }
            } else {
                // Descriptor has a single primary key. Validate an id 
                // attribute was found, unless we are an inheritance subclass.
                if (! xmlDmd.hasPrimaryKeyFields() && ! xmlDmd.isInheritanceSubclass()) {
                    // here, we want to default the primary key for this entity
                    processDefaultId(xmlDmd);
                }
            }
        
            // process secondary-table(s)
            processSecondaryTables(node, xmlDmd);

            // process listeners
            processEntityListeners(node, xmlDmd);
        }
    }

    /**
     * INTERNAL:
     * Parse the xml descriptor instance document represented by the provided 
     * URL. The result will be a session containing a TopLink project with 
     * descriptors for any entities defined in the document, as well as any 
     * globally defined settings (set on the entity-mappings element). Any 
     * existing project information (set from reading in project XML) and will 
     * not be overwritten.
     * 
     * @param session TopLink session containing a TopLink project to which 
     * persistence information (contained within the provided persistence ORM 
     * instance document) will be written
     * @param xmlDoumentURL the URL of the instance document to be processed
     * @param classLoader the class loader to be used
     */
    public Session processEntityMappings(AbstractSession session, URL xmlDocumentURL) {
        init(session, xmlDocumentURL);
        Node node;
        
        // process persistence unit default overrides
        // process catalog
        if ((node = m_xPathEngine.selectSingleNode(m_document, new String[] {XMLConstants.ENTITY_MAPPINGS, XMLConstants.CATALOG, XMLConstants.TEXT})) != null) {
            m_catalog = node.getNodeValue();
        }
        
        // process schema
        if ((node = m_xPathEngine.selectSingleNode(m_document, new String[] {XMLConstants.ENTITY_MAPPINGS, XMLConstants.SCHEMA, XMLConstants.TEXT})) != null) {
            m_schema = node.getNodeValue();
        }
        
        // process access
        if ((node = m_xPathEngine.selectSingleNode(m_document, new String[] {XMLConstants.ENTITY_MAPPINGS, XMLConstants.ACCESS, XMLConstants.TEXT})) != null) {
            m_access = node.getNodeValue();
        }

        // process named-queries
        processNamedQueries(m_document, new String[] {XMLConstants.ENTITY_MAPPINGS, XMLConstants.NAMED_QUERY});

        // process named-native-queries
        processNamedNativeQueries(m_document, new String[] {XMLConstants.ENTITY_MAPPINGS, XMLConstants.NAMED_NATIVE_QUERY});

        // process sql-result-set-mappings
        processSQLResultSetMappings(m_document, new String[] {XMLConstants.ENTITY_MAPPINGS, XMLConstants.SQL_RESULT_SET_MAPPING});
        
        // pre-process table-generators
        NodeList results = m_xPathEngine.selectNodes(m_document, new String[] {XMLConstants.ENTITY_MAPPINGS, XMLConstants.TABLE_GENERATOR});
        if (results != null) {
        	int resultCount = results.getLength();
            for (int i = 0; i < resultCount; i++) {
                preProcessTableGenerator(results.item(i));
            }
        }

        // pre-process sequence-generators
        results = m_xPathEngine.selectNodes(m_document, new String[] {XMLConstants.ENTITY_MAPPINGS, XMLConstants.SEQUENCE_GENERATOR});
        if (results != null) {
        	int resultCount = results.getLength();
            for (int i = 0; i < resultCount; i++) {
                preProcessSequenceGenerator(results.item(i));
            }
        }

        // process entities
        processEntities();
        
        // handle sequencing
        m_sequencingProcessor.process(session.getProject().getLogin());

        for (Iterator dmdIt = m_relatedEntities.iterator(); dmdIt.hasNext();) {
            processRelatedEntity((XMLDescriptorMetadata) dmdIt.next());
        }
        
        return session;
    }

    /**
     * INTERNAL:
     * Process the related entities. That is, mappings and inheritance etc.
     */
    protected void processRelatedEntity(XMLDescriptorMetadata dmd) {
        // process an inheritance subclass specifics
        if (dmd.isInheritanceSubclass()) {
            processInheritanceSubclass(dmd);
        }
            
        // process the relationship accessors
        for (Iterator dmdIt = dmd.getRelationshipAccessors().iterator(); dmdIt.hasNext();) {
        	XMLAccessor accessor = (XMLAccessor) dmdIt.next();
            processRelationshipAccessor(accessor);  
        }
    }
    
    /**
     * INTERNAL:
     * Process all the event tags for the listener class - the names of the 
     * callback methods will be stored in the listener for initialization later 
     * on.
     */
    protected void processEvents(XMLListener listener, Node node) {
        // process pre-persist
        addEventMethodNameToListener(listener, node, XMLConstants.PRE_PERSIST);
        // process post-persist
        addEventMethodNameToListener(listener, node, XMLConstants.POST_PERSIST);
        // process pre-remove
        addEventMethodNameToListener(listener, node, XMLConstants.PRE_REMOVE);
        // process post-remove
        addEventMethodNameToListener(listener, node, XMLConstants.POST_REMOVE);
        // process pre-update
        addEventMethodNameToListener(listener, node, XMLConstants.PRE_UPDATE);
        // process post-update
        addEventMethodNameToListener(listener, node, XMLConstants.POST_UPDATE);
        // process post-load
        addEventMethodNameToListener(listener, node, XMLConstants.POST_LOAD);
    }
    
    /**
     * INTERNAL:
     * Process all the event tags for the listener class - the names of the 
     * callback methods will be stored in the map for initialization later on.
     */
    protected void processEventMethodNames(HashMap methodMap, Node node) {
        // process pre-persist
        addEventMethodNameToMap(methodMap, node, XMLConstants.PRE_PERSIST);
        // process post-persist
        addEventMethodNameToMap(methodMap, node, XMLConstants.POST_PERSIST);
        // process pre-remove
        addEventMethodNameToMap(methodMap, node, XMLConstants.PRE_REMOVE);
        // process post-remove
        addEventMethodNameToMap(methodMap, node, XMLConstants.POST_REMOVE);
        // process pre-update
        addEventMethodNameToMap(methodMap, node, XMLConstants.PRE_UPDATE);
        // process post-update
        addEventMethodNameToMap(methodMap, node, XMLConstants.POST_UPDATE);
        // process post-load
        addEventMethodNameToMap(methodMap, node, XMLConstants.POST_LOAD);
    }
    
    /**
     * INTERNAL:
     * Process an id element if there is one.
     */
    protected void processId(MetadataAccessor metadataAccessor, DatabaseField field) {
    	XMLAccessor accessor = (XMLAccessor) metadataAccessor;
        if (accessor.isId()) {
            processId(accessor, field, accessor.getGeneratedValueNode());
        }
    }

    /**
     * INTERNAL:
     * Process an id-class. It is used to specify a composite primary key class. 
     * The primary keys will be processed and stored from the PK class so that 
     * they may be validated against the fields or properties of the entity 
     * bean. The access type of a primary key class is determined by the access 
     * type of the entity for which it is the primary key.
     */
    protected void processIdClass(XMLDescriptorMetadata xmlDmd) {
        Class cls = xmlDmd.getJavaClass();
        Node node = m_xmlHelper.locateNode(cls);
        Node result = m_xPathEngine.selectSingleNode(node, new String[] {XMLConstants.ID_CLASS});
        
        if (result != null) {
            // process @class (required)
            processIdClass(m_xmlHelper.getClassForNode(result), xmlDmd);
        }
    }

    /**
     * INTERNAL:
     * Process inheritance. Inheritance must be specified on the entity class 
     * that is the root of then entity class hierarchy.
     */
    protected void processInheritanceRoot(MetadataDescriptor metadataDescriptor) {
        processInheritanceRoot(m_xmlHelper.locateNode(metadataDescriptor.getJavaClass()), (XMLDescriptorMetadata) metadataDescriptor);
    }
    
    /**
     * INTERNAL:
     * Process inheritance. Inheritance must be specified on the entity class 
     * that is the root of then entity class hierarchy.
     */
    protected void processInheritanceRoot(Node entityNode, XMLDescriptorMetadata dmd) {
        if (dmd.ignoreInheritanceAnnotations()) {
            // Project XML/XML merging. Project XML wins
            getLogger().logWarningMessage(MetadataLogger.IGNORE_INHERITANCE, dmd);
        } else {
            Node inheritanceNode = m_xPathEngine.selectSingleNode(entityNode, new String[] {XMLConstants.INHERITANCE});
            if (inheritanceNode != null || dmd.isInheritanceRoot()) {
                // Default is SINGLE_TABLE if no inheritance node is found,
                // but the descriptor metadata is tagged as an inheritance root.
                String strategy = XMLConstants.SINGLE_TABLE;
                
                if (inheritanceNode != null) {
                    // process @strategy
                    Node rootResult = m_xPathEngine.selectSingleNode(inheritanceNode, new String[] {XMLConstants.ATT_STRATEGY});
                    strategy = rootResult.getNodeValue();
                }
	
	            // Store the strategy on the metadata descriptor.
	            dmd.setInheritanceStrategy(strategy);
	            
	            // Process discriminator-column.
	            processDiscriminatorColumn(entityNode, dmd);
	            
	            // Process discriminator-value.
	            processDiscriminatorValue(entityNode, dmd);
            }
        }
    }

    /**
     * INTERNAL:
     * Process a join-column and return a populated JoinColumnDetails.
     */
    protected MetadataJoinColumn processJoinColumn(Node joinColumnNode, XMLAccessor accessor, String targetTable, MetadataDescriptor sourceDmd) {
        // set defaults
        String name = "";
        String referencedColumnName = "";
        boolean isUnique = false;
        boolean isNullable = true;
        boolean isInsertable = true;
        boolean isUpdatable = true;
        String columnDefinition = "";
        String tableName = "";
        Node result;
        
        // process @name - a.k.a. fkColumn
        if ((result = m_xPathEngine.selectSingleNode(joinColumnNode, new String[] {XMLConstants.ATT_NAME})) != null) {
            name = result.getNodeValue();
        }
        
        // process @referenced-column-name - a.k.a. pkColumn
        if ((result = m_xPathEngine.selectSingleNode(joinColumnNode, new String[] {XMLConstants.ATT_REFERENCED_COLUMN_NAME})) != null) {
            referencedColumnName = result.getNodeValue();
        }
        
        // process @unique
        if ((result = m_xPathEngine.selectSingleNode(joinColumnNode, new String[] {XMLConstants.ATT_UNIQUE})) != null) {
            isUnique = Boolean.parseBoolean(result.getNodeValue());
        }
        
        // process @nullable
        if ((result = m_xPathEngine.selectSingleNode(joinColumnNode, new String[] {XMLConstants.ATT_NULLABLE})) != null) {
            isNullable = Boolean.parseBoolean(result.getNodeValue());
        }
        
        // process @insertable
        if ((result = m_xPathEngine.selectSingleNode(joinColumnNode, new String[] {XMLConstants.ATT_INSERTABLE})) != null) {
            isInsertable = Boolean.parseBoolean(result.getNodeValue());
        }
        
        // process @updatable
        if ((result = m_xPathEngine.selectSingleNode(joinColumnNode, new String[] {XMLConstants.ATT_UPDATABLE})) != null) {
            isUpdatable = Boolean.parseBoolean(result.getNodeValue());
        }
        
        // process @column-definition
        if ((result = m_xPathEngine.selectSingleNode(joinColumnNode, new String[] {XMLConstants.ATT_COLUMN_DEFINITION})) != null) {
            columnDefinition = result.getNodeValue();
        }
        
        // process @table - will be attached to fkColumn
        if ((result = m_xPathEngine.selectSingleNode(joinColumnNode, new String[] {XMLConstants.ATT_SECONDARY_TABLE})) != null) {
            tableName = result.getNodeValue();
        }
        
        return processJoinColumn(
    			accessor,
        		targetTable, 
        		sourceDmd, 
        		name, 
        		referencedColumnName, 
        		columnDefinition, 
        		tableName, 
        		isUnique, 
        		isNullable, 
        		isInsertable, 
        		isUpdatable);
    }

    /**
     * INTERNAL:
     * Process any join-columns and return a list of JoinColumnDetails.
     */
    protected ArrayList processJoinColumns(MetadataAccessor metadataAccessor) {
        ArrayList allJoinColumns = new ArrayList();
        XMLAccessor accessor = (XMLAccessor) metadataAccessor; 
        
        XMLDescriptorMetadata dmd = accessor.getMetadataDescriptor();
        String targetTableName = dmd.getPrimaryTableName();
        MetadataDescriptor referenceDmd = accessor.getReferenceMetadataDescriptor();
        String sourceTableName = referenceDmd.getPrimaryTableName();
        
        Node node = accessor.getAttributeNode();
        String attributeName = accessor.getAttributeName();
        
        // process join-column(s)
        NodeList joinColumns;
        if (dmd.hasAssociationOverrideFor(accessor)) {
        	joinColumns = (NodeList)(dmd.getAssociationOverrideFor(accessor)[0]);
        } else {
            joinColumns = m_xPathEngine.selectNodes(node, new String[] {XMLConstants.JOIN_COLUMN});
        }
        
        if (referenceDmd.hasCompositePrimaryKey()) {
            // composite primary key: the number of <join-column> elements 
        	// specified should equal the number of primary key fields
            if (joinColumns == null || joinColumns.getLength() != referenceDmd.getPrimaryKeyFields().size()) {
                getValidator().throwIncompleteJoinColumnsSpecified(dmd.getJavaClass(), attributeName);
            }
            
            int columnCount = joinColumns.getLength();
            for (int i = 0; i < columnCount; i++) {
                allJoinColumns.add(processJoinColumn(joinColumns.item(i), accessor, targetTableName, referenceDmd));
            }
        } else {
            // single primary key
            if (joinColumns != null && joinColumns.getLength()!=0) {
            	if (joinColumns.getLength()>1) {
                    getValidator().throwExcessiveJoinColumnsSpecified(dmd.getJavaClass(), attributeName);
            	}
                
                allJoinColumns.add(processJoinColumn(joinColumns.item(0), accessor, targetTableName, referenceDmd));
            } else {
                addJoinColumnDefault(allJoinColumns, sourceTableName, targetTableName, dmd);
            }
        }
            
        return allJoinColumns;
    }

    /**
     * INTERNAL:
     * Process join-table information.
     */
    protected void processJoinTable(MetadataAccessor metadataAccessor, ManyToManyMapping mapping) {
    	XMLAccessor accessor = (XMLAccessor) metadataAccessor;
    	
    	// the relationship may be only partially defined in XML,
    	// i.e. the target-entity class definition may not be present 
		if (!accessor.isTargetEntityDefinedInDocument()) {
			mapping.setRequiresCompletion(true);
		}
    	
        DatabaseTable relationTable = getRelationTable(accessor);
        mapping.setRelationTable(relationTable);
        String relationTableName = relationTable.getQualifiedName();

        // process join-column and inverse-join-column
        ArrayList sourceKeys = new ArrayList();
        ArrayList targetKeys = new ArrayList();
        
        Node joinTableNode = accessor.getJoinTableNode();
        if (joinTableNode != null) {
	        // process join-columns
	        NodeList jCols = m_xPathEngine.selectNodes(joinTableNode, new String[] {XMLConstants.JOIN_COLUMN});
	        if (jCols != null) {
	        	int columnCount = jCols.getLength();
	        	for (int i=0; i<columnCount; i++) {
	                sourceKeys.add(processJoinColumn(jCols.item(i), accessor, relationTableName, accessor.getMetadataDescriptor()));
	        	}
	        }            	
	
	        // process inverse-join-column
	        jCols = m_xPathEngine.selectNodes(joinTableNode, new String[] {XMLConstants.INVERSE_JOIN_COLUMN});
	        if (jCols != null) {
	        	int columnCount = jCols.getLength();
	        	for (int i = 0; i < columnCount; i++) {
	                targetKeys.add(processJoinColumn(jCols.item(i), accessor, relationTableName, accessor.getReferenceMetadataDescriptor()));
	        	}
	        }            	
        }
        
        processJoinTable(accessor, mapping, sourceKeys, targetKeys);
    }    

    /**
     * INTERNAL:
     * Process entity listeners.
     */
    protected void processEntityListeners(Node node, XMLDescriptorMetadata xmlDmd) {
		// Process exclude-superclass-listeners.
        xmlDmd.setExcludeSuperclassListeners(m_xPathEngine.selectSingleNode(node, new String[] {XMLConstants.EXCLUDE_SUPERCLASS_LISTENERS}) != null);
        
    	// Process exclude-default-listeners.
        xmlDmd.setExcludeDefaultListeners(m_xPathEngine.selectSingleNode(node, new String[] {XMLConstants.EXCLUDE_DEFAULT_LISTENERS}) != null);
    	
    	// Process entity listeners.
        processEntityListeners(node, new String[] {XMLConstants.ENTITY_LISTENERS, XMLConstants.ENTITY_LISTENER}, xmlDmd);
        
        // Process entity event callback methods.
        processEntityEventListener(node, xmlDmd);
    }
    
    /**
     * INTERNAL:
     * Process entity listeners.
     */
    protected void processEntityListeners(Node node, String[] xPath, XMLDescriptorMetadata xmlDmd) {
        NodeList listenerNodes = m_xPathEngine.selectNodes(node, xPath);
        
        if (listenerNodes.getLength() > 0) {
        	int nodeCount = listenerNodes.getLength();
            
            for (int i = 0; i < nodeCount; i++) {
                Node listenerNode = listenerNodes.item(i);
                
                // Process @class (required).
                String listenerClassName = m_xmlHelper.getClassNameForNode(listenerNode);
                XMLEntityListener listener = new XMLEntityListener(listenerClassName, xmlDmd.getJavaClassName());
                processEvents(listener, listenerNode);
                xmlDmd.addEntityListenerEventListener(listener);
            }
        }
    }
    
    /**
     * INTERNAL:
     * Process an entity class for callback methods.
     */
    protected void processEntityEventListener(Node node, XMLDescriptorMetadata xmlDmd) {
        if (m_xmlHelper.entityHasCallbackMethod(node)) {
            XMLEntityClassListener listener = new XMLEntityClassListener(xmlDmd.getJavaClassName());
            processEvents(listener, node);
            xmlDmd.setEntityEventListener(listener);
        }
    }

    /**
     * INTERNAL:
     * Process a M-M into a TopLink ManyToMany mapping.
     */
    protected void processManyToMany(MetadataAccessor metadataAccessor) {
    	XMLAccessor accessor = (XMLAccessor) metadataAccessor;
        processManyToMany(
        		accessor, 
        		accessor.getTargetEntity(), 
        		getCascadeTypes(accessor), 
        		accessor.usesIndirection(), 
                accessor.getMappedBy());
    }
    
    /**
     * INTERNAL:
     * Process a M-1 into a TopLink OneToOne mapping.
     * Note: Target foreign keys are not valid for m-1 mapping
     */
    protected void processManyToOne(MetadataAccessor metadataAccessor) {
    	XMLAccessor accessor = (XMLAccessor) metadataAccessor;
    	processManyToOne(
        		accessor, 
                accessor.getTargetEntity(),
        		getCascadeTypes(accessor),
        		accessor.usesIndirection(),
        		accessor.isOptional());
    }

    /**
     * INTERNAL:
     * Process map-key for a one-to-many or many-to-many mapping. 
     * @return the map key method name that should be used, null otherwise
     */
	protected String processMapKey(MetadataAccessor metadataAccessor, CollectionMapping mapping) {
		XMLAccessor accessor = (XMLAccessor) metadataAccessor;
        
        if (!accessor.hasMapKey()) {
        	return null;
		}
        
        return processMapKey(accessor, mapping, accessor.getMapKey());
    }
    
    /**
     * INTERNAL:
     * Process any mapped-superclasses if there are any. There may be
     * several mapped-superclasses for any given Entity.
     */
    protected void processMappedSuperclasses(XMLDescriptorMetadata dmd) {
        for (Iterator iterator = dmd.getMappedSuperclasses().iterator(); iterator.hasNext(); ) {
        	Class cls = (Class) iterator.next();
            // Process the accessors from the mapped superclass.
            processAccessors(cls, dmd);
            
            // Process the attribute overrides on this MappedSuperclass, if
            // there are any before processing its parent MappedSuperclass.
            processAttributeOverrides(cls, dmd);
        }
    }
    
    /**
     * INTERNAL:
     */
    protected void processNamedNativeQueries(Node node, String[] xPath) {
        processNamedNativeQueries(node, xPath, "");
    }
    
    /**
     * INTERNAL:
     * Process named-queries at either the entity-mappings or entity level. The
     * queries will be stored in a hashmap - EntityManagerSetupImpl will call 
     * 'addNamedQueriesToSession()' after processing; this is when the queries
     * will be added to the session.
     * 
     * @param node either entity-mapping or entity node
     * @param xpath to search for named-native-query tag(s) be null if 
     * processing at an entity-mapping level
     * @param entityClass the owning class - for warning messages
     */
    protected void processNamedNativeQueries(Node node, String[] xPath, String entityClassName) {
        NodeList queries = m_xPathEngine.selectNodes(node, xPath);
        if (queries != null) {
            // for each named native query...
        	int queryCount = queries.getLength();
            for (int i=0; i<queryCount; i++) {
                Node queryNode = queries.item(i);
                
                // process @name (required)
                String queryName = m_xPathEngine.selectSingleNode(queryNode, new String[] {XMLConstants.ATT_NAME}).getNodeValue();
                
                // make sure the query doesn't already exist in the map
                if (m_namedNativeQueries.containsKey(queryName)) {
                    getLogger().logWarningMessage(MetadataLogger.IGNORE_QUERY, queryName);
                    continue;
                }

                // process query
                Node ejbqlNode = m_xPathEngine.selectSingleNode(queryNode, new String[] {XMLConstants.QUERY, XMLConstants.TEXT}); 
                String ejbql = "";
                if (ejbqlNode != null && ejbqlNode.getNodeValue() != null) {
                	ejbql = ejbqlNode.getNodeValue();
                }

                // create properties for this query
                HashMap props = new HashMap();
                // store the query string
                props.put(XMLConstants.QUERY, ejbql);
                
                // process @result-class
                Node result = m_xPathEngine.selectSingleNode(queryNode, new String[] {XMLConstants.ATT_RESULT_CLASS});
                if (result != null && !result.getNodeValue().equals("")) {
                    props.put(XMLConstants.ATT_RESULT_CLASS, m_xmlHelper.getFullyQualifiedClassName(result.getNodeValue()));
                }
                
                // process @result-set-mapping
                result = m_xPathEngine.selectSingleNode(queryNode, new String[] {XMLConstants.ATT_RESULT_SET_MAPPING});
                if (result != null && !result.getNodeValue().equals("")) {
                    props.put(XMLConstants.ATT_RESULT_SET_MAPPING, result.getNodeValue());
                }

                // process hints
            	props.put(XMLConstants.QUERY_HINT, processQueryHints(queryNode, queryName, entityClassName));
                
                // store the query info for later processing
                m_namedNativeQueries.put(queryName, props);
            }
        }
    }

    /**
     * INTERNAL:
     */
    protected void processNamedQueries(Node node, String[] xPath) {
        processNamedQueries(node, xPath, "");
    }
    
    /**
     * INTERNAL:
     * Process named queries at either the entity-mapping or entity level. The 
     * queries will be stored in a hashmap - EntityManagerSetupImpl will call 
     * 'addNamedQueriesToSession()' after processing; this is when the queries
     * will be added to the session.
     * 
     * @param node either entity-mapping or entity node
     * @param xpath to search for named-native-query tag(s) be null if 
     * processing at an entity-mapping level
     * @param entityClass the owning class - for warning messages
     */
    protected void processNamedQueries(Node node, String[] xPath, String entityClassName) {
        NodeList queries = m_xPathEngine.selectNodes(node, xPath);
        
        if (queries != null) {
            // for each named query...
        	int queryCount = queries.getLength();
            
            for (int i = 0; i < queryCount; i++) {
                Node queryNode = queries.item(i);

                // process @name (required)
                String queryName = m_xPathEngine.selectSingleNode(queryNode, new String[] {XMLConstants.ATT_NAME}).getNodeValue();
                
                // make sure the query doesn't already exist in the map
                if (m_namedQueries.containsKey(queryName)) {
                    getLogger().logWarningMessage(MetadataLogger.IGNORE_QUERY, queryName, entityClassName);
                    continue;
                }

                // process query
                Node ejbqlNode = m_xPathEngine.selectSingleNode(queryNode, new String[] {XMLConstants.QUERY, XMLConstants.TEXT}); 
                String ejbql = "";
                if (ejbqlNode != null && ejbqlNode.getNodeValue() != null) {
                	ejbql = ejbqlNode.getNodeValue();
                }

                // create properties for this query
                HashMap props = new HashMap();
                // store the query string
                props.put(XMLConstants.QUERY, ejbql);
                // process hints
            	props.put(XMLConstants.QUERY_HINT, processQueryHints(queryNode, queryName, entityClassName));
                // store the query info for later processing
                m_namedQueries.put(queryName, props);
            }
        }
    }
    
    /**
     * INTERNAL:
     * Process a one-to-many into a TopLink OneToMany mapping. If a join-table 
     * is found however, we must create a ManyToMany mapping.
     */
    protected void processOneToMany(MetadataAccessor metadataAccessor) {
    	XMLAccessor accessor = (XMLAccessor) metadataAccessor;
    	processOneToMany(accessor, accessor.getAttributeNode(), accessor.getMappedBy());
    }
	
    /**
     * INTERNAL:
     * Process a one-to-one into a TopLink OneToOne mapping.
     */
    protected void processOneToOne(MetadataAccessor metadataAccessor) {
    	XMLAccessor accessor = (XMLAccessor) metadataAccessor;
        processOneToOne(accessor,  
                        accessor.getTargetEntity(), 
                        getCascadeTypes(accessor), 
                        accessor.usesIndirection(),
                        accessor.isOptional(), 
                        accessor.getMappedBy());
    }
    
    /**
     * INTERNAL:
     * Process an order-by for collection mappings.
     */
    protected void processOrderBy(MetadataAccessor metadataAccessor, CollectionMapping mapping) {
    	XMLAccessor accessor = (XMLAccessor) metadataAccessor; 
    	if (accessor.hasOrderBy()) {
    		processOrderBy(accessor, mapping, accessor.getOrderByNode().getNodeValue());
    	}
    }
    
    /**
     * INTERNAL:
     */
    protected void processOwningMappingKeys(OneToOneMapping mapping, MetadataAccessor accessor) {
    	if (!((XMLAccessor)accessor).isTargetEntityDefinedInDocument()) {
    		mapping.setRequiresCompletion(true);
    	}
    	
        super.processOwningMappingKeys(mapping, accessor);
    }
    
    /**
     * INTERNAL:
     */
    protected void processPersistenceUnitDefault(String key, Node contextNode, HashMap puDefaults) {
        Node node;
        String nodeValue;
        Object existingValue;
        
        if ((node = m_xPathEngine.selectSingleNode(contextNode, new String[] {key, XMLConstants.TEXT})) != null) {
        	nodeValue = node.getNodeValue();
        	existingValue = puDefaults.get(key);
        	
            if (existingValue != null && (!((String)existingValue).equals(nodeValue))) {
                getValidator().throwPersistenceUnitMetadataConflict(key);
        	}
            
        	puDefaults.put(key, nodeValue);
        } else {
        	// Here, the current document does not define a PU default value for 
            // this key, so we'll set the value to an empty string - if another 
            // document contains a PU default value for this key, it is a 
            // conflict.
        	puDefaults.put(key, "");
        }
    }
    
    /**
     * INTERNAL:
     * Process persistence unit metadata and defaults, and apply them to each 
     * entity in the collection. Any conflicts in elements defined in multiple 
     * documents will cause an exception to be thrown.  The first instance 
     * encountered wins, i.e. any conflicts between PU metadata definitions in 
     * multiple instance documents will cause an exception to be thrown.  The 
     * one exception to this rule is default listeners: all default listeners 
     * found will be added to a list in the order that they are read from the 
     * instance document(s). 
     */
    public void processPersistenceUnitMetadata(AbstractSession session, Collection mappingFiles, Collection entities) {
        m_session = session;
        Boolean cascadePersist = null;
        Boolean metadataComplete = null;
        HashMap puDefaults = new HashMap();
        HashMap defaultListenerList = new HashMap();
        ArrayList orderedDefaultListenerList = new ArrayList();
        
        // WIP - go through this block of code and can clean it up!
    	if (! mappingFiles.isEmpty()) {
            m_xPathEngine = XPathEngine.getInstance();
            puDefaults.put(XMLConstants.ENTITY_LISTENERS, defaultListenerList);
    	
            // For each orm xml instance document, process persistence unit 
            // metadata/defaults
            for (Iterator mapIt = mappingFiles.iterator(); mapIt.hasNext(); ) {
                try {
                    // Initialize a helper for navigating the instance document.
                    m_xmlHelper = new XMLHelper((URL) mapIt.next(), m_loader);
                
                    // Create a context node referencing persistence-unit-metadata.
                    String[] xPath = new String[] {XMLConstants.ENTITY_MAPPINGS, XMLConstants.PU_METADATA};
                    Node contextNode = m_xPathEngine.selectSingleNode(m_xmlHelper.getDocument(), xPath);
                
                    if (contextNode == null) {
                        continue;
                    }

                    // process xml-mapping-metadata-complete
                    Node result = m_xPathEngine.selectSingleNode(contextNode, new String[] {XMLConstants.METADATA_COMPLETE});
                    if (result == null) {
                        // no metadata-complete element found
                        if (metadataComplete == null) {
                            metadataComplete = false;
                        } else {
                            if (metadataComplete) {
                                // a previous document contained a metadata-complete element, indicating 
                                // metadata-complete, while this document does not - throw a conflict exception
                                getValidator().throwPersistenceUnitMetadataConflict(XMLConstants.METADATA_COMPLETE);
                            }
                        }
                    } else {
                        // metadata-complete is set
                        if (metadataComplete == null) {
                            metadataComplete = true;
                            puDefaults.put(XMLConstants.METADATA_COMPLETE, true);
                        } else {
                            if (!metadataComplete) {
                                // a previous document did not define a metadata-complete element, indicating non-
                                // metadata complete, while this document does - throw a conflict exception
                                getValidator().throwPersistenceUnitMetadataConflict(XMLConstants.METADATA_COMPLETE);
                            }
                        }
                    }

                    // process persistence unit defaults
                    contextNode = m_xPathEngine.selectSingleNode(contextNode, new String[] {XMLConstants.PU_DEFAULTS});
                    if (contextNode == null) {
                        continue;
                    }

                    // process access
                    processPersistenceUnitDefault(XMLConstants.ACCESS, contextNode, puDefaults);
                    m_defaultAccess = (String) puDefaults.get(XMLConstants.ACCESS);
                
                    // process schema
                    processPersistenceUnitDefault(XMLConstants.SCHEMA, contextNode, puDefaults);
                    m_defaultSchema = (String) puDefaults.get(XMLConstants.SCHEMA);
                
                    // process catalog
                    processPersistenceUnitDefault(XMLConstants.CATALOG, contextNode, puDefaults);
                    m_defaultCatalog = (String) puDefaults.get(XMLConstants.CATALOG);
        		
                    // process cascade-persist
                    result = m_xPathEngine.selectSingleNode(contextNode, new String[] {XMLConstants.CASCADE_PERSIST});
                    if (result == null) {
                        // no cascade-persist element found
                        if (cascadePersist == null) {
                            cascadePersist = false;
                        } else {
                            if (cascadePersist) {
                                // a previous document contained a cascade-persist element, indicating 
                                // cascade-persist, while this document does not - throw a conflict exception
                                getValidator().throwPersistenceUnitMetadataConflict(XMLConstants.CASCADE_PERSIST);
                            }
                        }
                    } else {
                        // cascade-persist is set
                        if (cascadePersist == null) {
                            cascadePersist = true;
                            puDefaults.put(XMLConstants.CASCADE_PERSIST, true);
                        } else {
                            if (!cascadePersist) {
                                // a previous document did not define a cascade-persist element, indicating 
                                // cascade-persist should not be applied to all relationship mappings in the 
                                // persistence unit, while this document does - throw a conflict exception
                                getValidator().throwPersistenceUnitMetadataConflict(XMLConstants.CASCADE_PERSIST);
                            }
                        }
                    }
                
                    // Process the default entity-listeners. No conflict checking 
                    // will be done, that is, any and all default listeners will be 
                    // added to the list
                    xPath = new String[] {XMLConstants.ENTITY_LISTENERS, XMLConstants.ENTITY_LISTENER};
                    NodeList listenerNodes = m_xPathEngine.selectNodes(contextNode, xPath);
                
                    if (listenerNodes != null && listenerNodes.getLength() > 0) {
                        int listenerCount = listenerNodes.getLength(); 
                    
                        for (int i = 0; i < listenerCount; i++) {
                            Node listenerNode = listenerNodes.item(i);
                            String listenerClassName = m_xmlHelper.getClassNameForNode(listenerNode);
                            HashMap methodMap = new HashMap();
                            processEventMethodNames(methodMap, listenerNode);
                            defaultListenerList.put(listenerClassName, methodMap);
                            orderedDefaultListenerList.add(listenerClassName);
                        }
                    }
                } catch (RuntimeException re) {
                    throw re;
                }
            }
        }
    	
        // Build our list of metadata descriptors, applying any persistence 
        // unit metadata/defaults that we found. All known descriptors are 
        // added to the project right away.
		if (! entities.isEmpty()) {
	    	for (Iterator iterator = entities.iterator(); iterator.hasNext(); ) {
	    		Class entityClass = (Class) iterator.next();
                
	    		XMLDescriptorMetadata xmlDmd = new XMLDescriptorMetadata(entityClass, null);
	    		m_session.getProject().addDescriptor(xmlDmd.getDescriptor());
                getDescriptorMetadataMap().put(entityClass, xmlDmd);
                
                if (m_defaultAccess.equals(XMLConstants.FIELD)) {
                    xmlDmd.setIsXmlFieldAccess();
                } else if (m_defaultAccess.equals(XMLConstants.PROPERTY)) {
                    xmlDmd.setIsXmlPropertyAccess();
                } else {
                    // set nothing.
                }
                
	    		xmlDmd.setCatalog(m_defaultSchema);
	    		xmlDmd.setSchema(m_defaultCatalog);
                
                if (puDefaults.get(XMLConstants.CASCADE_PERSIST) != null) {
                    xmlDmd.setShouldUseCascadePersist(true);    
                }
	
	    		// Apply default listeners.
                processDefaultListeners(orderedDefaultListenerList, defaultListenerList, xmlDmd);
	    		
				// Indicate whether to ignore annotations or not.
				xmlDmd.setShouldIgnoreAnnotations(puDefaults.get(XMLConstants.METADATA_COMPLETE) != null);
	    	}
		}
    }
    
    /**
     * INTERNAL:
     */
    protected MetadataJoinColumn processPrimaryKeyJoinColumnNode(Node pkJoinColNode, String sourceTable, String targetTable, MetadataDescriptor dmd) {
        Node result;
        String name = "";
        String referencedColumnName = "";
        String columnDefinition = "";
        
        // process @name
        if ((result = m_xPathEngine.selectSingleNode(pkJoinColNode, new String[] {XMLConstants.ATT_NAME})) != null) {
            name = result.getNodeValue();
        }
        
        // process @referenced-column-name
        if ((result = m_xPathEngine.selectSingleNode(pkJoinColNode, new String[] {XMLConstants.ATT_REFERENCED_COLUMN_NAME})) != null) {
            referencedColumnName = result.getNodeValue();
        }
        
        // process @column-details
        if ((result = m_xPathEngine.selectSingleNode(pkJoinColNode, new String[] {XMLConstants.ATT_COLUMN_DEFINITION})) != null) {
            columnDefinition = result.getNodeValue();
        }
        
        return processPrimaryKeyJoinColumn(name, referencedColumnName, columnDefinition, sourceTable, targetTable, dmd);    
    }
    
    /**
     * INTERNAL:
     */
    protected ArrayList processPrimaryKeyJoinColumns(Object[] pkJoinCols, String sourceTableName, String targetTableName, MetadataDescriptor dmd) {
    	ArrayList joinColumnDetails = new ArrayList();
	    for (int i = 0; i < pkJoinCols.length; i++) {
	    	joinColumnDetails.add(processPrimaryKeyJoinColumnNode((Node) pkJoinCols[i], sourceTableName, targetTableName, dmd));
	    }
	    
        return joinColumnDetails;
    }

    /**
     * INTERNAL:
	 * Process the primary key join columns for a given entity.
     * @param element will be either a Node or Class depending on where this 
     * method is called from
     * @return ArrayList containing 0 or more JoinColumnDetails
     */
    protected ArrayList processPrimaryKeyJoinColumns(String sourceTable, String targetTable, Object element, MetadataDescriptor dmd) {
        Node entityNode = m_xmlHelper.locateNode(dmd.getJavaClass());
    	Node node;
    	
        if (element instanceof String) {
    		node = m_xmlHelper.locateNodeForAttribute(entityNode, (String) element);
    	} else {
    		node = entityNode;
    	}

    	ArrayList joinColumnDetails = new ArrayList();
		Class entityClass = dmd.getJavaClass();
		
        // look for primary-key-join-column(s) elements
        NodeList primaryKeyJoinColumns = m_xPathEngine.selectNodes(node, new String[] {XMLConstants.PK_JOIN_COLUMN});
        
        if (dmd.hasCompositePrimaryKey()) {
            // the number of primary-key-join-column entries should equal the number of primary key fields
            if (primaryKeyJoinColumns == null || primaryKeyJoinColumns.getLength() != dmd.getPrimaryKeyFields().size()) {
                getValidator().throwIncompletePrimaryKeyJoinColumnsSpecified(entityClass);
            }
        } else {
            if (primaryKeyJoinColumns != null && primaryKeyJoinColumns.getLength() > 1) {
                getValidator().throwExcessivePrimaryKeyJoinColumnsSpecified(entityClass);
            }
        }

        // process primary-key-join-column(s)
        if (primaryKeyJoinColumns != null) {
        	int columnCount = primaryKeyJoinColumns.getLength();
            for (int i=0; i<columnCount; i++) {
                Node pkJoinColumnNode = primaryKeyJoinColumns.item(i);
                Node result;
                String name = "";
                String referencedColumnName = "";
                String columnDefinition = "";
                
                // process @name
                if ((result = m_xPathEngine.selectSingleNode(pkJoinColumnNode, new String[] {XMLConstants.ATT_NAME})) != null) {
                    name = result.getNodeValue();
                }
        
                // process @referenced-column-name
                if ((result = m_xPathEngine.selectSingleNode(pkJoinColumnNode, new String[] {XMLConstants.ATT_REFERENCED_COLUMN_NAME})) != null) {
                    referencedColumnName = result.getNodeValue();
                }
        
                // process @column-details
                if ((result = m_xPathEngine.selectSingleNode(pkJoinColumnNode, new String[] {XMLConstants.ATT_COLUMN_DEFINITION})) != null) {
                    columnDefinition = result.getNodeValue();
                }

                joinColumnDetails.add(processPrimaryKeyJoinColumn(name, referencedColumnName, columnDefinition, sourceTable, targetTable, dmd));
            }
		}

        // add defaults if necessary
		addJoinColumnDefault(joinColumnDetails, sourceTable, targetTable, dmd);
		
		return joinColumnDetails;
	}
	
	/**
	 * INTERNAL:
	 * Process any query-hints for a given node and store them in a map.
	 * @return HashMap of query-hints for a given node, or an empty HashMap if 
     * none exist.
	 */
	protected HashMap processQueryHints(Node queryNode, String queryName, String entityClass) {
		HashMap hintMap = new HashMap();
		NodeList hints = m_xPathEngine.selectNodes(queryNode, new String[] {XMLConstants.QUERY_HINT});
	    if (hints != null) {
	        // for each query hint...
	    	int hintCount = hints.getLength();
	        for (int h = 0; h < hintCount; h++) {
	            Node hintNode = hints.item(h);
	            // process @name (required)
	            String hintName = m_xPathEngine.selectSingleNode(hintNode, new String[] {XMLConstants.ATT_NAME}).getNodeValue();
	            // process @value (required)
	            String hintValue = m_xPathEngine.selectSingleNode(hintNode, new String[] {XMLConstants.ATT_VALUE}).getNodeValue();
	            
                // determine the type of the hint
                Object typedHintValue = getTypedHintValue(hintName, hintValue, queryName, entityClass);
	            if (typedHintValue != null) {
	                // store the hint name/value
	            	hintMap.put(hintName, typedHintValue);
	            }
	        }
	    }
        
	    return hintMap;
    }
    
    /**
     * INTERNAL:
     * Process secondary-table(s) for a given entity.  Assumed here is that the 
     * entity class has been processed for a primary table and primary key.
     */
    protected void processSecondaryTables(Node entityNode, XMLDescriptorMetadata dmd) {
        NodeList secondaryTables = m_xPathEngine.selectNodes(entityNode, new String[] {XMLConstants.SECONDARY_TABLE});
        if (secondaryTables != null) {
        	int tableCount = secondaryTables.getLength();
            for (int i = 0; i < tableCount; i++) {
                Node tableNode = secondaryTables.item(i);

                // set defaults
                String catalog = m_catalog;
                String schema = m_schema;

                // process @name (required)
                Node result = m_xPathEngine.selectSingleNode(tableNode, new String[] {XMLConstants.ATT_NAME});
                String tableName = result.getNodeValue();
                
                // process @catalog
                if ((result = m_xPathEngine.selectSingleNode(tableNode, new String[] {XMLConstants.ATT_CATALOG})) != null) {
                    catalog = result.getNodeValue();
                }
        
                // process @schema
                if ((result = m_xPathEngine.selectSingleNode(tableNode, new String[] {XMLConstants.ATT_SCHEMA})) != null) {
                    schema = result.getNodeValue();
                }
                
                processSecondaryTable(
                		tableName, 
                		schema, 
                		catalog, 
                		getUniqueConstraints(tableNode), 
                		getPrimaryKeyJoinColumns(tableNode),
                		dmd);
            }
        }
    }
	
    /**
     * INTERNAL:
     * Process sql-result-set-mappings and store them on the session.
     */
    protected void processSQLResultSetMappings(Node node, String[] xPath) {
        NodeList sqlResultSetNodes = m_xPathEngine.selectNodes(node, xPath);
        if (sqlResultSetNodes != null) {
        	int nodeCount = sqlResultSetNodes.getLength();
        
            for (int i = 0; i < nodeCount; i++) {
                m_session.getProject().addSQLResultSetMapping(createSQLResultSetMapping(sqlResultSetNodes.item(i))); 
            }
        }
    }
    
    /**
     * INTERNAL:
     * Process table information.
     * 
     * @param dmd the XMLDescriptorMetadata object of the tables's owning entity
     * @param node the DOM node representing the entity that owns the table
     */
    protected void processTable(XMLDescriptorMetadata dmd, Node node) {
    	// project xml/entity-mappings xml merging:  project xml wins
        if (dmd.ignoreTableAnnotations()) {
            getLogger().logWarningMessage(MetadataLogger.IGNORE_TABLE, dmd.getJavaClass());
        } else {
	        // process table
	        Node result;
	        if ((result = m_xPathEngine.selectSingleNode(node, new String[] {XMLConstants.TABLE})) != null) {
		        // set default/global values
		        String schemaName = m_schema;
		        String catalogName = m_catalog;
		        String tableName = "";
	            Node attr;
	            
	            // process @name
	            if ((attr = m_xPathEngine.selectSingleNode(result, new String[] {XMLConstants.ATT_NAME})) != null) {
	                tableName = attr.getNodeValue();
	            }
	            
                // process @schema
	            if ((attr = m_xPathEngine.selectSingleNode(result, new String[] {XMLConstants.ATT_SCHEMA})) != null) {
	                schemaName = attr.getNodeValue();
	            }
	            
                // process @catalog
	            if ((attr = m_xPathEngine.selectSingleNode(result, new String[] {XMLConstants.ATT_CATALOG})) != null) {
	                catalogName = attr.getNodeValue();
	            }

	        	// create the table and add set it as primary
	            processTable(tableName, catalogName, schemaName, getUniqueConstraints(result), dmd);
	        } else {
	            processDefaultTable(dmd);
	        }
        }
    }

    /**
     * INTERNAL:
     */
    protected void processTemporal(MetadataAccessor metadataAccessor, DirectToFieldMapping mapping) {
    	XMLAccessor accessor = (XMLAccessor) metadataAccessor;
            
    	if (accessor.hasTemporal()) {
            processTemporal(accessor, accessor.getTemporalNode().getNodeValue(), mapping);
    	} else {
            getValidator().throwNoTemporalTypeSpecified(accessor.getJavaClass(), accessor.getAttributeName());
        }
    }
    
    /**
     * INTERNAL:
     * Apply unique-constraints to a given database table.
     */
    protected void processUniqueConstraints(Object[] constraints, DatabaseTable dbTable) {
    	for (int i = 0; i < constraints.length; i++) {
            dbTable.addUniqueConstraint((String[]) constraints[i]);
    	}
    }
}
