/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the "License").  You may not use this file except 
 * in compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * glassfish/bootstrap/legal/CDDLv1.0.txt or 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html. 
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * HEADER in each file and include the License file at 
 * glassfish/bootstrap/legal/CDDLv1.0.txt.  If applicable, 
 * add the following below this CDDL HEADER, with the 
 * fields enclosed by brackets "[]" replaced with your 
 * own identifying information: Portions Copyright [yyyy] 
 * [name of copyright owner]
 */
// Copyright (c) 1998, 2006, Oracle. All rights reserved.  
package oracle.toplink.essentials.internal.expressions;

import java.util.*;
import oracle.toplink.essentials.expressions.*;

/**
 * Used to itterate an expression tree, through inner subclasses.
 */
public abstract class ExpressionIterator {

    /** Allow the iteration to build a result. */
    protected Object result;

    /** Some iterations require a statement. */
    protected SQLSelectStatement statement;

    /** Some iterations require a more general parameter. */
    protected Object parameter;

    /**
     * Block constructor comment.
     */
    public ExpressionIterator() {
        super();
    }

    public Object getResult() {
        return result;
    }

    public SQLSelectStatement getStatement() {
        return statement;
    }

    /**
     * Answers if this expression has already been visited.
     * For a faster iteration override to insure expressions are only
     * visited/processed once.
     */
    public boolean hasAlreadyVisited(Expression expression) {
        return false;
    }

    /**
     * INTERNAL:
     * This method must be defined by subclasses to implement the logic of the iteratation.
     */
    public abstract void iterate(Expression expression);

    /**
     * INTERNAL:
     */
    public void iterateOn(Vector expressions) {
        for (Enumeration expressionEnum = expressions.elements(); expressionEnum.hasMoreElements();) {
            iterate((Expression)expressionEnum.nextElement());
        }
    }

    /**
     * INTERNAL:
     * Return the call.
     */
    public void iterateOn(Expression expression) {
        expression.iterateOn(this);
    }

    public void setResult(Object result) {
        this.result = result;
    }

    public void setStatement(SQLSelectStatement statement) {
        this.statement = statement;
    }

    /**
     * Normally an Iterator will not go into the where clause of
     * an SQLSubSelectExpression.  I.e. when aliasing the parent statement
     * is aliased before the subselects may even be normalized.  An iterator to
     * alias the SubSelect must be run later.
     */
    public boolean shouldIterateOverSubSelects() {
        return false;
    }
}
