/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the "License").  You may not use this file except 
 * in compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * glassfish/bootstrap/legal/CDDLv1.0.txt or 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html. 
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * HEADER in each file and include the License file at 
 * glassfish/bootstrap/legal/CDDLv1.0.txt.  If applicable, 
 * add the following below this CDDL HEADER, with the 
 * fields enclosed by brackets "[]" replaced with your 
 * own identifying information: Portions Copyright [yyyy] 
 * [name of copyright owner]
 */
// Copyright (c) 1998, 2006, Oracle. All rights reserved.  
package oracle.toplink.essentials.internal.parsing;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import oracle.toplink.essentials.queryframework.*;
import oracle.toplink.essentials.expressions.Expression;
import oracle.toplink.essentials.expressions.ExpressionBuilder;
import oracle.toplink.essentials.descriptors.ClassDescriptor;

/**
 * INTERNAL
 * <p><b>Purpose</b>: Represent a SELECT
 * <p><b>Responsibilities</b>:<ul>
 * <li> Hold the distinct status
 * <li> Modify a query based on the contents
 *
 * The SELECT statement determines the return type of an EJBQL query.
 * The SELECT may also determine the distinct state of a query
 *
 * A SELECT can be one of the following:
 *  1. SELECT OBJECT(someObject)... This query will return a collection of objects
 *  2. SELECT anObject.anAttribute ... This will return a collection of anAttribute
 *  3. SELECT <aggregateFunction> ... This will return a single value
 *      The allowable aggregateFunctions are: AVG, COUNT, MAX, MIN, SUM
 *          SELECT AVG(emp.salary)... Returns the average of all the employees salaries
 *          SELECT COUNT(emp)... Returns a count of the employees
 *          SELECT COUNT(emp.firstName)... Returns a count of the employee's firstNames
 *          SELECT MAX(emp.salary)... Returns the maximum employee salary
 *          SELECT MIN(emp.salary)... Returns the minimum employee salary
 *          SELECT SUM(emp.salary)... Returns the sum of all the employees salaries
 *
 * </ul>
 *    @author Jon Driscoll
 *    @since TopLink 5.0
 */
public class SelectNode extends QueryNode {

    private List selectExpressions = new ArrayList();
    private boolean distinct =false;
    
    public SelectNode() {
    }

    /**
     * INTERNAL
     * Add an Order By Item to this node
     */
    private void addSelectExpression(Object theNode) {
        selectExpressions.add(theNode);
    }

    /** */
    public List getSelectExpressions() {
        return selectExpressions;
    }

    /** */
    public void setSelectExpressions(List exprs) {
        selectExpressions = exprs;
    }

    /** */
    public boolean usesDistinct() {
        return distinct;
    }
    
    /** */
    public void setDistinct(boolean distinct) {
        this.distinct = distinct;
    }

    /**
     * INTERNAL
     * Apply this node to the passed query
     */
    public void applyToQuery(DatabaseQuery theQuery, GenerationContext context) {
        ObjectLevelReadQuery readQuery = (ObjectLevelReadQuery)theQuery;
        if (selectExpressions.isEmpty()) {
            return;
        }

        //set the distinct state
        //BUG 3168673: Don't set distinct state if we're using Count
        if (!(isSingleSelectExpression() && getFirstSelectExpressionNode().isCountNode())) {
            // Set the distinct state for the query
            if (usesDistinct()) {
                getParseTree().setDistinctState(ObjectLevelReadQuery.USE_DISTINCT);
                readQuery.setDistinctState(ObjectLevelReadQuery.USE_DISTINCT);
            }
        }

        if (readQuery instanceof ReportQuery) {
            ReportQuery reportQuery = (ReportQuery)readQuery;
            reportQuery.returnWithoutReportQueryResult();
            if (isSingleSelectExpression() && 
                !getFirstSelectExpressionNode().isConstructorNode()) {
                reportQuery.returnSingleAttribute();
            }
        } 
        for (Iterator i = selectExpressions.iterator(); i.hasNext();) {
            Node node = (Node)i.next();
            node.applyToQuery(readQuery, context);            
        }

        //indicate on the query if "return null if primary key null"
        //This means we want nulls returned if we expect an outer join
        if (this.hasOneToOneSelected(context)) {
            readQuery.setProperty("return null if primary key is null", Boolean.TRUE);
        } else {
            readQuery.removeProperty("return null if primary key is null");
        }

    }

    /**
     * INTERNAL
     **/
    public boolean hasOneToOneSelected(GenerationContext context) {
        // Iterate the select expression and return true if one of it has a
        // oneToOne selected.
        for (Iterator i = selectExpressions.iterator(); i.hasNext();) {
            Node node = (Node)i.next();
            if (hasOneToOneSelected(node, context)) {
                return true;
            }
        }
        return false;
    }
    
    /**
     * INTERNAL
     * Answer true if there is a one-to-one relationship selected.
     * This includes a chain of relationships.
     * True: SELECT employee.address FROM ..... //Simple 1:1
     * True: SELECT a.b.c.d FROM ..... //where a->b, b->c and c->d are all 1:1.
     * False: SELECT OBJECT(employee) FROM ..... //simple SELECT
     * False: SELECT phoneNumber.areaCode FROM ..... //direct-to-field
     **/
    private boolean hasOneToOneSelected(Node node, GenerationContext context) {
        //BUG 3240484: Not SELECTing 1:1 if it's in a COUNT
        if (node.isCountNode()) {
            return false;
        }

        if (node.isAggregateNode()) {
            // delegate to aggregate expression
            return hasOneToOneSelected(node.getLeft(), context);
        }
         
        if (node.isVariableNode()){
            return !nodeRefersToObject(node, context);
        }
      
        // check whether it is a direct-to-field mapping
        return !selectingDirectToField(node, context);
    }

    /**
     * INTERNAL
     * Answer true if there is an attribute selected
     * True: SELECT employee.firstName FROM .....
     * False: SELECT OBJECT(employee) FROM .....
     **/
    public boolean hasAttributeSelected(Node node, GenerationContext context) {
        if (node == null) {
            return false;
        }

        //handle SELECT employee.firstName
        if (selectingDirectToField(node, context)) {
            return true;
        }

        //handle SELECT COUNT(employee.firstName) and Aggregates: MIN, MAX, ...
        if (node.isAggregateNode()) {
            return hasAttributeSelected(node.getLeft(), context);
        }
        return false;
    }

    /**
    * Verify that the selected alias is a valid alias. If it's not valid,
    * an Exception will be thrown, likely EJBQLException.aliasResolutionException.
    *
    * Valid: SELECT OBJECT(emp) FROM Employee emp WHERE ...
    * Invalid: SELECT OBJECT(badAlias) FROM Employee emp WHERE ...
    */
    public void verifySelectedAlias(GenerationContext context) {
        for (Iterator i = selectExpressions.iterator(); i.hasNext();) {
            Node node = (Node)i.next();
            //if the node is a DotNode, there is no selected alias
            if (node.isDotNode()) {
                return;
            }
            node.resolveClass(context);
        }
    }

    /**
    * Answer true if the variable name given as argument is SELECTed.
    *
    * True: "SELECT OBJECT(emp) ...." & variableName = "emp"
    * False: "SELECT OBJECT(somethingElse) ..." & variableName = "emp"
    */
    public boolean isSelected(String variableName) {
        for (Iterator i = selectExpressions.iterator(); i.hasNext();) {
            Node node = (Node)i.next();
            //Make sure we've SELECted a VariableNode
            if (node.isVariableNode() && 
                ((VariableNode)node).getCanonicalVariableName().equals(variableName)) {
                return true;
            }
        }
        return false;
    }

    public boolean isSelectNode() {
        return true;
    }

    /**
     * INTERNAL
     * Validate node.
     */
    public void validate(ParseTreeContext context) {
        for (Iterator i = selectExpressions.iterator(); i.hasNext(); ) {
            Node item = (Node)i.next();
            item.validate(context);
        }
    }

    /**
     * resolveClass: Answer the class associated with my left node.
     */
    public Class resolveClass(GenerationContext context) {
        return getReferenceClass(context);
    }
    
    /**
     * INTERNAL
     * Return a TopLink expression generated using the left node
     */
    public Expression generateExpression(GenerationContext context) {
        Expression expression = null;

        for (Iterator i = selectExpressions.iterator(); i.hasNext();) {
            Node node = (Node)i.next();

            //ignore SELECT OBJECT(emp) .....
            if (node.isVariableNode()) {
                //BUG 3105651: Make sure the variable is NOT part of an IN clause
                String variableNameForLeft = ((VariableNode)node).getCanonicalVariableName();
                if (context.getParseTreeContext().isRangeVariable(variableNameForLeft)) {
                    continue;
                }
            }
            
            //ignore SELECT phoneNumber.areaCode ....
            if (selectingDirectToField(node, context)) {
                continue;
            }

            //for AGGREGATE, ignore unless it's a COUNT containing a 1:1
            if (node.isAggregateNode()) {
                //BUG 3240484: COUNT never has an expression for the SELECT. This will
                //be handled by the Expression framework changes from BUG 3268040.
                //Therefore, all Aggregates can be treated the same way
                continue;
            }

            //if we're selecting a one-to-one, then indicate we want outer joins
            //i.e. employee.address => new ExpressionBuilder.getAllowingNull("address");
            //BUG 3264658: Don't use outer joins if we're COUNTing.
            if (!hasAttributeSelected(node, context) && (!node.isCountNode())) {
                ((SelectGenerationContext)context).useOuterJoins();
            }
            Expression generatedExpression = node.generateExpression(context);
            ((SelectGenerationContext)context).dontUseOuterJoins();
            if (node.isVariableNode()){

                Expression nodeExpression = context.selectedExpressionFor(((VariableNode)node).getCanonicalVariableName());
                if (nodeExpression != null){
                    if (nodeExpression.getBuilder().getQueryClass() == context.getBaseQueryClass()){
                        generatedExpression = nodeExpression.equal(generatedExpression);
                    } else {
                        generatedExpression = generatedExpression.equal(nodeExpression);
                    }
                }
            } else {
                continue;
            }
            if (expression == null) {
                expression = generatedExpression;
            } else {
                expression.and(generatedExpression);
            }
        }
        return expression;
    }

  /**
   * Compute the Reference class for this query 
   * @param context 
   * @return the class this query is querying for
   */
    public Class getReferenceClass(GenerationContext context) {
        return getClassOfFirstVariable(context);
    }
    
    /** */
    private Class getClassOfFirstVariable(GenerationContext context) {
        Class clazz = null;
        String variable = getParseTree().getFromNode().getFirstVariable();
        ParseTreeContext parseTreeContext = context.getParseTreeContext();
        if (parseTreeContext.isRangeVariable(variable)) {
            String schema = parseTreeContext.schemaForVariable(variable);
            // variables is defines in a range variable declaration, so there
            // is a schema name for this variable
            clazz = parseTreeContext.classForSchemaName(schema, context);
        } else {
            // variable is defined in a JOIN clause, so there is a a defining
            // node for the variable
            Node path = parseTreeContext.pathForVariable(variable);
            clazz = path.resolveClass(context);
        }
        return clazz;
    }

    /**
    * INTERNAL
    * Answer true if a variable in the IN clause is SELECTed
    */
    public boolean isVariableInINClauseSelected(GenerationContext context) {
        for (Iterator i = selectExpressions.iterator(); i.hasNext();) {
            Node node = (Node)i.next();
        
            if (node.isVariableNode()) {
                String variableNameForLeft = ((VariableNode)node).getCanonicalVariableName();
                if (!context.getParseTreeContext().isRangeVariable(variableNameForLeft)) {
                    return true;
                }
            }
        }
        return false;
    }

    /**
     * INTERNAL
     * Answer true if this node refers to an object described later in the EJBQL
     * True: SELECT p FROM Project p
     * False: SELECT p.id FROM Project p
     **/
    public boolean nodeRefersToObject(Node node, GenerationContext context) {
        if (!node.isVariableNode()){
            return false;
        }
        String name = ((VariableNode)node).getCanonicalVariableName();
        String alias = context.getParseTreeContext().schemaForVariable(name);
        if (alias != null){
            ClassDescriptor descriptor = context.getSession().getDescriptorForAlias(alias);
            if (descriptor != null){
                return true;
            }
        }
        return false;
    }

    /**
     * INTERNAL
     * Answer true if the SELECT ends in a direct-to-field.
     * true: SELECT phone.areaCode
     * false: SELECT employee.address
     */
    private boolean selectingDirectToField(Node node, GenerationContext context) {

        if ((node == null) || !node.isDotNode()) {     
            return false;
        }
        return ((DotNode)node).endsWithDirectToField(context);
    }

    /** 
     * Returns the first select expression node.
     */
    private Node getFirstSelectExpressionNode() {
        return selectExpressions.size() > 0 ? 
            (Node)selectExpressions.get(0) : null;
    }

    /** */
    private boolean isSingleSelectExpression() {
        return selectExpressions.size() == 1;
    }
    
    
}
