/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the "License").  You may not use this file except 
 * in compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * glassfish/bootstrap/legal/CDDLv1.0.txt or 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html. 
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * HEADER in each file and include the License file at 
 * glassfish/bootstrap/legal/CDDLv1.0.txt.  If applicable, 
 * add the following below this CDDL HEADER, with the 
 * fields enclosed by brackets "[]" replaced with your 
 * own identifying information: Portions Copyright [yyyy] 
 * [name of copyright owner]
 */
// Copyright (c) 1998, 2006, Oracle. All rights reserved.  
package oracle.toplink.essentials.internal.parsing;

import java.lang.reflect.Field;

import oracle.toplink.essentials.descriptors.ClassDescriptor;
import oracle.toplink.essentials.mappings.AttributeAccessor;
import oracle.toplink.essentials.mappings.AggregateMapping;
import oracle.toplink.essentials.mappings.DatabaseMapping;
import oracle.toplink.essentials.internal.helper.Helper;
import oracle.toplink.essentials.internal.sessions.AbstractSession;
import oracle.toplink.essentials.internal.security.PrivilegedAccessController;

/**
 * INTERNAL
 * <p><b>Purpose</b>: Implement type helper methods specified by TypeHelper.
 * This implementation uses Class instances to represent a type.
 */
public class TypeHelperImpl 
    extends BasicTypeHelperImpl implements TypeHelper {

    /** The session. */
    private final AbstractSession session;

    /** The class loader used to resolve type names. */
    private final ClassLoader classLoader;

    /** */
    public TypeHelperImpl(AbstractSession session, ClassLoader classLoader) {
        this.session = session;
        this.classLoader = classLoader;
    }
    
    /** */
    public Object resolveTypeName(String typeName) {
        try {
            return PrivilegedAccessController.getClassForName(
                typeName, true, classLoader);
        } catch (ClassNotFoundException ex) {
            return null;
        }
    }

    /** */
    public Object resolveAttribute(Object ownerClass, String attribute) {
        Class clazz = (Class)ownerClass;
        DatabaseMapping mapping = resolveAttributeMapping((Class)ownerClass, attribute);
        Object type = getType(mapping);
        if ((type == null) && (mapping != null)) {
            // mapping might not be initialized, so the type returned from the
            // mapping might be null => check directly
            try {
                Field field = Helper.getField(clazz, attribute);
                if (field != null) {
                    type = field.getType();
                }
            } catch (NoSuchFieldException ex) {
                // ignore, return null
            }
        }
        return type;
    }
    
    /** */
    public Object resolveSchema(String schemaName) {
        ClassDescriptor descriptor = session.getDescriptorForAlias(schemaName);
        return (descriptor != null) ? descriptor.getJavaClass() : null;
    }

    /** */
    public String getTypeName(Object type) {
        if (type == null) {
            return null;
        }
        return ((Class)type).getName();
    }

    /** */
    public Object resolveEnumConstant(Object type, String constant) {
        Class clazz = (Class)type;
        Object[] constants = clazz.getEnumConstants();
        if (constants != null) {
            for (int i = 0; i < constants.length; i++) {
                if (constant.equals(constants[i].toString())) {
                    return constants[i];
                }
            }
        }
        return null;
    }

    // ===== Internal helper methods =====

    /** */
    private DatabaseMapping resolveAttributeMapping(Class ownerClass, String attribute) {
        ClassDescriptor descriptor = session.getDescriptor(ownerClass);
        return (descriptor==null) ? null : descriptor.getMappingForAttributeName(attribute);
    }

    /** */
    private Object getType(DatabaseMapping mapping) {
        if (mapping == null) {
            return null;
        }
        Object type = null;
        if (mapping.isForeignReferenceMapping()) {
            ClassDescriptor descriptor = mapping.getReferenceDescriptor();
            type = descriptor == null ? null : descriptor.getJavaClass();
        } else if (mapping.isAggregateMapping()) {
            type = ((AggregateMapping)mapping).getReferenceClass();
        } else {
            type = mapping.getAttributeClassification();
        }
        return type;
    }

}
