/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the "License").  You may not use this file except 
 * in compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * glassfish/bootstrap/legal/CDDLv1.0.txt or 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html. 
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * HEADER in each file and include the License file at 
 * glassfish/bootstrap/legal/CDDLv1.0.txt.  If applicable, 
 * add the following below this CDDL HEADER, with the 
 * fields enclosed by brackets "[]" replaced with your 
 * own identifying information: Portions Copyright [yyyy] 
 * [name of copyright owner]
 */
// Copyright (c) 1998, 2005, Oracle. All rights reserved.  
package oracle.toplink.essentials.sessions;

import java.util.*;
import oracle.toplink.essentials.expressions.*;
import oracle.toplink.essentials.exceptions.*;
import oracle.toplink.essentials.queryframework.*;
import oracle.toplink.essentials.internal.sessions.AbstractRecord;

/**
 * PUBLIC:
 * IdentityMapAccessor provides the public interface into all functionality associated with
 * TopLink identity maps.
 * As appropriate IdentityMapAccessor can be obtained from a session with it's getIdentityMapAccessor()
 * method.
 * Methods that used to be called on the Session to access identity maps can now be called
 * through the IdentityMapAccessor.
 * For instance, to initialize identity maps the code used to be:
 * <code>session.initializeIdentityIdentityMaps()</code>
 * The code now is:
 * <code>session.getIdentityMapAccessor().initializeIdentityMaps()</code>
 * @see oracle.toplink.essentials.sessions.Session
 */
public interface IdentityMapAccessor {

    /**
     * ADVANCED:
     * Return if there is an object for the primary key.
     */
    public boolean containsObjectInIdentityMap(Object domainObject);

    /**
     * ADVANCED:
     * Return if there is an object for the primary key.
     */
    public boolean containsObjectInIdentityMap(Vector primaryKey, Class theClass);

    /**
     * ADVANCED:
     * Return if there is an object for the row containing primary key and the class.
     */
    public boolean containsObjectInIdentityMap(Record rowContainingPrimaryKey, Class theClass);

    /**
     * ADVANCED:
     * Query the cache in-memory.
     * If the expression is too complex an exception will be thrown.
     * Only return objects that are invalid in the cache if specified.
     */
    public Vector getAllFromIdentityMap(Expression selectionCriteria, Class theClass, Record translationRow, InMemoryQueryIndirectionPolicy valueHolderPolicy, boolean shouldReturnInvalidatedObjects) throws QueryException;

    /**
     * ADVANCED:
     * Query the cache in-memory.
     * If the expression is too complex an exception will be thrown.
     */
    public Vector getAllFromIdentityMap(Expression selectionCriteria, Class theClass, Record translationRow, InMemoryQueryIndirectionPolicy valueHolderPolicy) throws QueryException;

    /**
     * ADVANCED:
     * Return the object from the identity with primary and class of the given object.
     */
    public Object getFromIdentityMap(Object domainObject);

    /**
     * ADVANCED:
     * Return the object from the identity with the primary and class.
     */
    public Object getFromIdentityMap(Vector primaryKey, Class theClass);

    /**
     * ADVANCED:
     * Return the object from the identity with the primary and class.
     */
    public Object getFromIdentityMap(Record rowContainingPrimaryKey, Class theClass);

    /**
     * ADVANCED:
     * Return the object from the identity with the primary and class.
     * Only return invalidated objects if requested.
     */
    public Object getFromIdentityMap(Vector primaryKey, Class theClass, boolean shouldReturnInvalidatedObjects);

    /**
     * ADVANCED:
     * Return the object from the identity with the primary and class.
     * Only return invalidated objects if requested.
     */
    public Object getFromIdentityMap(Record rowContainingPrimaryKey, Class theClass, boolean shouldReturnInvalidatedObjects);

    /**
     * ADVANCED:
     * Query the cache in-memory.
     * If the object is not found null is returned.
     * If the expression is too complex an exception will be thrown.
     */
    public Object getFromIdentityMap(Expression selectionCriteria, Class theClass, Record translationRow) throws QueryException;

    /**
     * ADVANCED:
     * Query the cache in-memory.
     * If the object is not found null is returned.
     * If the expression is too complex an exception will be thrown.
     */
    public Object getFromIdentityMap(Expression selectionCriteria, Class theClass, Record translationRow, InMemoryQueryIndirectionPolicy valueHolderPolicy) throws QueryException;

    /**
     * ADVANCED:
     * Return the remaining life of this object.  This method is associated with use of
     * TopLink's cache invalidation feature and returns the difference between the next expiry
     * time of the object and its read time.  The method will return 0 for invalidated objects.
     */
    public long getRemainingValidTime(Object object);

    /**
     * ADVANCED:
     * Extract the write lock value from the identity map.
     */
    public Object getWriteLockValue(Object domainObject);

    /**
     * ADVANCED:
     * Extract the write lock value from the identity map.
     */
    public Object getWriteLockValue(Vector primaryKey, Class theClass);

    /**
     * PUBLIC:
     * Reset the entire object cache.
     * <p> NOTE: be careful using this method. This method blows away both this session's and its parents caches,
     * this includes the server cache or any other cache. This throws away any objects that have been read in.
     * Extream caution should be used before doing this because object identity will no longer
     * be maintained for any objects currently read in.  This should only be called
     * if the application knows that it no longer has references to object held in the cache.
     */
    public void initializeAllIdentityMaps();

    /**
    * PUBLIC:
    * Reset the identity map for only the instances of the class.
    * For inheritance the user must make sure that they only use the root class.
    * Caution must be used in doing this to ensure that the objects within the identity map
    * are not referenced from other objects of other classes or from the application.
    */
    public void initializeIdentityMap(Class theClass);

    /**
     * PUBLIC:
     * Reset the entire local object cache.
     * This throws away any objects that have been read in.
     * Extream caution should be used before doing this because object identity will no longer
     * be maintained for any objects currently read in.  This should only be called
     * if the application knows that it no longer has references to object held in the cache.
     */
    public void initializeIdentityMaps();

    /**
     * ADVANCED:
     * Set an object to be invalid in the TopLink identity maps.
     * If the object does not exist in the cache, this method will return
     * without any action
     */
    public void invalidateObject(Object object);

    /**
     * ADVANCED:
     * Set an object to be invalid in the TopLink identity maps.
     * If the object does not exist in the cache, this method will return
     * without any action
     */
    public void invalidateObject(Vector primaryKey, Class theClass);

    /**
     * ADVANCED:
     * Set an object to be invalid in the TopLink identity maps.
     * If the object does not exist in the cache, this method will return
     * without any action
     */
    public void invalidateObject(Record rowContainingPrimaryKey, Class theClass);

    /**
     * ADVANCED:
     * Set all of the objects in the given collection to be invalid in the TopLink Identity Maps
     * This method will take no action for any objects in the collection that do not exist in the cache.
     */
    public void invalidateObjects(Vector collection);

    /**
     * ADVANCED:
     * Set all of the objects from the given Expression to be invalid in the TopLink Identity Maps
     */
    public void invalidateObjects(Expression selectionCriteria);

    /**
     * ADVANCED:
     * Set all of the objects for all classes to be invalid in TopLink's 
     * identity maps. It will recurse on inheritance.
     */
    public void invalidateAll();
    
    /**
     * ADVANCED:
     * Set all of the objects of a specific class to be invalid in TopLink's identity maps
     * Will set the recurse on inheritance to true.
     */
    public void invalidateClass(Class myClass);

    /**
       * ADVANCED:
       * Set all of the objects of a specific class to be invalid in TopLink's identity maps.
     * User can set the recurse flag to false if they do not want to invalidate
     * all the classes within an inheritance tree.
       */
    public void invalidateClass(Class myClass, boolean recurse);

    /**
     * ADVANCED:
     * Return true if this object is valid in TopLink's identity maps
     * return false otherwise
     */
    public boolean isValid(Object object);

    /**
     * ADVANCED:
     * Return true if this object is valid in TopLink's identity maps
     * return false otherwise
     */
    public boolean isValid(Vector primaryKey, Class theClass);

    /**
     * ADVANCED:
     * Return true if this object is valid in TopLink's identity maps
     * return false otherwise
     */
    public boolean isValid(AbstractRecord rowContainingPrimaryKey, Class theClass);

    /**
     * PUBLIC:
     * Used to print all the objects in the identity map of the passed in class.
     * The output of this method will be logged to this session's SessionLog at SEVERE level.
     */
    public void printIdentityMap(Class businessClass);

    /**
     * PUBLIC:
     * Used to print all the objects in every identity map in this session.
     * The output of this method will be logged to this session's SessionLog at SEVERE level.
     */
    public void printIdentityMaps();

    /**
     * PUBLIC:
     * Used to print all the locks in every identity map in this session.
     * The output of this method will be logged to this session's SessionLog at FINEST level.
     */
    public void printIdentityMapLocks();

    /**
     * ADVANCED:
     * Register the object with the identity map.
     * The object must always be registered with its version number if optimistic locking is used.
     */
    public Object putInIdentityMap(Object domainObject);

    /**
     * ADVANCED:
     * Register the object with the identity map.
     * The object must always be registered with its version number if optimistic locking is used.
     */
    public Object putInIdentityMap(Object domainObject, Vector key);

    /**
     * ADVANCED:
     * Register the object with the identity map.
     * The object must always be registered with its version number if optimistic locking is used.
     */
    public Object putInIdentityMap(Object domainObject, Vector key, Object writeLockValue);

    /**
     * ADVANCED:
     * Register the object with the identity map.
     * The object must always be registered with its version number if optimistic locking is used.
     * The readTime may also be included in the cache key as it is constructed
     */
    public Object putInIdentityMap(Object domainObject, Vector key, Object writeLockValue, long readTime);

    /**
    * ADVANCED:
    * Remove the object from the object cache.
    * Caution should be used when calling to avoid violating object identity.
    * The application should only call this is it knows that no references to the object exist.
    */
    public Object removeFromIdentityMap(Object domainObject);

    /**
     * ADVANCED:
     * Remove the object from the object cache.
     */
    public Object removeFromIdentityMap(Vector key, Class theClass);

    /**
    * ADVANCED:
    * Update the write lock value in the identity map.
    */
    public void updateWriteLockValue(Object domainObject, Object writeLockValue);

    /**
     * ADVANCED:
     * Update the write lock value in the identity map.
     */
    public void updateWriteLockValue(Vector primaryKey, Class theClass, Object writeLockValue);

    /**
     * ADVANCED:
     * This can be used to help debugging an object identity problem.
     * An object identity problem is when an object in the cache references an object not in the cache.
     * This method will validate that all cached objects are in a correct state.
     */
    public void validateCache();
}
