/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */
 
package com.sun.gjc.spi;

import java.io.PrintWriter;
import java.sql.Connection;
import java.sql.SQLException;
import javax.resource.spi.ManagedConnectionFactory;
import javax.resource.spi.ConnectionManager;
import javax.resource.ResourceException;
import javax.naming.Reference;
import com.sun.gjc.spi.ConnectionHolder.ConnectionType;

import com.sun.logging.*;
import java.util.logging.Logger;
import java.util.logging.Level;
import com.sun.enterprise.connectors.ConnectionManagerImpl;
import com.sun.enterprise.connectors.ConnectorConstants;

/**
 * Holds the <code>java.sql.Connection</code> object, which is to be 
 * passed to the application program.
 *
 * @version	1.0, 02/07/31
 * @author	Binod P.G
 */
public class DataSource implements javax.sql.DataSource, java.io.Serializable,
		com.sun.appserv.jdbc.DataSource, javax.resource.Referenceable{ 
				   
    private ManagedConnectionFactory mcf;
    private ConnectionManager cm;				   				   
    private int loginTimeout;
    private PrintWriter logWriter;
    private String description;				   
    private Reference reference;
    
    private ConnectionType conType_;

    private static Logger _logger;
    static {
        _logger = LogDomains.getLogger( LogDomains.RSR_LOGGER );
    }
    private boolean debug = false;

    /**
     * Constructs <code>DataSource</code> object. This is created by the 
     * <code>ManagedConnectionFactory</code> object.
     *
     * @param	mcf	<code>ManagedConnectionFactory</code> object 
     *			creating this object.
     * @param	cm	<code>ConnectionManager</code> object either associated
     *			with Application server or Resource Adapter.
     */
    public DataSource (ManagedConnectionFactory mcf, ConnectionManager cm) {
    	this.mcf = mcf;
    	if (cm == null) {
    	    this.cm = (ConnectionManager) new com.sun.gjc.spi.ConnectionManager();
    	} else {
    	    this.cm = cm;
            conType_ = findConnectionType();
    	}    	
    }   
    
    /**
     * Retrieves the <code> Connection </code> object.
     * 
     * @return	<code> Connection </code> object.
     * @throws SQLException In case of an error.      
     */
    public Connection getConnection() throws SQLException {
    	try {
            ConnectionHolder con = (ConnectionHolder) 
                cm.allocateConnection( mcf, null );
            setConnectionType( con );

    	    return (Connection) con;
    	} catch (ResourceException re) {
	    _logger.log(Level.WARNING, "jdbc.exc_get_conn", re.getMessage());
    	    throw new SQLException (re.getMessage());
    	}
    }
    
    /**
     * Retrieves the <code> Connection </code> object.
     * 
     * @param	user	User name for the Connection.
     * @param	pwd	Password for the Connection.
     * @return	<code> Connection </code> object.
     * @throws SQLException In case of an error.      
     */
    public Connection getConnection(String user, String pwd) throws SQLException {
    	try {
    	    ConnectionRequestInfo info = new ConnectionRequestInfo (user, pwd);
    	    ConnectionHolder con = (ConnectionHolder)
                cm.allocateConnection(mcf,info);
            setConnectionType( con );
            return (Connection) con;
    	} catch (ResourceException re) {
	    _logger.log(Level.WARNING, "jdbc.exc_get_conn", re.getMessage());
    	    throw new SQLException (re.getMessage());
    	}
    }    

    /**
     * Retrieves the actual SQLConnection from the Connection wrapper
     * implementation of SunONE application server. If an actual connection is
     * supplied as argument, then it will be just returned.
     *
     * @param con Connection obtained from <code>Datasource.getConnection()</code>
     * @return <code>java.sql.Connection</code> implementation of the driver.
     * @throws <code>java.sql.SQLException</code> If connection cannot be obtained.
     */
    public Connection getConnection(Connection con) throws SQLException {

        Connection driverCon = con; 
        if (con instanceof com.sun.gjc.spi.ConnectionHolder) {
           driverCon = ((com.sun.gjc.spi.ConnectionHolder) con).getConnection(); 
        } 

        return driverCon;
    }
   
    /**
     * Gets a connection that is not in the scope of any transaction. This
     * can be used to save performance overhead incurred on enlisting/delisting
     * each connection got, irrespective of whether its required or not.
     * Note here that this meethod does not fit in the connector contract
     * per se.
     *
     * @return <code>java.sql.Connection</code>
     * @throws <code>java.sql.SQLException</code> If connection cannot be obtained
     */
    public Connection getNonTxConnection() throws SQLException {
        try {
            ConnectionHolder con = (ConnectionHolder)
                ((com.sun.enterprise.connectors.ConnectionManagerImpl)
                cm).allocateNonTxConnection(mcf, null);
            setConnectionType( con, true );
            
            return (Connection) con;
        } catch( ResourceException re ) {
            _logger.log(Level.WARNING, "jdbc.exc_get_conn", re.getMessage());
            throw new SQLException( re.getMessage() );
        }
    }

    /**
     * Gets a connection that is not in the scope of any transaction. This
     * can be used to save performance overhead incurred on enlisting/delisting
     * each connection got, irrespective of whether its required or not.
     * Note here that this meethod does not fit in the connector contract
     * per se.
     *
     * @param user User name for authenticating the connection
     * @param password Password for authenticating the connection
     * @return <code>java.sql.Connection</code>
     * @throws <code>java.sql.SQLException</code> If connection cannot be obtained
     */
    public Connection getNonTxConnection(String user, String password) throws SQLException {
        try {
            ConnectionRequestInfo cxReqInfo = new ConnectionRequestInfo(user, password);
            ConnectionHolder con = (ConnectionHolder)
                ((com.sun.enterprise.connectors.ConnectionManagerImpl)
	        cm).allocateNonTxConnection( mcf, cxReqInfo);

            setConnectionType( con, true );
            
            return (Connection) con;
        } catch( ResourceException re ) {
            _logger.log(Level.WARNING, "jdbc.exc_get_conn", re.getMessage());
            throw new SQLException( re.getMessage() );
        }
    }
   
    /**
     * Get the login timeout
     *
     * @return login timeout.
     * @throws	SQLException	If a database error occurs.
     */
    public int getLoginTimeout() throws SQLException{
    	return	loginTimeout;
    }
    
    /**
     * Set the login timeout
     *
     * @param	loginTimeout	Login timeout.
     * @throws	SQLException	If a database error occurs.
     */
    public void setLoginTimeout(int loginTimeout) throws SQLException{
    	this.loginTimeout = loginTimeout;
    }
    
    /** 
     * Get the logwriter object.
     *
     * @return <code> PrintWriter </code> object.
     * @throws	SQLException	If a database error occurs.
     */
    public PrintWriter getLogWriter() throws SQLException{
    	return	logWriter;
    }
        
    /**
     * Set the logwriter on this object.
     *
     * @param <code>PrintWriter</code> object.
     * @throws	SQLException	If a database error occurs.
     */
    public void setLogWriter(PrintWriter logWriter) throws SQLException{
    	this.logWriter = logWriter;
    }        
        
    /**
     * Retrieves the description.
     *
     * @return	Description about the DataSource.
     */
    public String getDescription() {
    	return description;
    }
    
    /**
     * Set the description.
     *
     * @param description Description about the DataSource.
     */
    public void setDescription(String description) {
    	this.description = description;
    }    
    
    /**
     * Get the reference.
     *
     * @return <code>Reference</code>object.
     */
    public Reference getReference() {
    	return reference;
    }
    
    /**
     * Get the reference.
     *
     * @param	reference <code>Reference</code> object.
     */
    public void setReference(Reference reference) {
    	this.reference = reference;
    }   

    private ConnectionType findConnectionType() {
        ConnectionType cmType = ConnectionType.STANDARD;
        
        if ( cm instanceof javax.resource.spi.LazyAssociatableConnectionManager ) {
            if (! ((com.sun.enterprise.connectors.ConnectionManagerImpl)cm).
                    getJndiName().endsWith( ConnectorConstants.PM_JNDI_SUFFIX ) ) {
                cmType = ConnectionType.LAZY_ASSOCIATABLE;
            }
        } else if ( cm instanceof 
                javax.resource.spi.LazyEnlistableConnectionManager ) {
            if (! ((com.sun.enterprise.connectors.ConnectionManagerImpl)cm).
                    getJndiName().endsWith( ConnectorConstants.PM_JNDI_SUFFIX) &&
                ! ((com.sun.enterprise.connectors.ConnectionManagerImpl)cm).
                    getJndiName().endsWith( ConnectorConstants.NON_TX_JNDI_SUFFIX)) {
                cmType = ConnectionType.LAZY_ENLISTABLE;
            }
        }

        return cmType;
    }

    private void setConnectionType( ConnectionHolder con ) {
        this.setConnectionType( con, false);
    }
    
    private void setConnectionType( ConnectionHolder con, boolean isNonTx ) {
        con.setConnectionType( conType_ );
        if ( conType_ == ConnectionType.LAZY_ASSOCIATABLE ) {
            con.setLazyAssociatableConnectionManager( 
                (javax.resource.spi.LazyAssociatableConnectionManager) cm);
        } else if (conType_ == ConnectionType.LAZY_ENLISTABLE) {
            if ( isNonTx ) {
                //if this is a getNonTxConnection call on the DataSource, we
                //should not LazyEnlist
                con.setConnectionType( ConnectionType.STANDARD );
            } else {
                con.setLazyEnlistableConnectionManager(
                    (javax.resource.spi.LazyEnlistableConnectionManager) cm );
            }
        }
    }

}
