/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */
package com.sun.appserv.management.client.prefs;

import java.util.Arrays;

/** A factory class to create instances of LoginInfoStore.
 * @since Appserver 9.0
 */
public class LoginInfoStoreFactory {
    
    /** Private constructor.
     */
    private LoginInfoStoreFactory() {
    }
    
    /** Returns the store that is represented by given class name. The parameter must
     * implement the {@link LoginInfoStore} interface. If a null is passed, an instance of the default
     * store {@link MemoryHashLoginInfoStore} is returned.
     * @param storeImplClassName fully qualified name of the class implementing LoginInfoStore. May be null.
     * @return the instance of LoginInfoStore of your choice
     * @throws IllegalArgumentException if the parameter does not implement LoginInfoStore
     * @throws StoreException if the construction of default store results in problems
     * @throws ClassNotFoundException if the given class could not be loaded
     */
    public static LoginInfoStore getStore(final String storeImplClassName) 
        throws StoreException, ClassNotFoundException, IllegalAccessException, InstantiationException {
        LoginInfoStore store = null;
        if (storeImplClassName == null)
            store = getDefaultStore();
        else 
            store = getCustomStore(storeImplClassName);
        return ( store );
    }
    
    private static LoginInfoStore getDefaultStore() throws StoreException {
        return ( new MemoryHashLoginInfoStore() );
    }
    
    private static LoginInfoStore getCustomStore(final String icn) 
        throws ClassNotFoundException, IllegalAccessException, InstantiationException{
        final Class ic  = Class.forName(icn);
        final String in = LoginInfoStore.class.getName();
        if (ic == null || !isStore(ic))
            throw new IllegalArgumentException("Class: " + ic.getName() + " does not implement: " + in);
        final LoginInfoStore store = (LoginInfoStore) ic.newInstance();
        return ( store );
    }
    
    private static boolean isStore(final Class c) {
        final Class[] ifs = c.getInterfaces();
        final Class sc    = LoginInfoStore.class;
        return ( Arrays.asList(ifs).contains(sc) );
    }
}