/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */
package com.sun.appserv.management.util.misc;

import java.util.Set;
import java.util.HashSet;
import java.util.Collection;
import java.util.Collections;
import java.util.Arrays;


/**
    Utilities for working with sets using JDK 1.5 generics.
 */
public final class GSetUtil
{
		private
	GSetUtil( )
	{
		// disallow instantiation
	}
	
		public static <T> T
	getSingleton( final Set<T> s )
	{
		if ( s.size() != 1 )
		{
			throw new IllegalArgumentException( s.toString() );
		}
		return( s.iterator().next() );
	}
	
		public static <T> void
	addArray(
		final Set<T>	set,
		final T[]	    array )
	{
		for( final T item : array )
		{
			set.add( item );
		}
	}
	
		public static <T> Set<T>
	newSet( final Collection<T> c )
	{
		final HashSet<T>	set	= new HashSet<T>();
		
		set.addAll( c );
		
		return( set );
	}
	
	
	/**
		Create a new Set with one member.
	 */
		public static <T> Set<T>
	newSet( final T item )
	{
		final Set<T>	set	= new HashSet<T>();
		set.add( item );
		
		return( set );
	}
	
	/**
		Create a new Set containing all members of another.
		The returned Set is always a HashSet.
	 */
		public static <T> HashSet<T>
	copySet( final Set<? extends T> s1 )
	{
		final HashSet<T>	set	= new HashSet<T>();
		
		set.addAll( s1 );
		
		return( set );
	}
	
	
		public static <T> Set<? extends T>
	newSet(
		final T m1,
		final T m2 )
	{
		final HashSet<T>	set	= new HashSet<T>();
		
		set.add( m1 );
		set.add( m2 );
		
		return( set );
	}
	
	/*
		public static <T> Set<T>
	newSet(
		final T m1,
		final T m2, 
		final T m3 )
	{
		final HashSet<T>	set	= new HashSet<T>();
		
		set.add( m1 );
		set.add( m2 );
		set.add( m3 );
		
		return( set );
	}
	*/
	
		public static <T> Set<T>
	newSet(
		final T m1,
		final T m2, 
		final T m3, 
		final T m4 )
	{
		final HashSet<T>	set	= new HashSet<T>();
		
		set.add( m1 );
		set.add( m2 );
		set.add( m3 );
		set.add( m4 );
		
		return( set );
	}
	
	
	/**
		Create a new Set containing all array elements.
	 */
		public static <T> Set<T>
	newSet( final T[]  objects )
	{
		return( newSet( objects, 0, objects.length ) );
	}
	
		public static <T,TT extends T> Set<T>
	newSet( final Set<T> s1, final Set<TT> s2 )
	{
	    final Set<T>    both    = new HashSet<T>();
	    both.addAll( s1 );
	    both.addAll( s2 );
	    
	    return both;
	}
	


	/**
		Create a new Set containing all array elements.
	 */
		public static <T> Set<T>
	newSet(
		final T[]   objects,
		final int   startIndex,
		final int   numItems )
	{
		final Set<T>	set	= new HashSet<T>();
		
		for( int i = 0; i < numItems; ++i )
		{
			set.add( objects[ startIndex + i ] );
		}

		return( set );
	}
	
    /**
		Convert a Set to a String[]
	 */
		public static String[]
	toStringArray( final Set<?>	s )
	{
		final String[]	strings	= new String[ s.size() ];
		
		int	i = 0;
		for( final Object o : s )
		{
			strings[ i ]	= "" + o;
			++i;
		}
		
		return( strings );
	}
	
		public static String[]
	toSortedStringArray( final Set<?>	s )
	{
		final String[]	strings	= toStringArray( s );
		
		Arrays.sort( strings );
		
		return( strings );
	}
	
	    public static Set<String>
	newStringSet( final String... args)
	{
	    return newUnmodifiableSet( args );
	}
	
	
	    public static <T> Set<T>
	newUnmodifiableSet( final T... args)
	{
	    final Set<T>   set   = new HashSet<T>();
	    
	    for( final T s : args )
	    {
	        set.add( s );
	    }
	    return set;
	}
	
	    public static Set<String>
	newUnmodifiableStringSet( final String... args)
	{
	    return Collections.unmodifiableSet( newStringSet( args ) );
	}
	
	    public static Set<String>
	newStringSet( final Object... args)
	{
	    final Set<String>   set   = new HashSet<String>();
	    
	    for( final Object o : args )
	    {
	        set.add( o == null ? null : "" + o );
	    }
	    return set;
	}
	
	
	/**
		Create a new Set with one member.  Additional items
		may be added.
	 */
		public static <T> Set<T>
	newSingletonSet( final T m1 )
	{
		final Set<T>	set	= new HashSet<T>();
		
		set.add( m1 );
		
		return( set );
	}
	
	
	/**
		Return a new Set of all items in both set1 and set2.
	 */
		public static <T> Set<T>
	intersectSets(
		final Set<T>	set1,
		final Set<T> set2 )
	{
		final Set<T>	result	= newSet( set1 );
		result.retainAll( set2 );
		
		return( result );
	}
	
	
	/**
		Return a new Set of all items in set1 not in set2.
	 */
		public static <T> Set<T>
	removeSet(
		final Set<T>	 set1,
		final Set<T>   set2 )
	{
		final Set<T>	result	= SetUtil.newSet( set1 );
		result.removeAll( set2 );
		
		return( result );
	}
	
	
	/**
		Return a new Set of all items not common to both sets.
	 */
		public static <T> Set<T>
	newNotCommonSet(
		final Set<T> set1,
		final Set<T> set2 )
	{
		final Set<T>	result	= newSet( set1, set2 );
		final Set<T>	common	= intersectSets( set1, set2);
		
		result.removeAll( common );
		
		return( result );
	}
}























