/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */
 
package com.sun.enterprise.tools.admingui;

import org.xml.sax.EntityResolver;
import org.xml.sax.InputSource;

import java.io.Serializable;
import java.io.InputStream;
import java.net.URLDecoder;

import com.iplanet.jato.util.NonSyncStringBuffer;
import com.sun.web.ui.common.CCSystem;


/**
 * This class implements the default entity resolver for common components.
 *
 * This entity resolver class specifically handles external references
 * in the XML files used by the common component internal servlets.
 * Typically, these are system identifier references to the DTD files
 * for the component XML definitions.  The class handles system identifier
 * references with the following design patterns in the DOCTYPE element:
 * <.p>
 * "com_sun_web_ui/dtd/name.dtd"
 * <.bk>
 * "/com_sun_web_ui/dtd/name.dtd"
 * <.bk>
 * "tags/dtd/name.dtd"
 * <.bk>
 * "/tags/dtd/name.dtd"
 * <.bk>
 * "app_name/dtd/name.dtd"
 * <.bk>
 * "/app_name/dtd/name.dtd"
 * <.p>
 * Where "app_name" is the context path name of a web application and
 * "name.dtd" is a DTD file name.
 * <.p>
 * Note that some parsers will attempt to resolve a reference
 * of this form (without a scheme prefix) as a file system relative
 * path name, so will add the "file://" scheme prefix to the reference.
 * If the system identifier reference does not begin with a slash
 * character, the parser may also prepend the directory specified in
 * the System property "user.dir".  If either prefix is encountered,
 * this class will remove the prefix and prepend the "http://host:port/"
 * to the reference, where "host" is the virtual server host name and
 * "port" is the console non-secure port number.
 *
 * @version 1.10 05/10/04
 * @author  Sun Microsystems, Inc.
 */
public class LockhartEntityResolver implements EntityResolver, Serializable {

    // Define constants
    private static final String COM_SUN_WEB_UI = "com_sun_web_ui";
    private static final String DTD_DIR = "dtd";
    private static final String DEFAULT_XML_ENCODING = "UTF-8";

    // Set XML file encoding to its expected default.  Really should get
    // this value from the parser, who reads it from the XML file itself.
    // For now, we set it to the recommended (by W3C and Lockhart) encoding.
    private static String xmlEncoding = DEFAULT_XML_ENCODING;

    /**
     * Default constructor
     * Set the port number by reading the appropriate System property.
     * Set the host name from the requset object made available by 
     * the Jato RequestManager. 
     */

    /**
     * Resolve an external entity. If external entity cannot be resolved 
     * return null, so that the parser will use the system identifier 
     * provided in the XML document. This method is being overriden to 
     * supply the tag DTDs from the com_sun_web_ui web application that 
     * will be part of the 2.1 installation. The DTDs will be accessed
     * thru the non secure console port similar to help. For now the port 
     * is being set as a System parameter by the start up script.
     *
     * @param publicId	The public identifier of the external reference
     * @param systemId	The system identifier of the external reference
     *
     * @return An InputSource object encapsulating the fixed reference
     */

    public InputSource resolveEntity(String publicId, String systemId) {

	// Fix the system identifier for an XML document.
	// Remove any "file://" prefix from the system identifier
	// and optionally the working directory for the JVM.
	// Some parsers prefix "file://<working_directory>" to the
	// system identifier in a misguided attempt to use file I/O
	// to find the dtd file.  Remove the working directory path.
	// Handle a DOCTYPE tag with the following SYSTEM value syntax
	// after removing the file prefix and working directory path:
	//
	// "filename.dtd"			(dynamically generated files)
	// "/tags/dtd/filename.dtd"		(old 1.0 web apps)
	// "tags/dtd/filename.dtd"
	// "/com_sun_web_ui/dtd/filename.dtd"	(2.0 and later web apps)
	// "com_sun_web_ui/dtd/filename.dtd"
	// "/appname/dtd/filename.dtd"		(be nice to web apps)
	// "appname/dtd/filename.dtd"
	//
	// Resulting system identifier will be one of:
	//
	// "http://hostname:unsecure_port/resource_context/dtd/filename.dtd"
	// "http://hostname:unsecure_port/appname/dtd/filename.dtd"
	//
	// If we do not match our design pattern or cannot form the
	// http prefix, just return null to signal the parser that
	// it is on its own.  Typically, this will result in the
	// parser throwing an exception.

	String fileName = getSystemId(systemId);
	if (fileName == null) {
	    return (null);
	}

	InputSource isrc = null;
	try {
		    isrc = buildInputSource(fileName);
	} catch (Exception e) {
            System.out.println("cannot convert DTD, using default behaviour: " 
		+ e.getMessage());
	    isrc = null;
	}
	return (isrc);

    } // resolveEntity

    // ==================================================================
    //
    // Protected methods
    //
    // ==================================================================

    /**
     * Process the system identifier.  If a "file:" URL, removes the
     * scheme prefix.  If a fully qualified path name was specified that
     * begins with the server process current working directory, that
     * portion of the path is removed (some parsers arbitrarily prepend
     * the current working directory to a relative URI system Id in hopes
     * of finding the external reference via file I/O).  All path separators
     * are converted to forward slashes, if necessary.  If not a file
     * based URL and not a relative URI, a null reference is returned.
     *
     * @param systemId The unprocessed system Id from the parser
     *
     * @return The processed system Id string
     */
    protected String getSystemId(String systemId) {

	if (systemId == null) {
	    return (null);
	}
        //System.out.println("entity resolver systemid: " + systemId);
	String path;
	try {
	    path = URLDecoder.decode(systemId, xmlEncoding);
	} catch (Exception ex) {
	    // Bad encoding scheme, try raw system Id
	    path = systemId;
	}

	int index = path.lastIndexOf('/');
	path = path.substring(index+1);
	return (path);

    } // getSystemId

    // ==================================================================
    //
    // Private methods
    //
    // ==================================================================

    // Build the new sytem identifier.  Generate a URL using the HTTP
    // scheme and unsecure port.  Encode the URL string to handle spaces.

    private InputSource buildInputSource(String fileName) {

	String str;
	NonSyncStringBuffer buffer =
	    new NonSyncStringBuffer(128);
	buffer.append(COM_SUN_WEB_UI);
	buffer.append(CCSystem.URL_SEPARATOR + DTD_DIR);
	buffer.append(CCSystem.URL_SEPARATOR + fileName);

	String url = buffer.toString();
	InputStream is = getClass().getClassLoader().getResourceAsStream(url);
	return (new InputSource(is));

    } // buildInputSource

} // CCDefaultEntityResolver
