/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */
package com.sun.enterprise.tools.jsfext.layout;

import com.sun.enterprise.tools.jsfext.layout.descriptor.LayoutDefinition;
import com.sun.enterprise.tools.jsfext.util.Util;

import java.lang.reflect.InvocationTargetException;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

import javax.faces.context.FacesContext;


/**
 *  <p>	This abstract class provides the base functionality for all
 *	<code>LayoutDefinitionManager</code> implementations.  It provides an
 *	static method used to obtain an instance of a concrete
 *	<code>LayoutDefinitionManager</code>: {@link #getManager(FacesContext)}.
 *	It also provides another version of this method which allows a specific
 *	instance to be specified by classname:
 *	{@link #getManager(String className)} (typically not used, the
 *	environment should be setup to provide the correct
 *	<code>LayoutDefinitionManager</code>).  Once an instance is obtained,
 *	the {@link #getLayoutDefinition(String key)} method can be invoked to
 *	obtain a {@link LayoutDefinition}.
 *  </p>
 *
 *  @author Ken Paulsen	(ken.paulsen@sun.com)
 */
public abstract class LayoutDefinitionManager {


    /**
     *	<p> Constructor.</p>
     */
    protected LayoutDefinitionManager() {
	super();
    }


    /**
     *	<p> This method is responsible for finding/creating the requested
     *	    {@link LayoutDefinition}.</p>
     *
     *	@param	key	The key used to identify the requested
     *			{@link LayoutDefintion}.
     */
    public abstract LayoutDefinition getLayoutDefinition(String key) throws IOException;


    /**
     *	<p> This is a factory method for obtaining the
     *	    {@link LayoutDefinitionManager}.  This implementation uses the
     *	    external context's initParams to look for the
     *	    {@link LayoutDefinitionManager} class.  If it exists, the specified
     *	    concrete {@link LayoutDefinitionManager} class will be used.
     *	    Otherwise, the default {@link LayoutDefinitionManager} will be
     *	    used.  The initParam key is:
     *	    {@link #LAYOUT_DEFINITION_MANAGER_KEY}.</p>
     *
     *	@param	context	The <code>FacesContext</code>.
     *
     *	@see #LAYOUT_DEFINITION_MANAGER_KEY
     */
    public static LayoutDefinitionManager getManager(FacesContext context) {
// FIXME: Decide how to define the LAYOUT_DEFINITION_MANAGER
// FIXME: Properties should be settable on the LDM, such as entity resolvers...
	Map initParams = context.getExternalContext().getInitParameterMap();
	String className = DEFAULT_LAYOUT_DEFINITION_MANAGER_IMPL;
	if (initParams.containsKey(LAYOUT_DEFINITION_MANAGER_KEY)) {
	    className = (String) initParams.get(LAYOUT_DEFINITION_MANAGER_KEY);
	}
	return getManager(className);
    }


    /**
     *	<p> This method is a singleton factory method for obtaining an instance
     *	    of a <code>LayoutDefintionManager</code>.  It is possible that
     *	    multiple different implementations of
     *	    <code>LayoutDefinitionManager</code>s will be used within the same
     *	    JVM.  This is OK, the purpose of the
     *	    <code>LayoutDefinitionManager</code> is primarily performance.
     *	    Someone may provide a different <code>LayoutDefinitionManager</code>
     *	    to locate {@link LayoutDefiniton}'s in a different way (XML,
     *	    database, file, java code, etc.).</p>
     */
    public static LayoutDefinitionManager getManager(String className) {
	LayoutDefinitionManager ldm =
	    (LayoutDefinitionManager) _instances.get(className);
	if (ldm == null) {
	    try {
		ClassLoader loader = Util.getClassLoader(className);
		ldm = (LayoutDefinitionManager) loader.loadClass(className).
		    getMethod("getInstance", (Class []) null).
			invoke((Object) null, (Object []) null);
	    } catch (ClassNotFoundException ex) {
		throw new RuntimeException(ex);
	    } catch (NoSuchMethodException ex) {
		throw new RuntimeException(ex);
	    } catch (IllegalAccessException ex) {
		throw new RuntimeException(ex);
	    } catch (InvocationTargetException ex) {
		throw new RuntimeException(ex);
	    } catch (NullPointerException ex) {
		throw new RuntimeException(ex);
	    } catch (ClassCastException ex) {
		throw new RuntimeException(ex);
	    }
	    _instances.put(className, ldm);
	}
	return ldm;
    }


    /**
     *	<p> Retrieve an attribute by key.</p>
     *
     *	@param	key	The key used to retrieve the attribute
     *
     *	@return	The requested attribute or null
     */
    public Object getAttribute(String key) {
	return _attributes.get(key);
    }


    /**
     *	<p> Associate the given key with the given Object as an attribute.</p>
     *
     *	@param	key	The key associated with the given object (if this key
     *	    is already in use, it will replace the previously set attribute
     *	    object).
     *
     *	@param	value	The Object to store.
     */
    public void setAttribute(String key, Object value) {
	_attributes.put(key, value);
    }


    /**
     *	<p> This map contains sub-class specific attributes that may be needed
     *	    by specific implementations of
     *	    <code>LayoutDefinitionManager</code>s.  For example, setting an
     *	    <code>EntityResolver</code> on a
     *	    <code>LayoutDefinitionManager</code> that creates
     *	    <code>LayoutDefinitions</code> from XML files.</p>
     */
    private Map _attributes = new HashMap();


// FIXME: Rethink this... since I am allowing LDM's to be parameterized via
// FIXME: attributes, it is not enough to have 1 LDM... we will need 1 per
// FIXME: application.  Or... I need to move the attributes.
    /**
     *	<p> Static map of <code>LayoutDefinitionManager</code s.  Normally
     *	    this will only contain the default <code>LayoutManager</code>.</p>
     */
    private static Map _instances = new HashMap(2);


    /**
     *	<p> This constant defines the default
     *	    <code>LayoutDefinitionManager</code> implementation class name.</p>
     */
    public static final String DEFAULT_LAYOUT_DEFINITION_MANAGER_IMPL =
	"com.sun.enterprise.tools.jsfext.layout.xml.XMLLayoutDefinitionManager";


    /**
     *	<p> This constant defines the <code>LayoutDefinitionManager</code>
     *	    implementation key for initParams. ("layoutManagerImpl")</p>
     */
    public static final String LAYOUT_DEFINITION_MANAGER_KEY =
	"layoutManagerImpl";
}
