/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.enterprise.tools.guiframework.event.handlers;

import com.sun.enterprise.tools.guiframework.view.HandlerContext;
import com.sun.enterprise.tools.guiframework.view.descriptors.ViewDescriptor;
import com.sun.enterprise.tools.guiframework.view.event.ErrorEvent;

import com.iplanet.jato.RequestContext;
import com.iplanet.jato.RequestManager;
import com.iplanet.jato.view.View;
import com.iplanet.jato.view.ViewBean;
import com.iplanet.jato.view.event.ChildContentDisplayEvent;
import com.iplanet.jato.view.event.JspDisplayEvent;

import java.util.EventObject;

import javax.servlet.ServletRequest;


/**
 *  This class provides a simple implementation for an error handler.
 */
public class ErrorHandler {

    /**
     *	<P>This method is valid only for ErrorEvent objects.  It will cast to
     *	this event type and pull out all the error information.  It will then
     *	return this information as the following output parameters:</P>
     *
     *	<UL><LI>EXCEPTION -- The exception itself.</LI>
     *	    <LI>EXCEPTION_MESSAGE -- The top exception message.</LI>
     *	    <LI>EXCEPTION_CLASS_NAME -- The top exception class name.</LI>
     *	    <LI>EXCEPTION_VIEW -- View which corresponds to the ViewDescriptor
     *		that is handling the Exception / Error.</LI>
     *	    <LI>EXCEPTION_VIEW_NAME -- View which corresponds to the
     *		ViewDescriptor that is handling the Exception / Error.</LI>
     *	    <LI>EXCEPTION_VIEW_DESCRIPTOR -- ViewDescriptor which is handling
     *		the Exception / Error.</LI>
     *	    <LI>EXCEPTION_VIEW_DESCRIPTOR_NAME -- ViewDescriptor which is
     *		handling the Exception / Error.</LI>
     *	    <LI>EXCEPTION_VIEW_BEAN -- The responsible ViewBean (may be same
     *		as View).</LI>
     *	    <LI>EXCEPTION_VIEW_BEAN_NAME -- The responsible ViewBean (may be
     *		same as View).</LI>
     *	    <LI>EXCEPTION_STACK_TRACE -- The regular stack trace</LI>
     *	    <LI>EXCEPTION_CAUSE_VIEW -- The responsible View object.</LI>
     *	    <LI>EXCEPTION_CAUSE_VIEW_DESCRIPTOR -- The responsible
     *		ViewDescriptor.</LI>
     *	    <LI>EXCEPTION_CAUSE_MESSAGE -- The cause exception message (null if
     *		no chained exceptions).</LI>
     *	    <LI>EXCEPTION_CAUSE_CLASS_NAME -- The cause exception class name
     *		(null if no chained exceptions).</LI>
     *	    <LI>EXCEPTION_CAUSE_FULL_TRACE -- The full stack trace to the cause
     *		exception.</LI>
     *	    <LI>REQUEST_PARAMETERS -- The parameters passed from the browser to
     *		the server.</LI></UL>
     */
    public void handleError(RequestContext reqCtx, HandlerContext handlerCtx) {
	ErrorEvent errEvent = (ErrorEvent)handlerCtx.getEvent();

	ServletRequest request =
	    RequestManager.getRequestContext().getRequest();

	handlerCtx.setOutputValue(EXCEPTION,
	    errEvent.getException());

	handlerCtx.setOutputValue(EXCEPTION_MESSAGE,
	    errEvent.getExceptionMessage());

	handlerCtx.setOutputValue(EXCEPTION_CLASS_NAME,
	    errEvent.getExceptionClassName());

	View vw = errEvent.getView();
	handlerCtx.setOutputValue(EXCEPTION_VIEW, vw);

	handlerCtx.setOutputValue(EXCEPTION_VIEW_NAME,
	    (vw == null) ? "(Null View)" : vw.getName());

	ViewDescriptor desc = errEvent.getViewDescriptor();
	handlerCtx.setOutputValue(EXCEPTION_VIEW_DESCRIPTOR, desc);

	handlerCtx.setOutputValue(EXCEPTION_VIEW_DESCRIPTOR_NAME,
	    (desc == null) ? "(Null ViewDescriptor)" : desc.getName());

	vw = errEvent.getViewBean();
	handlerCtx.setOutputValue(EXCEPTION_VIEW_BEAN, vw);

	handlerCtx.setOutputValue(EXCEPTION_VIEW_BEAN_NAME,
	    (vw == null) ? "(Null ViewBean)" : vw.getName());

	handlerCtx.setOutputValue(EXCEPTION_STACK_TRACE,
	    errEvent.getRegularTrace());

	handlerCtx.setOutputValue(EXCEPTION_CAUSE_VIEW,
	    errEvent.getCauseView());

	handlerCtx.setOutputValue(EXCEPTION_CAUSE_VIEW_DESCRIPTOR,
	    errEvent.getCauseViewDescriptor());

	handlerCtx.setOutputValue(EXCEPTION_CAUSE_MESSAGE,
	    errEvent.getCauseMessage());

	handlerCtx.setOutputValue(EXCEPTION_CAUSE_CLASS_NAME,
	    errEvent.getCauseClassName());

	handlerCtx.setOutputValue(EXCEPTION_CAUSE_FULL_TRACE,
	    errEvent.getFullTrace());

	// Get the Request Parameters
	java.util.Enumeration names = request.getParameterNames();
	String name = null;
	String nvps = "";
	while (names.hasMoreElements()) {
	    name = names.nextElement().toString();
	    nvps += "\t'"+name+"'='"+request.getParameter(name)+"'\n";
	}
	handlerCtx.setOutputValue(REQUEST_PARAMETERS, nvps);
    }

    public void printTag(RequestContext reqCtx, HandlerContext handlerCtx) {
	System.out.println("#######");
	System.out.println("The tag object = "+((JspDisplayEvent)handlerCtx.getEvent()).getSourceTag());
	System.out.println("#######");
    }

    /**
     *	This handler is intended for endDisplay events only, and only after
     *	the ErrorHandler.handleError event has been invoked.  This handler
     *	will add information about the error to the HTML as an HTML comment
     *	so that this info is available via the HTML source.
     */
    public String addErrorInfo(RequestContext reqCtx, HandlerContext handlerCtx) {
	if (!(handlerCtx.getEvent() instanceof ChildContentDisplayEvent)) {
	    return null;
	}
	ChildContentDisplayEvent dispEvent = (ChildContentDisplayEvent)handlerCtx.getEvent();
	ServletRequest request =
	    RequestManager.getRequestContext().getRequest();

	// Get the original String
	StringBuffer buf = new StringBuffer(dispEvent.getContent());
	buf.append("\n\n<!--\n\n");
	buf.append("    Exception Info:\n");
	buf.append("===================\n\n");
	buf.append("    Exception Type: "+
	    request.getAttribute(EXCEPTION_CLASS_NAME)+"\n");
	buf.append(" Exception Message: "+
	    request.getAttribute(EXCEPTION_MESSAGE)+"\n");
	buf.append("        Root Cause: "+
	    request.getAttribute(EXCEPTION_CAUSE_CLASS_NAME)+"\n");
	buf.append("      Root Message: "+
	    request.getAttribute(EXCEPTION_CAUSE_MESSAGE)+"\n\n");
	buf.append("===================\n\n");

	buf.append("         View Info:\n");
	buf.append("===================\n\n");
	buf.append("          ViewBean: "+
	    request.getAttribute(EXCEPTION_VIEW_BEAN_NAME)+"\n");
	buf.append("              View: "+
	    request.getAttribute(EXCEPTION_VIEW_NAME)+"\n");
	buf.append("    ViewDescriptor: "+
	    request.getAttribute(EXCEPTION_VIEW_DESCRIPTOR_NAME)+"\n");
	buf.append("Request Parameters:\n"+
	    request.getAttribute(REQUEST_PARAMETERS)+"\n\n");
	buf.append("==================\n\n");

	buf.append("Below are 2 stack traces to help diagnose the problem.\n");
	buf.append("The first is the traditional stack trace, the second is\n");
	buf.append("a full stack trace to the root cause.\n\n");
	buf.append("       Stack Trace:\n");
	buf.append("===================\n\n");
	buf.append(""+request.getAttribute(EXCEPTION_STACK_TRACE)+"\n\n");
	buf.append("        Full Trace:\n");
	buf.append("===================\n\n");
	buf.append(""+request.getAttribute(EXCEPTION_CAUSE_FULL_TRACE)+"\n\n");
	buf.append("===================\n\n");
	buf.append("-->\n\n");

	return buf.toString();
    }



    //////////////////////////////////////////////////////////////////
    //	Class Variables
    //////////////////////////////////////////////////////////////////


    /**
     *	Request attribute key for the exception.
     */
    public static final String EXCEPTION =
	"EXCEPTION";

    /**
     *	Request Attribute key for the top exception message.
     */
    public static final String EXCEPTION_MESSAGE =
	"EXCEPTION_MESSAGE";

    /**
     *	Request attribute key for the top exception class name.
     */
    public static final String EXCEPTION_CLASS_NAME =
	"EXCEPTION_CLASS_NAME";

    /**
     *	Request attribute key for the View which corresponds to the
     *	ViewDescriptor that is handling the Exception / Error.
     */
    public static final String EXCEPTION_VIEW =
	"EXCEPTION_VIEW";

    /**
     *	Request attribute key for the View name which corresponds to the
     *	ViewDescriptor that is handling the Exception / Error.
     */
    public static final String EXCEPTION_VIEW_NAME =
	"EXCEPTION_VIEW_NAME";

    /**
     *	Request attribute key for the ViewDescriptor which is handling the
     *	Exception / Error.
     */
    public static final String EXCEPTION_VIEW_DESCRIPTOR =
	"EXCEPTION_VIEW_DESCRIPTOR";

    /**
     *	Request attribute key for the ViewDescriptor name which is handling the
     *	Exception / Error.
     */
    public static final String EXCEPTION_VIEW_DESCRIPTOR_NAME =
	"EXCEPTION_VIEW_DESCRIPTOR_NAME";

    /**
     *	Request attribute key for the responsible ViewBean (may be same as
     *	View).
     */
    public static final String EXCEPTION_VIEW_BEAN =
	"EXCEPTION_VIEW_BEAN";

    /**
     *	Request attribute key for the responsible ViewBean name (may be same as
     *	View).
     */
    public static final String EXCEPTION_VIEW_BEAN_NAME =
	"EXCEPTION_VIEW_BEAN_NAME";

    /**
     *	Request attribute key for the regular stack trace
     */
    public static final String EXCEPTION_STACK_TRACE =
	"EXCEPTION_STACK_TRACE";

    /**
     *	Request attribute key for the responsible View object.
     */
    public static final String EXCEPTION_CAUSE_VIEW =
	"EXCEPTION_CAUSE_VIEW";

    /**
     *	Request attribute key for the responsible ViewDescriptor.
     */
    public static final String EXCEPTION_CAUSE_VIEW_DESCRIPTOR =
	"EXCEPTION_CAUSE_VIEW_DESCRIPTOR";

    /**
     *	Request attribute key for the cause exception message (null if no
     *	chained exceptions).
     */
    public static final String EXCEPTION_CAUSE_MESSAGE =
	"EXCEPTION_CAUSE_MESSAGE";

    /**
     *	Request attribute key for the cause exception class name (null if no
     *	chained exceptions).
     */
    public static final String EXCEPTION_CAUSE_CLASS_NAME =
	"EXCEPTION_CAUSE_CLASS_NAME";

    /**
     *	Request attribute key for the full stack trace to the cause
     */
    public static final String EXCEPTION_CAUSE_FULL_TRACE =
	"EXCEPTION_CAUSE_FULL_TRACE";

    /**
     *	Request attribute key for the request parameters sent from the browser.
     */
    public static final String REQUEST_PARAMETERS = "REQUEST_PARAMETERS";
}
