/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.enterprise.deployment.node.ejb;

import java.util.Map;
import java.util.HashMap;
import java.util.List;
import java.util.LinkedList;
import java.util.Iterator;
import org.w3c.dom.Node;
import org.xml.sax.Attributes;

import com.sun.enterprise.deployment.node.XMLElement;
import com.sun.enterprise.deployment.node.MethodNode;
import com.sun.enterprise.deployment.node.DeploymentDescriptorNode;

import com.sun.enterprise.deployment.Descriptor;
import com.sun.enterprise.deployment.EjbBundleDescriptor;
import com.sun.enterprise.deployment.InterceptorBindingDescriptor;
import com.sun.enterprise.deployment.MethodDescriptor;
import com.sun.enterprise.deployment.EjbDescriptor;
import com.sun.enterprise.deployment.EjbInterceptor;
import com.sun.enterprise.deployment.xml.EjbTagNames;

public class InterceptorBindingNode extends DeploymentDescriptorNode {

    public InterceptorBindingNode() {
        super();

        registerElementHandler(new XMLElement(EjbTagNames.INTERCEPTOR_BUSINESS_METHOD), MethodNode.class);  

    }
    
    public void addDescriptor(Object newDescriptor) {

        if (newDescriptor instanceof MethodDescriptor) {

            InterceptorBindingDescriptor bindingDesc = 
                (InterceptorBindingDescriptor) getDescriptor();

            MethodDescriptor businessMethod = (MethodDescriptor)
                newDescriptor;
            businessMethod.setEjbClassSymbol(MethodDescriptor.EJB_BEAN);

            bindingDesc.setBusinessMethod(businessMethod);
            
        } else {
            super.addDescriptor(newDescriptor);
        }
    }

    /** 
     * receives notification of the end of an XML element by the Parser
     * 
     * @param element the xml tag identification
     * @return true if this node is done processing the XML sub tree
     */
    public boolean endElement(XMLElement element) {

        if (EjbTagNames.INTERCEPTOR_ORDER.equals(element.getQName())) {
            InterceptorBindingDescriptor desc = (InterceptorBindingDescriptor)
                getDescriptor();
            desc.setIsTotalOrdering(true);
        }

        return super.endElement(element);
    } 

    /**
     * all sub-implementation of this class can use a dispatch table to map xml element to
     * method name on the descriptor class for setting the element value. 
     *  
     * @return the map with the element name as a key, the setter method as a value
     */
    protected Map getDispatchTable() {
        Map table =  super.getDispatchTable();

        table.put(EjbTagNames.EJB_NAME, "setEjbName");
        table.put(EjbTagNames.INTERCEPTOR_CLASS, "appendInterceptorClass");
        table.put(EjbTagNames.EXCLUDE_DEFAULT_INTERCEPTORS, 
                  "setExcludeDefaultInterceptors");
        table.put(EjbTagNames.EXCLUDE_CLASS_INTERCEPTORS, 
                  "setExcludeClassInterceptors");

        return table;
    }    
    
    /**
     * Write interceptor bindings for this ejb.
     *
     * @param parent node in the DOM tree 
     * @param the descriptor to write
     */
    public void writeBindings(Node parent, EjbDescriptor ejbDesc) {

        List<EjbInterceptor> classInterceptors = ejbDesc.getInterceptorChain();
        if( classInterceptors.size() > 0 ) {
            writeTotalOrdering(parent, classInterceptors, ejbDesc, null);
        }

        Map<MethodDescriptor, List<EjbInterceptor>> methodInterceptorsMap =
            ejbDesc.getMethodInterceptorsMap();

        for(MethodDescriptor nextMethod : methodInterceptorsMap.keySet()) {

            List<EjbInterceptor> interceptors = 
                methodInterceptorsMap.get(nextMethod);
            
            if(interceptors.isEmpty()) {
                writeExclusionBinding(parent, ejbDesc, nextMethod);
            } else {
                writeTotalOrdering(parent, interceptors, ejbDesc, nextMethod);
            }

        }

        return;
    }

    private void writeTotalOrdering(Node parent, 
                                    List<EjbInterceptor> interceptors,
                                    EjbDescriptor ejbDesc,
                                    MethodDescriptor method) {

        Node bindingNode = appendChild(parent, 
                                       EjbTagNames.INTERCEPTOR_BINDING);

        appendTextChild(bindingNode, EjbTagNames.EJB_NAME, 
                        ejbDesc.getName());

        Node totalOrderingNode = appendChild(bindingNode,
                                             EjbTagNames.INTERCEPTOR_ORDER);
        for(EjbInterceptor next : interceptors) {

            appendTextChild(totalOrderingNode, EjbTagNames.INTERCEPTOR_CLASS,
                            next.getInterceptorClassName());
        }

        if( method != null ) {
            
            MethodNode methodNode = new MethodNode();
            methodNode.writeJavaMethodDescriptor
                (bindingNode, EjbTagNames.INTERCEPTOR_BUSINESS_METHOD, method);

        }

    }

    private void writeExclusionBinding(Node parent, EjbDescriptor ejbDesc, 
                                       MethodDescriptor method) {

        Node bindingNode = appendChild(parent, 
                                       EjbTagNames.INTERCEPTOR_BINDING);

        appendTextChild(bindingNode, EjbTagNames.EJB_NAME, 
                        ejbDesc.getName());

        appendTextChild(bindingNode, EjbTagNames.EXCLUDE_CLASS_INTERCEPTORS,
                        "true");

        MethodNode methodNode = new MethodNode();
        methodNode.writeJavaMethodDescriptor
            (bindingNode, EjbTagNames.INTERCEPTOR_BUSINESS_METHOD, method);
                                             
    }
}
