/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.enterprise.connectors.util;

import com.sun.enterprise.deployment.*;
import com.sun.enterprise.config.serverbeans.ElementProperty;
import com.sun.enterprise.connectors.*;
import com.sun.enterprise.util.*;
import com.sun.logging.LogDomains;
import java.util.logging.*;
import java.util.*;
import java.lang.*;
import java.lang.reflect.*;
import java.io.IOException;
import org.xml.sax.SAXParseException;

/**
 *  This is message listener configuration parser. It parses the 
 *  ra.xml file for the message listener specific configurations 
 *  like activationSpec javabean  properties, message listener types .
 *
 *  @author      Srikanth P
 *
 */

public class MessageListenerConfigParserImpl implements 
                               MessageListenerConfigParser {

    static Logger _logger = LogDomains.getLogger(LogDomains.RSR_LOGGER);
   
    /**
     *  Default constructor.
     *
     */

    public MessageListenerConfigParserImpl() {

    }

    /**
     * Return the ActivationSpecClass name for given rar and messageListenerType
     * @param desc ConnectorDescriptor pertaining to rar.
     * @param messageListenerType MessageListener type
     * @throws  ConnectorRuntimeException If moduleDir is null.
     *          If corresponding rar is not deployed. 
     */

    public String getActivationSpecClass( ConnectorDescriptor desc, 
             String messageListenerType) throws ConnectorRuntimeException 
    {
        if(desc == null) {
            throw new ConnectorRuntimeException("Invalid arguments");
        }

        MessageListener messageListeners[] = 
               ddTransformUtil.getMessageListeners(desc);

        String messageListenerClass = null;
        String[] messageListenerTypes = null; 
        if(messageListeners != null) {
            messageListenerTypes = new String[messageListeners.length];
            for(int i=0;i<messageListeners.length;++i) {
                if(messageListenerType.equals( 
                           messageListeners[i].getMessageListenerType())){
                    return messageListeners[i].getActivationSpecClass();
                }
            }
        }
        return null; 
    }

    /* Parses the ra.xml and returns all the Message listener types. 
     *
     * @param desc ConnectorDescriptor pertaining to rar.
     * @return Array of message listener types as strings.
     * @throws  ConnectorRuntimeException If moduleDir is null.
     *          If corresponding rar is not deployed. 
     *
     */

    public String[] getMessageListenerTypes(ConnectorDescriptor desc)
               throws ConnectorRuntimeException 
    {

        if(desc == null) {
            throw new ConnectorRuntimeException("Invalid arguments");
        }

        MessageListener messageListeners[] = 
               ddTransformUtil.getMessageListeners(desc);

        String[] messageListenerTypes = null;
        if(messageListeners != null) {
            messageListenerTypes = new String[messageListeners.length];
            for(int i=0;i<messageListeners.length;++i) {
                messageListenerTypes[i] = 
                           messageListeners[i].getMessageListenerType();
            }
        }
        return messageListenerTypes;
    }

    /** Parses the ra.xml for the ActivationSpec javabean 
     *  properties. The ActivationSpec to be parsed is 
     *  identified by the moduleDir where ra.xml is present and the 
     *  message listener type.
     *
     *  message listener type will be unique in a given ra.xml.
     *
     *  It throws ConnectorRuntimeException if either or both the
     *  parameters are null, if corresponding rar is not deployed,
     *  if message listener type mentioned as parameter is not found in ra.xml.
     *  If rar is deployed and message listener (type mentioned) is present  
     *  but no properties are present for the corresponding message listener, 
     *  null is returned.
     *
     *  @param desc ConnectorDescriptor pertaining to rar.
     *  @param  messageListenerType message listener type.It is uniqie
     *          across all <messagelistener> sub-elements in <messageadapter> 
     *          element in a given rar.
     *  @return Javabean properties with the property names and values
     *          of properties. The property values will be the values
     *          mentioned in ra.xml if present. Otherwise it will be the
     *          default values obtained by introspecting the javabean.
     *          In both the case if no value is present, empty String is
     *          returned as the value.
     *  @throws  ConnectorRuntimeException if either of the parameters are null.
     *           If corresponding rar is not deployed i.e moduleDir is invalid. 
     *           If messagelistener type is not found in ra.xml
     */

    public Properties getJavaBeanProps(ConnectorDescriptor desc, 
               String messageListenerType, String rarName) throws ConnectorRuntimeException 
    {

        if(desc == null || messageListenerType == null) {
            throw new ConnectorRuntimeException("Invalid arguments");
        }
       
        MessageListener allMessageListeners[] = 
               ddTransformUtil.getMessageListeners(desc);

        MessageListener messageListener = null;
        for(int i=0;i<allMessageListeners.length;++i) {
            if(messageListenerType.equals(
                    allMessageListeners[i].getMessageListenerType())) {
                messageListener = allMessageListeners[i];
            }
        }

        if(messageListener == null) {
            _logger.log(Level.FINE,
                    "No such MessageListener found in ra.xml", 
                    messageListenerType);
            throw new ConnectorRuntimeException(
                  "No such MessageListener found in ra.xml : " + 
                  messageListenerType);
        }

        /* ddVals           -> Properties present in ra.xml
        *  introspectedVals -> All properties with values
        *                      obtained by introspection of resource
        *                      adapter javabean
        *  mergedVals       -> merged props of raConfigPros and
        *                      allraConfigPropsWithDefVals
        */

        Properties mergedVals = null;
        Set ddVals = messageListener.getConfigProperties();
        String className = messageListener.getActivationSpecClass();
        if(className != null && className.length() != 0) {
            Properties introspectedVals = configParserUtil.introspectJavaBean(
                               className,ddVals);
            mergedVals = configParserUtil.mergeProps(ddVals,introspectedVals);
        }
        return mergedVals;
    }

    /** Returns the Properties object consisting of propertyname as the
     *  key and datatype as the value.
     *  @param  moduleDir The directory where rar is exploded.
     *  @param  messageListenerType message listener type.It is uniqie
     *          across all <messagelistener> sub-elements in <messageadapter> 
     *          element in a given rar.
     *  @return Properties object with the property names(key) and datatype
     *          of property(as value). 
     *  @throws  ConnectorRuntimeException if either of the parameters are null.
     *           If corresponding rar is not deployed i.e moduleDir is invalid. 
     *           If messagelistener type is not found in ra.xml
     */
    public Properties getJavaBeanReturnTypes(ConnectorDescriptor desc, 
               String messageListenerType) throws ConnectorRuntimeException 
    {

        if(desc == null || messageListenerType == null) {
            throw new ConnectorRuntimeException("Invalid arguments");
        }

        MessageListener allMessageListeners[] = 
               ddTransformUtil.getMessageListeners(desc);

        MessageListener messageListener = null;
        for(int i=0;i<allMessageListeners.length;++i) {
            if(messageListenerType.equals(
                    allMessageListeners[i].getMessageListenerType())) {
                messageListener = allMessageListeners[i];
            }
        }

        if(messageListener == null) {
            _logger.log(Level.FINE,
                    "No such MessageListener found in ra.xml", 
                    messageListenerType);
            throw new ConnectorRuntimeException(
                  "No such MessageListener found in ra.xml : " + 
                  messageListenerType);
        }

        /* ddVals           -> Properties present in ra.xml
        *  introspectedVals -> All properties with values
        *                      obtained by introspection of resource
        *                      adapter javabean
        *  mergedVals       -> merged props of raConfigPros and
        *                      allraConfigPropsWithDefVals
        */

        Properties mergedVals = null;
        Set ddVals = messageListener.getConfigProperties();
        String className = messageListener.getActivationSpecClass();
        if(className != null && className.length() != 0) {
            Properties introspectedVals = 
               configParserUtil.introspectJavaBeanReturnTypes(className,ddVals);
            mergedVals = configParserUtil.mergePropsReturnTypes(
                                              ddVals,introspectedVals);
        }
        return mergedVals;
    }
}
