/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */
package com.sun.enterprise.iiop;

import java.util.*;
import java.io.*;

import org.omg.CORBA.ORB;
import org.omg.IOP.TaggedComponent;

import com.sun.enterprise.util.Utility;
import com.sun.enterprise.util.ORBManager;
import com.sun.enterprise.iiop.security.GSSUtils;
import com.sun.enterprise.deployment.*;
import java.util.logging.*;
import com.sun.logging.*;

// These are internal APIs in the RMI-IIOP-POA implementation
// used because there are no public APIs for interceptors yet
import com.sun.corba.ee.spi.ior.iiop.IIOPProfile;
import com.sun.corba.ee.spi.ior.iiop.IIOPProfileTemplate;
import com.sun.corba.ee.spi.ior.IOR;
import com.sun.corba.ee.impl.encoding.CDRInputStream;
import com.sun.corba.ee.impl.encoding.CDROutputStream;
import com.sun.corba.ee.impl.encoding.EncapsInputStream;

import com.sun.corba.ee.org.omg.CSI.*;
import com.sun.corba.ee.org.omg.CSIIOP.*;

/** 
 * This is the class that manages the CSIV2 tagged component information
 * in the IORs.
 * @author Vivek Nagar
 * @author Harpreet Singh
 */

public final class CSIV2TaggedComponentInfo
{
    private static Logger _logger=null;
    static{
       _logger=LogDomains.getLogger(LogDomains.CORBA_LOGGER);
        }
    // Realm name is first picked up from the application object.
    // If the realm is unpopulated here, then we query it from
    // the IORDescriptor(as in for a standalone ejb case).
    // The fallback is "default"
    private String _realm_name = null;
    private byte[] _realm_name_bytes = null;
    private final static String DEFAULT_REALM = "default";
    
    public static final int SUPPORTED_IDENTITY_TOKEN_TYPES = 15;

    private ORB orb;
    private int sslMutualAuthPort;


    /** 
     * Constructor. 
     */
    public CSIV2TaggedComponentInfo(ORB orb) {
	this.orb = orb;
    }

    /**
     * Create the security mechanism list tagged component based 
     * on the deployer specified configuration information.
     * This method is on the server side for all ejbs. 
     */
    public TaggedComponent createSecurityTaggedComponent(int sslPort,
							 EjbDescriptor desc)
    {
        TaggedComponent tc = null;
	try {
            if(_logger.isLoggable(Level.FINE)){
                _logger.log(Level.FINE, "IIOP: Creating a Security Tagged Component");
            }
            // get the realm from the application object.
            _realm_name = desc.getApplication().getRealm();
            CompoundSecMech[] mechList = createCompoundSecMechs(sslPort, desc);
            tc =  createTaggedComponent(mechList);

        } catch(Exception e) {
            _logger.log(Level.SEVERE,"iiop.createcompund_exception",e);
        }
        return tc;
    }

    /**
     * This method is called on the server side for all non-EJB POAs. 
     */
    public TaggedComponent createSecurityTaggedComponent(int sslPort)
    {
        TaggedComponent tc = null;
	try {
	    boolean sslRequired = false;
	    String sslReq = 
		(String)(ORBManager.getCSIv2Props()).get(ORBManager.ORB_SSL_SERVER_REQUIRED);
	    if ( sslReq != null && sslReq.equals("true") ) {
		sslRequired = true;
	    }

	    boolean clientAuthReqd = false;
	    String clientAuthReq = 
		(String)(ORBManager.getCSIv2Props()).get(ORBManager.ORB_CLIENT_AUTH_REQUIRED);
	    if ( clientAuthReq != null && clientAuthReq.equals("true") ) {
		clientAuthReqd = true;
	    }

	    CompoundSecMech[] mechList = new CompoundSecMech[1]; 
	    TaggedComponent transportMech = createSSLInfo(sslPort, null, 
							  sslRequired);
	    // Create AS_Context
	    AS_ContextSec asContext = createASContextSec(null);

	    // Create SAS_Context
	    SAS_ContextSec sasContext = createSASContextSec(null);

	    short targetRequires = 
		    (clientAuthReqd ? EstablishTrustInClient.value : 0);

	    // Convert Profile.TaggedComponent to org.omg.IOP.TaggedComponent
	    mechList[0] = new CompoundSecMech(targetRequires,
					transportMech, asContext, sasContext);

	    tc =  createTaggedComponent(mechList);
        } catch(Exception e) {
            _logger.log(Level.SEVERE,"iiop.createcompund_exception",e);
        }
        return tc;
    }
    public TaggedComponent createSecurityTaggedComponent(java.util.List<com.sun.corba.ee.spi.folb.SocketInfo> scoketInfos)
    {
        TaggedComponent tc = null;
	try {
	    boolean sslRequired = false;
	    String sslReq = 
		(String)(ORBManager.getCSIv2Props()).get(ORBManager.ORB_SSL_SERVER_REQUIRED);
	    if ( sslReq != null && sslReq.equals("true") ) {
		sslRequired = true;
	    }

	    boolean clientAuthReqd = false;
	    String clientAuthReq = 
		(String)(ORBManager.getCSIv2Props()).get(ORBManager.ORB_CLIENT_AUTH_REQUIRED);
	    if ( clientAuthReq != null && clientAuthReq.equals("true") ) {
		clientAuthReqd = true;
	    }

	    CompoundSecMech[] mechList = new CompoundSecMech[1]; 
	    TaggedComponent transportMech = createSSLInfo(scoketInfos, null, 
							  sslRequired);
	    // Create AS_Context
	    AS_ContextSec asContext = createASContextSec(null);

	    // Create SAS_Context
	    SAS_ContextSec sasContext = createSASContextSec(null);

	    short targetRequires = 
		    (clientAuthReqd ? EstablishTrustInClient.value : 0);

	    // Convert Profile.TaggedComponent to org.omg.IOP.TaggedComponent
	    mechList[0] = new CompoundSecMech(targetRequires,
					transportMech, asContext, sasContext);

	    tc =  createTaggedComponent(mechList);
        } catch(Exception e) {
            _logger.log(Level.SEVERE,"iiop.createcompund_exception",e);
        }
        return tc;
    }
/*    public TaggedComponent createSecurityTaggedComponent(java.util.List<Integer> sslPorts)
    {
        TaggedComponent tc = null;
	try {
	    boolean sslRequired = false;
	    String sslReq = 
		(String)(ORBManager.getCSIv2Props()).get(ORBManager.ORB_SSL_SERVER_REQUIRED);
	    if ( sslReq != null && sslReq.equals("true") ) {
		sslRequired = true;
	    }

	    boolean clientAuthReqd = false;
	    String clientAuthReq = 
		(String)(ORBManager.getCSIv2Props()).get(ORBManager.ORB_CLIENT_AUTH_REQUIRED);
	    if ( clientAuthReq != null && clientAuthReq.equals("true") ) {
		clientAuthReqd = true;
	    }

	    CompoundSecMech[] mechList = new CompoundSecMech[sslPorts.size()]; 
	    
	    // Create AS_Context
	    AS_ContextSec asContext = createASContextSec(null);

	    // Create SAS_Context
	    SAS_ContextSec sasContext = createSASContextSec(null);

	    short targetRequires = 
		    (clientAuthReqd ? EstablishTrustInClient.value : 0);

            //for(int portNo : sslPorts){    
            for(int pIndex =0; pIndex < sslPorts.size(); pIndex++){      
                TaggedComponent transportMech = createSSLInfo(sslPorts.get(pIndex), null, 
                                                              sslRequired);
                // Convert Profile.TaggedComponent to org.omg.IOP.TaggedComponent
                mechList[pIndex] = new CompoundSecMech(targetRequires,
                                            transportMech, asContext, sasContext);
            }

	    tc =  createTaggedComponent(mechList);
        } catch(Exception e) {
            _logger.log(Level.SEVERE,"iiop.createcompund_exception",e);
        }
        return tc;
    }*/

    private TaggedComponent createTaggedComponent(CompoundSecMech[] mechList)
    {
 	CDROutputStream out = (CDROutputStream) orb.create_output_stream();
	out.putEndian();

	boolean stateful = false; 
	CompoundSecMechList list = new CompoundSecMechList(stateful,
							    mechList);
	CompoundSecMechListHelper.write(out, list);
	byte[] buf = out.toByteArray();
	TaggedComponent tc = new TaggedComponent(
				TAG_CSI_SEC_MECH_LIST.value, buf ) ;
	return tc;
    }

    public void setSSLMutualAuthPort(int port) {
	sslMutualAuthPort = port;
    }

    private int getSSLMutualAuthPort() {
	return sslMutualAuthPort;
    }

    /**
     * Create the security mechanisms. Only 1 such mechanism is created
     * although the spec allows multiple mechanisms (in decreasing order
     * of preference)
     */
    private CompoundSecMech[] createCompoundSecMechs(int sslPort,
                                        EjbDescriptor desc)
        throws IOException
    {
        if(_logger.isLoggable(Level.FINE)){
            _logger.log(Level.FINE, "IIOP: Creating CompoundSecMech");
        }

	Set iorDescSet = desc.getIORConfigurationDescriptors();
	int size = iorDescSet.size();
	if(size == 0) {
	    // No IOR config descriptors:
	    // Either none were configured or 1.2.x app.

	    // Create an IOR config desc with SSL supported
	    EjbIORConfigurationDescriptor eDesc = 
					new EjbIORConfigurationDescriptor();
	    eDesc.setIntegrity(EjbIORConfigurationDescriptor.SUPPORTED);
	    eDesc.setConfidentiality(EjbIORConfigurationDescriptor.SUPPORTED);
	    eDesc.setEstablishTrustInClient
				    (EjbIORConfigurationDescriptor.SUPPORTED);
	    iorDescSet.add(eDesc);
	    size = 1;

	    // Check if method permissions are set on the descriptor.
	    // If they are then enable username_password mechanism in as_context
	    Set permissions = desc.getPermissionedRoles();
	    if(permissions.size() > 0) {
                if(_logger.isLoggable(Level.FINE)){
                    _logger.log(Level.FINE,"IIOP:Application has protected methods");
                }
	        eDesc.setAuthMethodRequired(true);
	    }
	}
        CompoundSecMech[] mechList = new CompoundSecMech[size]; 
	Iterator itr = iorDescSet.iterator();
        if(_logger.isLoggable(Level.FINE)){
	    	_logger.log(Level.FINE,"IORDescriptor SIZE:" + size);
	}
	for(int i = 0; i < size; i++) {
	    EjbIORConfigurationDescriptor iorDesc = 
		(EjbIORConfigurationDescriptor) itr.next();
            int target_requires = getTargetRequires(iorDesc);
	    TaggedComponent comp = createSSLInfo(sslPort, iorDesc, false);
	    // Create AS_Context
            AS_ContextSec asContext = createASContextSec(iorDesc);
            // Create SAS_Context
            SAS_ContextSec sasContext = createSASContextSec(iorDesc);
	    // update the target requires value
	    int targ_req = target_requires | asContext.target_requires
		| sasContext.target_requires;
            // Convert Profile.TaggedComponent to org.omg.IOP.TaggedComponent
	    TaggedComponent transportMech = comp;
            mechList[i] = new CompoundSecMech((short)targ_req,
                            transportMech, asContext, sasContext);
	}
        return mechList;
    }

    /**
     * This method determines if all the mechanisms defined in the
     * CSIV2 CompoundSecMechList structure require protected 
     * invocations.
     */
    public boolean allMechanismsRequireSSL(Set iorDescSet) {
	int size = iorDescSet.size();
	if(size == 0) {
	    return false;
	}
	Iterator itr = iorDescSet.iterator();
	for(int i = 0; i < size; i++) {
	    EjbIORConfigurationDescriptor iorDesc = 
		(EjbIORConfigurationDescriptor) itr.next();
            int target_requires = getTargetRequires(iorDesc);
	    if(target_requires == 0) {
		return false;
	    }
	}
	return true;
    }

    /**
     * Create the AS layer context within a compound mechanism definition.
     */
    public AS_ContextSec createASContextSec(
			EjbIORConfigurationDescriptor iorDesc)
        		throws IOException
    {
        AS_ContextSec asContext = null;
        int target_supports = 0;
        int target_requires = 0;
        byte[] client_authentication_mechanism = {};
        byte[] target_name = {} ;
        String authMethod  = null;
        boolean authMethodRequired = false;

        if(_logger.isLoggable(Level.FINE)){
            _logger.log(Level.FINE, "IIOP: Creating AS_Context");
        }

        // If AS_ContextSec is not required to be generated in an IOR,
        // then optimize the code by not generating and filling in fields that are
        // irrelevant.

        if (iorDesc != null) {
            authMethod = iorDesc.getAuthenticationMethod();
            authMethodRequired = iorDesc.isAuthMethodRequired();
	}

        if ( (authMethod != null) && (authMethod.equalsIgnoreCase(EjbIORConfigurationDescriptor.NONE))) {

            asContext = new AS_ContextSec((short)target_supports,
                                          (short)target_requires,
                                          client_authentication_mechanism,
                                          target_name);
            return asContext;
	}
            
        /** Functionality for Realm Per App
         * Try to get the realm from the descriptor, else fill in default
         */
        if(_realm_name == null){// realm name should be populated at this point
            if(iorDesc != null){
                _realm_name = iorDesc.getRealmName();
            }
            if(_realm_name == null){
                _realm_name = DEFAULT_REALM;
                if(_logger.isLoggable(Level.FINE)){
                    _logger.log(Level.FINE, "IIOP:AS_Context: Realm Name = null,"
                                + " setting default realm for logging in");
                }
            }
        }
        if(_logger.isLoggable(Level.FINE)){
            _logger.log(Level.FINE, "IIOP:AS_Context: Realm Name for login = "+
                        _realm_name);
        }
        _realm_name_bytes = _realm_name.getBytes();
        
        target_name = GSSUtils.createExportedName(
                               GSSUtils.GSSUP_MECH_OID,
		               _realm_name_bytes);

        target_supports = EstablishTrustInClient.value;

        if  (authMethodRequired){
            target_requires = EstablishTrustInClient.value;
        }

        client_authentication_mechanism = getMechanism();

        asContext = new AS_ContextSec((short)target_supports,
                                        (short)target_requires,
                                        client_authentication_mechanism,
                                        target_name);

        return asContext;
    }

    /**
     * Create the SAS layer context within a compound mechanism definition.
     */
    public SAS_ContextSec createSASContextSec(
		    EjbIORConfigurationDescriptor iorDesc)
        	    throws IOException
    {
        SAS_ContextSec sasContext = null;
	// target_supports = 0 means that target supports ITTAbsent
        int target_supports = 0;
        int target_requires = 0;
        ServiceConfiguration[] priv = new ServiceConfiguration[0];
        String callerPropagation = null;
        byte[][] mechanisms = {};
        
        if(_logger.isLoggable(Level.FINE)){
            _logger.log(Level.FINE, "IIOP: Creating SAS_Context");
        }


	// this shall be non-zero if target_supports is non-zero
	int supported_identity_token_type = 0;

        if (iorDesc != null) {
            callerPropagation = iorDesc.getCallerPropagation();
        }

        if ((callerPropagation != null) 
	    && (callerPropagation.equalsIgnoreCase(EjbIORConfigurationDescriptor.NONE))){
            sasContext = new SAS_ContextSec((short)target_supports,
                                            (short)target_requires,
                                            priv, mechanisms,
					    supported_identity_token_type);
            return sasContext;
	}

	target_supports = IdentityAssertion.value;

        byte[] upm = getMechanism(); // Only username_password mechanism
        mechanisms = new byte[1][upm.length];
        for(int i = 0; i < upm.length; i++) {
            mechanisms[0][i] = upm[i];
        }

	// para 166 of CSIv2 spec says that the bit corresponding to the
	// ITTPrincipalName is non-zero if supported_mechanism has atleast
	// 1 element. Supported_mechanism has the value of GSSUP OID
	if (target_supports != 0){
	    supported_identity_token_type = SUPPORTED_IDENTITY_TOKEN_TYPES;
        }

        sasContext = new SAS_ContextSec((short)target_supports,
                                        (short)target_requires,
                                        priv, mechanisms,
					supported_identity_token_type);

        return sasContext;
    }

    /**
     * Create the SSL tagged component within a compound mechanism
     * definition.
     */
    private TaggedComponent createSSLInfo(int sslport, 
					  EjbIORConfigurationDescriptor iorDesc,
					  boolean sslRequired )
    {
	int targetSupports = 0;
	int targetRequires = 0;
        int ssl_port = sslport; // reassigned in case of mutual auth
        if(_logger.isLoggable(Level.FINE)){
            _logger.log(Level.FINE, "IIOP: Creating Transport Mechanism");
        }

	if ( iorDesc == null ) {
	    // this happens only for nameservice
	    targetSupports = Integrity.value | Confidentiality.value 
				| EstablishTrustInClient.value 
				| EstablishTrustInTarget.value;
	    if ( sslRequired ) {
		 targetRequires = Integrity.value | Confidentiality.value 
				    | EstablishTrustInClient.value;
	    }
	}
	else {
	    targetSupports = getTargetSupports(iorDesc);
	    targetRequires = getTargetRequires(iorDesc);
	    if((targetRequires & EstablishTrustInClient.value) 
                == EstablishTrustInClient.value) {
                    // ssl port will only be changed in case of mutual auth
		ssl_port = getSSLMutualAuthPort();
                if(_logger.isLoggable(Level.FINE)){
                    _logger.log(Level.FINE,"MUTUAL AUTH PORT=" + sslport);
                }   
            }
	}
        /* 
         * if both targetSupports and targetRequires are zero, then the 
         * mechanism does not support a transport_mechanism and hence
         * a TAG_NULL_TAG must be generated.
         */

        if ( (targetSupports | targetRequires) == 0  || ssl_port == -1) {
	     byte[] b = {} ;
	     TaggedComponent tc = new TaggedComponent(TAG_NULL_TAG.value, b);
             return tc;
	}
	String host_name = "";
	host_name = Utility.getLocalAddress();
	TransportAddress[] listTa = generateTransportAddresses(host_name, 
							       ssl_port);
	TLS_SEC_TRANS tls_sec = new TLS_SEC_TRANS((short)targetSupports,
						  (short)targetRequires, 
						  listTa);

 	CDROutputStream out = (CDROutputStream) orb.create_output_stream();
 	out.putEndian() ; 
 	TLS_SEC_TRANSHelper.write((org.omg.CORBA.portable.OutputStream)out, tls_sec);

	byte[] buf = out.toByteArray() ;
	// create new Tagged Component for SSL
	TaggedComponent tc = new TaggedComponent(
			TAG_TLS_SEC_TRANS.value, buf ) ;
	return tc;

    }
    private TransportAddress[] generateTransportAddresses(String host, 
							  int sslport){
	short short_port = Utility.intToShort(sslport);
	TransportAddress ta = new TransportAddress(host, short_port);
	TransportAddress[] listTa = new TransportAddress[1];
	listTa[0] = ta;
	return listTa;
   }
    private TaggedComponent createSSLInfo(java.util.List<com.sun.corba.ee.spi.folb.SocketInfo> scoketInfos, 
					  EjbIORConfigurationDescriptor iorDesc,
					  boolean sslRequired )
    {
	int targetSupports = 0;
	int targetRequires = 0;
        //int ssl_port = sslport; // reassigned in case of mutual auth
        if(_logger.isLoggable(Level.FINE)){
            _logger.log(Level.FINE, "IIOP: Creating Transport Mechanism");
        }

	if ( iorDesc == null ) {
	    // this happens only for nameservice
	    targetSupports = Integrity.value | Confidentiality.value 
				| EstablishTrustInClient.value 
				| EstablishTrustInTarget.value;
	    if ( sslRequired ) {
		 targetRequires = Integrity.value | Confidentiality.value 
				    | EstablishTrustInClient.value;
	    }
	}
	else {
	    targetSupports = getTargetSupports(iorDesc);
	    targetRequires = getTargetRequires(iorDesc);
	    if((targetRequires & EstablishTrustInClient.value) 
                == EstablishTrustInClient.value) {
                    // ssl port will only be changed in case of mutual auth
		//ssl_port = getSSLMutualAuthPort();
                if(_logger.isLoggable(Level.FINE)){
                    //_logger.log(Level.FINE,"MUTUAL AUTH PORT=" + sslport);
                }   
            }
	}
        /* 
         * if both targetSupports and targetRequires are zero, then the 
         * mechanism does not support a transport_mechanism and hence
         * a TAG_NULL_TAG must be generated.
         */

        //if ( (targetSupports | targetRequires) == 0  || ssl_port == -1) {
        if ( (targetSupports | targetRequires) == 0 ) {
	     byte[] b = {} ;
	     TaggedComponent tc = new TaggedComponent(TAG_NULL_TAG.value, b);
             return tc;
	}
	TransportAddress[] listTa = generateTransportAddresses(scoketInfos);
	TLS_SEC_TRANS tls_sec = new TLS_SEC_TRANS((short)targetSupports,
						  (short)targetRequires, 
						  listTa);

 	CDROutputStream out = (CDROutputStream) orb.create_output_stream();
 	out.putEndian() ; 
 	TLS_SEC_TRANSHelper.write((org.omg.CORBA.portable.OutputStream)out, tls_sec);

	byte[] buf = out.toByteArray() ;
	// create new Tagged Component for SSL
	TaggedComponent tc = new TaggedComponent(
			TAG_TLS_SEC_TRANS.value, buf ) ;
	return tc;

    }
    private TransportAddress[] generateTransportAddresses(java.util.List<com.sun.corba.ee.spi.folb.SocketInfo> socketInfos){
        TransportAddress[] listTa = new TransportAddress[socketInfos.size()];
	for(int i=0; i< socketInfos.size(); i++){
            com.sun.corba.ee.spi.folb.SocketInfo socketInfo = socketInfos.get(i);
            int sslport = socketInfo.port;
            String host = socketInfo.host;
            short short_port = Utility.intToShort(sslport);
            TransportAddress ta = new TransportAddress(host, short_port);
            listTa[i] = ta;
        }
	return listTa;
   }
    /**
     * Get the Compound security mechanism list from the given IOR.
     * @param the IOR.
     * @return the array of compound security mechanisms.
     */
    public CompoundSecMech[] getSecurityMechanisms(IOR ior) {
	IIOPProfile prof = ior.getProfile();
	IIOPProfileTemplate ptemp = (IIOPProfileTemplate)prof.
 			    getTaggedProfileTemplate();
	Iterator itr = ptemp.iteratorById(TAG_CSI_SEC_MECH_LIST.value);
	if(!itr.hasNext()) {
            if(_logger.isLoggable(Level.FINE)){
                String msg = "IIOP:TAG_CSI_SEC_MECH_LIST tagged component not found";
                _logger.log(Level.FINE, msg);
	    }
	    return null;
	}
	Object o = itr.next();
	    if(_logger.isLoggable(Level.FINE)){
	        _logger.log(Level.FINE,"Component:" + o);
            }
	if(itr.hasNext()) {
	    String msg = "More than one TAG_CSI_SEC_MECH_LIST tagged " + 
			 "component found ";
            _logger.log(Level.SEVERE,"iiop.many_tagged_component");
	    throw new RuntimeException(msg);
	}
	com.sun.corba.ee.spi.ior.TaggedComponent tcomp = 
		(com.sun.corba.ee.spi.ior.TaggedComponent) o;
	TaggedComponent comp = tcomp.getIOPComponent(orb);
	byte[] b = comp.component_data;
	CDRInputStream in = (CDRInputStream) new EncapsInputStream(orb, b, b.length);
	in.consumeEndian();
	CompoundSecMechList l = CompoundSecMechListHelper.read(in);
	CompoundSecMech[] list = l.mechanism_list;

	return list;
    }

    /**
     * Retrieve the SSL tagged component from the compound security
     * mechanism.
     */
    public TLS_SEC_TRANS getSSLInformation(CompoundSecMech mech){
	TaggedComponent pcomp = mech.transport_mech; 
	TLS_SEC_TRANS ssl = getSSLComponent(pcomp);
	return ssl;
	
    }
    private TLS_SEC_TRANS getSSLComponent(TaggedComponent comp)
    {
	TLS_SEC_TRANS ssl = null;        
        // a TAG_NULL_TAG implies that SSL is not required
        if (comp.tag == TAG_NULL_TAG.value){
            ssl = null;
        } else {
            byte[] b = comp.component_data;
            CDRInputStream in = (CDRInputStream)  new EncapsInputStream(orb, b, b.length);
            in.consumeEndian();
            ssl = TLS_SEC_TRANSHelper.read(in);
        }
	return ssl;
    }


    /**
     * Get the value of target_supports for the transport layer.
     */
    public int getTargetSupports(EjbIORConfigurationDescriptor iorDesc)
    {
	if ( iorDesc == null ) {
            return 0;
        }
	int supports = 0;
	String integrity = iorDesc.getIntegrity();
	if(!integrity.equalsIgnoreCase(EjbIORConfigurationDescriptor.NONE)) {
	    supports = supports | Integrity.value;
	}
	String confidentiality = iorDesc.getConfidentiality();
	if(!confidentiality.equalsIgnoreCase(EjbIORConfigurationDescriptor.NONE)) {
	    supports = supports | Confidentiality.value;
	}
	String establishTrustInTarget = iorDesc.getEstablishTrustInTarget();
	if(!establishTrustInTarget.equalsIgnoreCase(EjbIORConfigurationDescriptor.NONE)) {
	    supports = supports | EstablishTrustInTarget.value;
	}
	String establishTrustInClient = iorDesc.getEstablishTrustInClient();
	if(!establishTrustInClient.equalsIgnoreCase(EjbIORConfigurationDescriptor.NONE)) {
	    supports = supports | EstablishTrustInClient.value;
	}
	return supports;
    }

    /**
     * Get the value of target_requires for the transport layer.
     */
    public int getTargetRequires(EjbIORConfigurationDescriptor iorDesc)
    {
	if ( iorDesc == null ) {
            return 0;
        }
	int requires = 0;
	String integrity = iorDesc.getIntegrity();
	if(integrity.equalsIgnoreCase(EjbIORConfigurationDescriptor.REQUIRED)) {
	    requires = requires | Integrity.value;
	}
	String confidentiality = iorDesc.getConfidentiality();
	if(confidentiality.equalsIgnoreCase(EjbIORConfigurationDescriptor.REQUIRED)) {
	    requires = requires | Confidentiality.value;
	}
	String establishTrustInTarget = iorDesc.getEstablishTrustInTarget();
	if(establishTrustInTarget.equalsIgnoreCase(EjbIORConfigurationDescriptor.REQUIRED)) {
	    requires = requires | EstablishTrustInTarget.value;
	}
	String establishTrustInClient = iorDesc.getEstablishTrustInClient();
	if(establishTrustInClient.equalsIgnoreCase(EjbIORConfigurationDescriptor.REQUIRED)) {
	    requires = requires | EstablishTrustInClient.value;
	}
	return requires;
    }

    /**
     * Return the ASN.1 encoded representation of a GSS mechanism identifier.
     * Currently only the GSSUP Mechanism is supported.
     */
    private byte[] getMechanism()
	throws IOException{
        return GSSUtils.getDER(GSSUtils.GSSUP_MECH_OID);
    }
}
