
/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */
package com.sun.enterprise.repository;

import java.util.*;
import java.io.*;
import com.sun.enterprise.ServerConfiguration;
import com.sun.enterprise.util.FileUtil;
import javax.rmi.PortableRemoteObject;
import javax.naming.*;
// IASRI 4660742 START
import java.util.logging.*;
import com.sun.logging.*;
// IASRI 4660742 END

/**
 * Implementation of J2EEResourceFactory.  Reads and writes
 * file properties and converts them to J2EEResources.
 *
 * @author Kenneth Saks
 */ 
public class J2EEResourceFactoryImpl implements J2EEResourceFactory {

// IASRI 4660742 START
    private static Logger _logger=null;
    static{
       _logger=LogDomains.getLogger(LogDomains.ROOT_LOGGER);
        }
// IASRI 4660742 END
    private static final String FIELD_SEPARATOR = ".";
    private static final String PROP_INDICATOR  = "prop";
    private static final String NAME_ATTRIBUTE = "name";

    private static final String RESOURCE_PROP_RELATIVE_PATH = 
        "config" + File.separator + "resource.properties";

    // For comparators
    private static final int EQUAL       = 0;
    private static final int SORT_BEFORE = -1;
    private static final int SORT_AFTER  = 1;

    private static final boolean debug = false;

    // Used to store properties to disk.
    private Properties outputProperties_;

    public J2EEResourceFactoryImpl() {
        outputProperties_ = new Properties();
    }

    public J2EEResourceCollection loadDefaultResourceCollection()
        throws J2EEResourceException {
        String resourceFile = 
            FileUtil.getAbsolutePath(RESOURCE_PROP_RELATIVE_PATH);
        return loadResourceCollection(resourceFile);
    }

    public J2EEResourceCollection loadResourceCollection(String resourcesURL)
        throws J2EEResourceException {

        J2EEResourceCollection resources = new J2EEResourceCollectionImpl();
        InputStream inputStream          = null;

        try {
            Properties resourceProperties = new Properties();
            if( debug ) {
// IASRI 4660742                System.out.println("Loading resource properties from " + 
// IASRI 4660742                                   resourcesURL);
// START OF IASRI 4660742
		if(_logger.isLoggable(Level.FINE))
			_logger.log(Level.FINE,"Loading resource properties from " +resourcesURL);
// END OF IASRI 4660742
            }

            // Treat the URL as a file containing the properties
            // and load them into an input stream.
            File resourceFile   = new File(resourcesURL);
            if( !resourceFile.exists() ) {
                boolean created = resourceFile.createNewFile();
                if( !created ) {
                    throw new IOException("Could not create " + resourceFile);
                }
            }
            FileInputStream fis = new FileInputStream(resourceFile);
            inputStream         = new BufferedInputStream(fis);
            resourceProperties.load(inputStream);

            // Convert the properties to raw resource information.
            List rawInfoList = propsToRawResourceInfo(resourceProperties);

            J2EEResourceConverter converter = new GenericConverter();
            for(Iterator iter = rawInfoList.iterator(); iter.hasNext(); ) {
                RawResourceInfo rawInfo = (RawResourceInfo) iter.next();
                try {
                    J2EEResource resource = 
                        converter.rawInfoToResource(rawInfo);
                    resources.addResource(resource);
                } catch(J2EEResourceException jre) {
// IASRI 4660742                    System.out.println(jre);
// IASRI 4660742                    jre.printStackTrace();
// START OF IASRI 4660742
			_logger.log(Level.SEVERE,"enterprise.resource_exception",jre);
// END OF IASRI 4660742
                }
            }
            
        } catch(Exception e) {
            throw new J2EEResourceException(e);
        } finally {
            if( inputStream != null ) {
                try {
                    inputStream.close();
                } catch(IOException ioe) {}
            }
        }

        return resources;
    }

    public void storeDefaultResourceCollection(J2EEResourceCollection 
                                               resources)
        throws J2EEResourceException {
        String resourceFile = 
            FileUtil.getAbsolutePath(RESOURCE_PROP_RELATIVE_PATH);
        storeResourceCollection(resources, resourceFile);
    }
    
    public void storeResourceCollection(J2EEResourceCollection resources, 
                                        String resourcesURL)
        throws J2EEResourceException {

        OutputStream tempResourceOutputStream = null; 
        InputStream  tempResourceInputStream  = null;           
        OutputStream resourceOutputStream     = null;
        
        try {
            if( debug ) {
// IASRI 4660742                System.out.println("Storing resource properties to " + 
// IASRI 4660742                                   resourcesURL);
// START OF IASRI 4660742
		if(_logger.isLoggable(Level.FINE))
			_logger.log(Level.FINE,"Storing resource properties to " +resourcesURL);
// END OF IASRI 4660742
            }

            // Open a file to store the properties.
            File tempResourceFile = File.createTempFile("resource","");
            tempResourceFile.deleteOnExit();
            tempResourceOutputStream = new FileOutputStream(tempResourceFile);

            // Convert each resource to raw info.
            List allRawInfo  = new Vector();
            Set allResources = resources.getAllResources();
            J2EEResourceConverter converter = new GenericConverter();
            for(Iterator iter = allResources.iterator(); iter.hasNext(); ) {
                J2EEResource next = (J2EEResource) iter.next();
                RawResourceInfo rawInfo = converter.resourceToRawInfo(next);
                allRawInfo.add(rawInfo);
            }

            // Convert raw info to a sorted set of properties and store 
            // in a temporary file.
            storeRawInfoToPropsFile(allRawInfo, tempResourceOutputStream);
            tempResourceOutputStream.close();
            tempResourceOutputStream = null;

            // Remove extraneous comment lines from temporary resource output
            tempResourceInputStream = new FileInputStream(tempResourceFile);
            BufferedReader tempResourceReader = new BufferedReader
                (new InputStreamReader(tempResourceInputStream));
                
            resourceOutputStream = new FileOutputStream(resourcesURL);
            BufferedWriter resourceWriter = new BufferedWriter
                (new OutputStreamWriter(resourceOutputStream));
            String nextLine;

            while( (nextLine = tempResourceReader.readLine()) != null ) {
                if( !nextLine.startsWith("#") ) {
                    resourceWriter.write(nextLine);
                    resourceWriter.newLine();
                }
            }
            resourceWriter.close();

        } catch(Exception e) {
// IASRI 4660742            e.printStackTrace();
// START OF IASRI 4660742
			_logger.log(Level.SEVERE,"enterprise.resource_exception",e);
// END OF IASRI 4660742
            throw new J2EEResourceException(e);
        } finally {
            if( tempResourceOutputStream != null ) {
                try {
                    tempResourceOutputStream.close();
                } catch(IOException ioe) {}
            }
            if( tempResourceInputStream != null ) {
                try {
                    tempResourceInputStream.close();
                } catch(IOException ioe) {}
            }
            if( resourceOutputStream != null ) {
                try {
                    resourceOutputStream.close();
                } catch(IOException ioe) {}
            }
            
        }
    }

    public J2EEResource createResource(int type, String name) {

        J2EEResource resource = null;

        switch(type) {
            case J2EEResource.JMS_DESTINATION :
                resource = new JmsDestinationResource(name);
                break;
            case J2EEResource.JMS_CNX_FACTORY :
                resource = new JmsCnxFactoryResource(name);
                break;
            case J2EEResource.JDBC_RESOURCE :
                resource = new JdbcResource(name);
                break;
            case J2EEResource.JDBC_XA_RESOURCE :
                resource = new JdbcXAResource(name);
                break;
            case J2EEResource.JDBC_DRIVER :
                resource = new JdbcDriver(name);
                break;
            default :
                throw new java.lang.IllegalArgumentException();
        }

        return resource;
    }

    public ResourceProperty createProperty(String name) {
        return new ResourcePropertyImpl(name);
    }

    /**
     * Convert properties to an intermediate representation
     * (RawResourceInfo) that is independent of resource info type.
     */ 
    private List propsToRawResourceInfo(Properties resourceProperties) 
        throws Exception {

        Vector allRawInfo     = new Vector();
        Enumeration propNames = resourceProperties.propertyNames();

        // Input name/value pairs can be one of two types :
        // resource attributes and resource properties.  
        //
        // Resource attributes have form 
        // <resourceType>.<index>.<name>=<value>
        //
        // Resource properties have form 
        // <resourceType>.<index>.prop.<name>=<value>
        // 

        while(propNames.hasMoreElements()) {

            // Parse resource type and index.
            String nextProp     = (String) propNames.nextElement();
            StringTokenizer tokenizer = new StringTokenizer(nextProp,
                                                            FIELD_SEPARATOR);

            String resourceType = tokenizer.nextToken();
            String numberStr    = tokenizer.nextToken();
            int resourceIndex   = Integer.parseInt(numberStr);
            
            RawResourceInfo info = 
                new RawResourceInfo(resourceType, resourceIndex);

            // If new resource, add it to list.
            int elementIndex = allRawInfo.indexOf(info);
            if( elementIndex == -1 ) {
                allRawInfo.add(info);
            } else {
                info = (RawResourceInfo) allRawInfo.elementAt(elementIndex);
            }
            
            // Parse property/attribute name and vlaue and set 
            // raw info.
            String key = tokenizer.nextToken();
            String propValue = (String) resourceProperties.get(nextProp);

            if( key.equals(PROP_INDICATOR) && tokenizer.hasMoreTokens()) {
                key = tokenizer.nextToken();
                info.getProperties().put(key, propValue);
            } else {
                info.getAttributes().put(key, propValue);
            }
        }
        return allRawInfo;
    }

    /**
     * Convert intermediate representation back to a properties
     * file format.
     */
    private void storeRawInfoToPropsFile(List allRawInfo, OutputStream out)
        throws IOException {

        
        // Sort resources alphabetically by resource type
        // and name.
        Object[] rawInfoArray = allRawInfo.toArray();
        Arrays.sort(rawInfoArray, new RawInfoSorter());

        // For generating unique resource indexes.
        Hashtable indexCounters   = new Hashtable();
        
        for(int rawIndex = 0; rawIndex < rawInfoArray.length; rawIndex++) {
            RawResourceInfo rawInfo = 
                (RawResourceInfo) rawInfoArray[rawIndex];
            String resourceType     = rawInfo.getResourceType();
            Integer index = (Integer) indexCounters.get(resourceType);

            if( index == null ) {
                index = new Integer(0);
            }

            String propNamePrefix = 
                resourceType + FIELD_SEPARATOR + index + FIELD_SEPARATOR;

            // Increment counter for this resource type
            indexCounters.put(resourceType, 
                              new Integer(index.intValue() + 1));
            
            writeNameValueCollection(out, propNamePrefix,
                                     rawInfo.getAttributes());
            
            writeNameValueCollection(out, propNamePrefix + PROP_INDICATOR +
                                     FIELD_SEPARATOR, 
                                     rawInfo.getProperties());
        }
        return;
    }

    private void writeNameValueCollection(OutputStream out,
                                          String prefix,
                                          Hashtable nameValueCollection) 
        throws IOException {

        // Print key/value pairs alphabetically by key name, 
        // but always put resource's name first.
        Set entrySet = nameValueCollection.entrySet();
        Object[] nameValueArray = entrySet.toArray();
        Arrays.sort(nameValueArray, new NameValueSorter());
        for(int index = 0; index < nameValueArray.length; index++) {
            Map.Entry next = (Map.Entry) nameValueArray[index];
            writeNameValuePair(out, prefix, next.getKey().toString(), 
                               next.getValue().toString());
        }
    }

    private void writeNameValuePair(OutputStream out, String prefix,
                                    String name, String value) 
        throws IOException {
        // Use properties object so encoding of non-ASCII characters
        // is handled properly.  Only call store for one property
        // at a time, since properties object has no sorting capability.
        outputProperties_.clear();
        outputProperties_.put(prefix + name, value);
        outputProperties_.store(out, null);
    }
                                    
    private static class RawInfoSorter implements Comparator {

        public int compare(Object o1, Object o2) {
            
            int sortResult = EQUAL;

            RawResourceInfo raw1 = (RawResourceInfo) o1;
            RawResourceInfo raw2 = (RawResourceInfo) o2;

            if( !raw1.equals(raw2) ) {
                String type1 = raw1.getResourceType();
                String type2 = raw2.getResourceType();
                if( type1.equals(type2) ) {
                    // Sort by name within same type.
                    String name1 = 
                        (String) raw1.getAttributes().get(NAME_ATTRIBUTE);
                    String name2 = 
                        (String) raw2.getAttributes().get(NAME_ATTRIBUTE);
                    sortResult = name1.compareTo(name2);
                } else {
                    // Sort different types by resource type name.
                    sortResult = type1.compareTo(type2);
                }
            }
            return sortResult;
        }
    }

    private static class NameValueSorter implements Comparator {

        public int compare(Object o1, Object o2) {
            
            int sortResult = EQUAL;

            Map.Entry pair1 = (Map.Entry) o1;
            Map.Entry pair2 = (Map.Entry) o2;

            if( !pair1.equals(pair2) ) {
                String key1 = (String) pair1.getKey();
                String key2 = (String) pair2.getKey();
                if( key1.equals(NAME_ATTRIBUTE) ) {
                    sortResult = SORT_BEFORE;
                } else if( key2.equals(NAME_ATTRIBUTE) ) {
                    sortResult = SORT_AFTER;
                } else {
                    sortResult = key1.compareTo(key2);
                }
            }
            return sortResult;
        }
    }
        
}
