/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */
package com.sun.enterprise.resource;

import java.util.*;
import java.util.logging.*;

import javax.transaction.*;
import javax.transaction.xa.*;

import com.sun.logging.*;
import com.sun.enterprise.*;
import com.sun.enterprise.log.Log;

/**
 * Resource Manager for any resource request from a component.
 *
 * @author Binod PG
 */
public class ResourceManagerImpl implements ResourceManager {
    
    
    static Logger _logger = null;
    static {
        _logger = LogDomains.getLogger(LogDomains.RSR_LOGGER);
    }
    
    /**
     * Returns the transaction component is participating.
     *
     * @return Handle to the <code>Transaction</code> object.
     * @exception <code>PoolingException<code>
     */
    public Transaction getTransaction() throws PoolingException{
        InvocationManager invmgr =
        Switch.getSwitch().getInvocationManager();
        ComponentInvocation inv = invmgr.getCurrentInvocation();
        if (inv == null) {
            throw new InvocationException();
        }
        return inv.getTransaction();
    }
    
    /**
     * Returns the component invoking resource request.
     *
     * @return Handle to the component.
     */
    public Object getComponent(){
        
        InvocationManager invmgr =
        Switch.getSwitch().getInvocationManager();
        ComponentInvocation inv = invmgr.getCurrentInvocation();
        if (inv == null) {
            throw new InvocationException();
        }
        
        return inv.getInstance();
    }

    /**
     * Enlist the <code>ResourceHandle</code> in the transaction
     *
     * @param h	<code>ResourceHandle</code> object
     * @exception <code>PoolingException</code>
     */
    public void enlistResource(ResourceHandle h) throws PoolingException{
        registerResource(h);
    }
    
    /**
     * Register the <code>ResourceHandle</code> in the transaction
     *
     * @param h	<code>ResourceHandle</code> object
     * @exception <code>PoolingException</code>
     */
    public void registerResource(ResourceHandle handle)
            throws PoolingException 
    {
        try {
            Transaction tran = null;
            J2EETransactionManager tm = 
            Switch.getSwitch().getTransactionManager();
            // enlist if necessary
            if (handle.isTransactional()) {
                InvocationManager invmgr =
                Switch.getSwitch().getInvocationManager();
                ComponentInvocation inv = invmgr.getCurrentInvocation();
                              
                if (inv == null) {
                    //throw new InvocationException();
                    
                    //Go to the tm and get the transaction
		    //This is mimicking the current behavior of
		    //the SystemResourceManagerImpl registerResource method
		    //in that, you return the transaction from the TxManager
		    try {
                        tran = tm.getTransaction(); 	
                    } catch( Exception e ) {
		        tran = null;
			_logger.log(Level.INFO, e.getMessage());
		    }
                } else {
                    tran = inv.getTransaction();
                    tm.registerComponentResource(handle);
                }
                
                if (tran != null) {
                    enlist( tm, tran, handle );
                }
            }
                        
        } catch (Exception ex) {
            _logger.log(Level.SEVERE,"poolmgr.component_register_exception",ex);
            throw new PoolingException(ex.toString(), ex);
        }
    }
   
    //Overridden by the LazyEnlistableResourceManager to be a No-Op
    protected void enlist( J2EETransactionManager tm, Transaction tran, 
        ResourceHandle h ) throws PoolingException {
        try {
            tm.enlistResource( tran, h );
        } catch( Exception e ) {
            PoolingException pe = new PoolingException( e.getMessage() );
            pe.initCause( e );
            throw pe;
        }
    }
    
    /**
     * Get's the component's transaction and marks it for rolling back.
     */
    public void rollBackTransaction() {
        InvocationManager invmgr =
        Switch.getSwitch().getInvocationManager();
	J2EETransactionManager tm = Switch.getSwitch().getTransactionManager();
	Transaction tran = null;
        try {
            ComponentInvocation inv = invmgr.getCurrentInvocation();
            if (inv == null) {
                //throw new InvocationException();
                    
               //Go to the tm and get the transaction
               //This is mimicking the current behavior of
               //the SystemResourceManagerImpl registerResource method
               //in that, you return the transaction from the TxManager
	       try {
                   tran = tm.getTransaction(); 	
               } catch( Exception e ) {
	           tran = null;
                   _logger.log(Level.INFO, e.getMessage());
               }

            } else {
                tran = inv.getTransaction();
	    }
            if (tran != null) {
                tran.setRollbackOnly();
            }
        } catch (SystemException ex) {
            _logger.log(Level.WARNING,"poolmgr.system_exception",ex);
        } catch (IllegalStateException ex) {
            // ignore
        }
    }
    
    /**
     * delist the <code>ResourceHandle</code> from the transaction
     *
     * @param h	<code>ResourceHandle</code> object
     * @param xaresFlag flag indicating transaction success. This can
     *        be XAResource.TMSUCCESS or XAResource.TMFAIL
     * @exception <code>PoolingException</code>
     */
    public void delistResource(ResourceHandle resource, int xaresFlag) {
        unregisterResource(resource,xaresFlag);
    }
    
    /**
     * Unregister the <code>ResourceHandle</code> from the transaction
     *
     * @param h	<code>ResourceHandle</code> object
     * @param xaresFlag flag indicating transaction success. This can
     *        be XAResource.TMSUCCESS or XAResource.TMFAIL
     * @exception <code>PoolingException</code>
     */
    public void unregisterResource(ResourceHandle resource,
    int xaresFlag) {
        
        J2EETransactionManager tm =
        Switch.getSwitch().getTransactionManager();
	Transaction tran = null;
        
        try {
            // delist with TMSUCCESS if necessary
            if (resource.isTransactional()) {
                InvocationManager invmgr =
                Switch.getSwitch().getInvocationManager();
                ComponentInvocation inv = invmgr.getCurrentInvocation();
                if (inv == null) {
                    //throw new InvocationException();
                    
                    //Go to the tm and get the transaction
		    //This is mimicking the current behavior of
		    //the SystemResourceManagerImpl registerResource method
		    //in that, you return the transaction from the TxManager
		    try {
                        tran = tm.getTransaction(); 	
		    } catch ( Exception e ) {
		        tran = null;
			_logger.log( Level.INFO, e.getMessage() );
		    }
                } else {
                    tran = inv.getTransaction();
                    tm.unregisterComponentResource(resource);
		}
                if (tran != null && resource.isEnlisted() ) {
                    tm.delistResource(tran, resource, xaresFlag);
                }
            }
        } catch (SystemException ex) {
            _logger.log(Level.WARNING,"poolmgr.system_exception",ex);
        } catch (IllegalStateException ex) {
            // transaction aborted. Do nothing
        } catch (InvocationException ex) {
            // unregisterResource is called outside of component context
            // likely to be container-forced destroy. Do nothing
        }
        
    }
}
