/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.enterprise.security.auth.realm.ldap;

import java.util.*;
import java.util.logging.Logger;
import java.util.logging.Level;
import javax.naming.Context;
import com.sun.logging.LogDomains;

import com.sun.enterprise.security.acl.RoleMapper;
import com.sun.enterprise.security.auth.realm.User;
import com.sun.enterprise.security.auth.realm.Realm;
import com.sun.enterprise.security.auth.realm.BadRealmException;
import com.sun.enterprise.security.auth.realm.NoSuchUserException;
import com.sun.enterprise.security.auth.realm.NoSuchRealmException;
import com.sun.enterprise.security.auth.realm.AuthenticationHandler;
import com.sun.enterprise.security.auth.realm.InvalidOperationException;

import com.sun.enterprise.security.RealmConfig;
import com.sun.enterprise.security.auth.realm.IASRealm;


/**
 * Realm wrapper for supporting LDAP authentication.
 *
 * <P>See LDAPLoginModule documentation for more details on the operation
 * of the LDAP realm and login module.
 *
 * <P>The ldap realm needs the following properties in its configuration:
 * <ul>
 *   <li>directory - URL of LDAP directory to use
 *   <li>base-dn - The base DN to use for user searches.
 *   <li>jaas-ctx - JAAS context name used to access LoginModule for
 *       authentication.
 * </ul>
 *
 * <P>Besides JDK Context properties start with java.naming, javax.security,
 *  one can also set connection pool related properties starting with
 *  com.sun.jndi.ldap.connect.pool.
 *  See http://java.sun.com/products/jndi/tutorial/ldap/connect/config.html
 *  for details.
 *  Also, the following optional attributes can also be specified:
 * <ul>
 *   <li>search-filter - LDAP filter to use for searching for the user
 *       entry based on username given to iAS. The default value is
 *       <code>uid=%s</code> where %s is expanded to the username.
 *   <li>group-base-dn - The base DN to use for group searches. By default
 *       its value is the same as base-dn.
 *   <li>group-search-filter - The LDAP filter to use for searching group
 *       membership of a given user. The default value is <code>
 *       uniquemember=%d</code> where %d is expanded to the DN of the
 *       user found by the user search.
 *   <li>group-target - The attribute which value(s) are interpreted as
 *       group membership names of the user. Default value is <code>cn</code>.
 *   <li>search-bind-dn - The dn of ldap user. optional and no default value.
 *   <li>search-bind-password - The password of search-bind-dn.optional and 
 *       no default value.
 *   <li>pool-size - The JNDI ldap connection pool size.
 * </ul>
 * 
 * @see com.sun.enterprise.security.auth.login.LDAPLoginModule
 *
 */
public final class LDAPRealm extends IASRealm
{
    // Descriptive string of the authentication type of this realm.
    public static final String AUTH_TYPE = "ldap";

    // These are property names which should be in auth-realm in server.xml
    public static final String PARAM_DIRURL="directory";
    public static final String PARAM_USERDN="base-dn";
   
    // These are optional, defaults are provided
    // %s = subject name
    // %d = DN of user search result
    public static final String PARAM_SEARCH_FILTER="search-filter";
    public static final String PARAM_GRPDN="group-base-dn";
    public static final String PARAM_GRP_SEARCH_FILTER="group-search-filter";
    public static final String PARAM_GRP_TARGET="group-target";
    public static final String PARAM_MODE="mode";
    public static final String PARAM_JNDICF="jndiCtxFactory";
    public static final String PARAM_POOLSIZE="pool-size";
    
    // These are optional, no default values are provided
    public static final String PARAM_BINDDN="search-bind-dn";
    public static final String PARAM_BINDPWD="search-bind-password";

    // Only find-bind mode is supported so mode attribute is not exposed yet
    public static final String MODE_FIND_BIND="find-bind";

    // Expansion strings
    public static final String SUBST_SUBJECT_NAME="%s";
    public static final String SUBST_SUBJECT_DN="%d";

    // Defaults
    private static final String SEARCH_FILTER_DEFAULT=
                                     "uid="+SUBST_SUBJECT_NAME;
    private static final String GRP_SEARCH_FILTER_DEFAULT=
                                     "uniquemember="+SUBST_SUBJECT_DN;
    private static final String GRP_TARGET_DEFAULT="cn";
    private static final String MODE_DEFAULT=MODE_FIND_BIND;
    private static final String JNDICF_DEFAULT=
                                     "com.sun.jndi.ldap.LdapCtxFactory";
    private static final int POOLSIZE_DEFAULT=5;
    
    private static final String SUN_JNDI_POOL = "com.sun.jndi.ldap.connect.pool";
    private static final String SUN_JNDI_POOL_ = "com.sun.jndi.ldap.connect.pool.";
    private static final String SUN_JNDI_POOL_MAXSIZE = "com.sun.jndi.ldap.connect.pool.maxsize";
    // dynamic group related properties.
    private static final String DYNAMIC_GROUP_OBJECT_FACTORY =
                                        "com.sun.jndi.ldap.obj.LdapGroupFactory";
    public static final String DYNAMIC_GROUP_FACTORY_OBJECT_PROPERTY =
                                        "java.naming.factory.object";
    private static final String DYNAMIC_GROUP_STATE_FACTORY = 
                                        "com.sun.jndi.ldap.obj.LdapGroupFactory";
    public static final String DYNAMIC_GROUP_STATE_FACTORY_PROPERTY =
                                        "java.naming.factory.state";
    
    public static final String DYNAMIC_GROUP_FILTER = 
                "(&(objectclass=groupofuniquenames)(objectclass=*groupofurls*))";
    
    private HashMap groupCache;
    private Vector emptyVector;
    private Properties ldapBindProps = new Properties();

    /**
     * Initialize a realm with some properties.  This can be used
     * when instantiating realms from their descriptions.  This
     * method may only be called a single time.  
     *
     * @param props Initialization parameters used by this realm.
     * @exception BadRealmException If the configuration parameters
     *     identify a corrupt realm.
     * @exception NoSuchRealmException If the configuration parameters
     *     specify a realm which doesn't exist.
     *
     */
    public synchronized void init(Properties props)
        throws BadRealmException, NoSuchRealmException
    {
        String url = props.getProperty(PARAM_DIRURL);
        this.setProperty(PARAM_DIRURL, url);
        ldapBindProps.setProperty(Context.PROVIDER_URL, url);
        
        String dn = props.getProperty(PARAM_USERDN);
        this.setProperty(PARAM_USERDN, dn);

        String jaasCtx = props.getProperty(IASRealm.JAAS_CONTEXT_PARAM);
        this.setProperty(IASRealm.JAAS_CONTEXT_PARAM, jaasCtx);
        
        if (url==null || dn==null || jaasCtx==null) {
            String msg = sm.getString("ldaprealm.badconfig", url, dn, jaasCtx);
            throw new BadRealmException(msg);
        }

        String mode = props.getProperty(PARAM_MODE, MODE_DEFAULT);
        if (!MODE_DEFAULT.equals(mode)) {
            String msg = sm.getString("ldaprealm.badmode", mode);
            throw new BadRealmException(msg);
        }
        this.setProperty(PARAM_MODE, mode);

        String ctxF = props.getProperty(PARAM_JNDICF, JNDICF_DEFAULT);
        this.setProperty(PARAM_JNDICF, ctxF);
        ldapBindProps.setProperty(Context.INITIAL_CONTEXT_FACTORY, ctxF);

        String searchFilter = props.getProperty(
                PARAM_SEARCH_FILTER, SEARCH_FILTER_DEFAULT);
        this.setProperty(PARAM_SEARCH_FILTER,searchFilter);

        String grpDN = props.getProperty(PARAM_GRPDN, dn);
        this.setProperty(PARAM_GRPDN, grpDN);

        String grpSearchFilter = props.getProperty(
                PARAM_GRP_SEARCH_FILTER, GRP_SEARCH_FILTER_DEFAULT);
        this.setProperty(PARAM_GRP_SEARCH_FILTER, grpSearchFilter);

        String grpTarget = props.getProperty(
                PARAM_GRP_TARGET, GRP_TARGET_DEFAULT);
        this.setProperty(PARAM_GRP_TARGET, grpTarget);

        String objectFactory = props.getProperty(
                DYNAMIC_GROUP_FACTORY_OBJECT_PROPERTY, DYNAMIC_GROUP_OBJECT_FACTORY);
        this.setProperty(DYNAMIC_GROUP_FACTORY_OBJECT_PROPERTY, objectFactory);
        ldapBindProps.setProperty(DYNAMIC_GROUP_FACTORY_OBJECT_PROPERTY, objectFactory);

        String stateFactory = props.getProperty(
                DYNAMIC_GROUP_STATE_FACTORY_PROPERTY, DYNAMIC_GROUP_STATE_FACTORY);
        this.setProperty(DYNAMIC_GROUP_STATE_FACTORY_PROPERTY, stateFactory);
        ldapBindProps.setProperty(DYNAMIC_GROUP_STATE_FACTORY_PROPERTY, stateFactory);

        String bindDN = props.getProperty(PARAM_BINDDN);
        if (bindDN != null) {
            this.setProperty(PARAM_BINDDN, bindDN);
            ldapBindProps.setProperty(Context.SECURITY_PRINCIPAL, bindDN);
        } 
        String bindPWD = props.getProperty(PARAM_BINDPWD);
        if (bindPWD != null) {
            this.setProperty(PARAM_BINDPWD, bindPWD);
            ldapBindProps.setProperty(Context.SECURITY_CREDENTIALS, bindPWD);
        }

        Enumeration penum = props.propertyNames();
        while (penum.hasMoreElements()) {
            String propName = (String)penum.nextElement();
            if (propName.startsWith("java.naming.") ||
                    propName.startsWith("javax.security.")) {
                ldapBindProps.setProperty(propName, props.getProperty(propName));
            } else if (propName.startsWith(SUN_JNDI_POOL_) &&
                    !SUN_JNDI_POOL_MAXSIZE.equals(propName)) {
                if (System.getProperty(propName) == null) {
                    System.setProperty(propName, props.getProperty(propName));
                }
            }
        }

        String poolSize =
            Integer.getInteger(PARAM_POOLSIZE,POOLSIZE_DEFAULT).toString();
        String sunPoolSizeStr = props.getProperty(SUN_JNDI_POOL_MAXSIZE,
            poolSize);
        //Precedence rule: SUN_JNDI_POOL_MAXSIZE > PARAM_POOLSIZE > POOLSIZE_DEFAULT
        try {
            sunPoolSizeStr = Integer.valueOf(sunPoolSizeStr).toString();
        } catch(Exception ex) {
            sunPoolSizeStr = poolSize;
        }
        if (System.getProperty(SUN_JNDI_POOL_MAXSIZE) == null) {       
            System.setProperty(SUN_JNDI_POOL_MAXSIZE, sunPoolSizeStr);
        }
        this.setProperty(PARAM_POOLSIZE, sunPoolSizeStr);

        String usePool = props.getProperty(SUN_JNDI_POOL, "true");
        ldapBindProps.setProperty(SUN_JNDI_POOL, usePool);

        if (_logger.isLoggable(Level.FINE)) {
            Properties tempProps = (Properties)ldapBindProps.clone();
            tempProps.remove(Context.SECURITY_CREDENTIALS);
            _logger.log(Level.FINE, "LDAPRealm : " + tempProps);
        }

        groupCache = new HashMap();
        emptyVector = new Vector();
    }


    /**
     * Returns a short (preferably less than fifteen characters) description
     * of the kind of authentication which is supported by this realm.
     *
     * @return Description of the kind of authentication that is directly
     *     supported by this realm.
     */
    public String getAuthType()
    {
        return AUTH_TYPE;
    }
    


    /**
     * Get binding properties defined in server.xml for LDAP server.
     *
     */
    public Properties getLdapBindProps()
    {
        return (Properties)ldapBindProps.clone();
    }


    /**
     * Returns the name of all the groups that this user belongs to.
     * Note that this information is only known after the user has
     * logged in. This is called from web path role verification, though
     * it should not be.
     *
     * @param username Name of the user in this realm whose group listing
     *     is needed.
     * @return Enumeration of group names (strings).
     * @exception InvalidOperationException thrown if the realm does not
     *     support this operation - e.g. Certificate realm does not support
     *     this operation.
     */
    public Enumeration getGroupNames (String username)
        throws InvalidOperationException, NoSuchUserException
    {
        Vector v = (Vector)groupCache.get(username);
        if (v == null) {
            if (_logger.isLoggable(Level.FINE)) {
                _logger.log(Level.FINE, "No groups available for: "+username);
            }
            return emptyVector.elements();
        } else {
            return v.elements();
        }
    }


    /**
     * Set group membership info for a user.
     *
     * <P>See bugs 4646133,4646270 on why this is here.
     *
     */
    public void setGroupNames(String username, String[] groups)
    {
        Vector v = new Vector(groups.length);
        for (int i=0; i<groups.length; i++) {
            v.add(groups[i]);
        }
        groupCache.put(username, v);
    }


    
}
