/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the "License").  You may not use this file except 
 * in compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * glassfish/bootstrap/legal/CDDLv1.0.txt or 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html. 
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * HEADER in each file and include the License file at 
 * glassfish/bootstrap/legal/CDDLv1.0.txt.  If applicable, 
 * add the following below this CDDL HEADER, with the 
 * fields enclosed by brackets "[]" replaced with your 
 * own identifying information: Portions Copyright [yyyy] 
 * [name of copyright owner]
 */
// Copyright (c) 1998, 2006, Oracle. All rights reserved.  
package oracle.toplink.essentials.ejb.cmp3;

import java.io.File;
import java.util.HashSet;
import java.util.Map;
import java.util.HashMap;
import java.util.Collection;
import java.util.Iterator;
import java.util.Enumeration;
import java.util.Vector;
import java.net.URL;
import javax.persistence.*;
import javax.persistence.spi.*;
import oracle.toplink.essentials.internal.ejb.cmp3.EntityManagerFactoryImpl;
import oracle.toplink.essentials.internal.ejb.cmp3.JavaSECMPInitializer;
import oracle.toplink.essentials.internal.ejb.cmp3.EntityManagerSetupImpl;
import oracle.toplink.essentials.ejb.cmp3.persistence.PersistenceUnitProcessor;
import oracle.toplink.essentials.threetier.ServerSession;
import oracle.toplink.essentials.logging.AbstractSessionLog;
import oracle.toplink.essentials.logging.SessionLog;
import oracle.toplink.essentials.internal.sessions.AbstractSession;
import oracle.toplink.essentials.tools.sessionmanagement.SessionManager;

import java.util.Properties; 
import oracle.toplink.essentials.tools.schemaframework.SchemaManager;

/**
 * This is the TopLink EJB 3.0 provider
 * The default constructor can be used to build the provider by reflection, after which it can
 * be used to create EntityManagerFactories
 */

public class EntityManagerFactoryProvider implements javax.persistence.spi.PersistenceProvider {

    // The following constants are used in persistence xml to specify properties
    public static final String JDBC_DRIVER_PROPERTY = "toplink.jdbc.driver";
    public static final String JDBC_CONNECTION_STRING_PROPERTY = "toplink.jdbc.url";
    public static final String JDBC_USER_PROPERTY = "toplink.jdbc.user";
    public static final String JDBC_PASSWORD_PROPERTY = "toplink.jdbc.password";
    public static final String TOPLINK_PLATFORM_PROPERTY = "toplink.platform.class.name";
    public static final String TOPLINK_SERVER_PLATFORM_PROPERTY = "toplink.server.platform.class.name";
    public static final String TOPLINK_EXTERNAL_TRANSACTION_CONTROLLER_PROPERTY = "toplink.external.transaction.controller.class.name";
    public static final String TOPLINK_LOGGING_LEVEL = "toplink.logging.level";
    public static final String TOPLINK_ORM_THROW_EXCEPTIONS = "toplink.orm.throw.exceptions";
    public static final String TOPLINK_VALIDATION_ONLY_PROPERTY = "toplink.validation-only";

    public static final String DDL_GENERATION   = "toplink.ddl-generation";
    public static final String CREATE_ONLY      = "create-tables";
    public static final String DROP_AND_CREATE  = "drop-and-create-tables";
    public static final String NONE             = "none";
    public static final String APP_LOCATION     = "toplink.application-location";
    public static final String CREATE_JDBC_DDL_FILE = "toplink.create-ddl-jdbc-file-name";
    public static final String DROP_JDBC_DDL_FILE   = "toplink.drop-ddl-jdbc-file-name";
    public static final String MAX_WRITE_CONNECTIONS = "toplink.max-write-connections"; 
    public static final String MIN_WRITE_CONNECTIONS = "toplink.min-write-connections";
    public static final String MAX_READ_CONNECTIONS = "toplink.max-read-connections";
    public static final String MIN_READ_CONNECTIONS = "toplink.min-read-connections";
    public static final String BIND_ALL_PARAMETERS = "toplink.bind-all-parameters";
    public static final String DESCRIPTOR_CACHE_SIZE = "toplink.cache.default-size";

    public static final String DEFAULT_APP_LOCATION = "." + File.separator;
    public static final String DEFAULT_CREATE_JDBC_FILE_NAME = "createDDL.jdbc";
    public static final String DEFAULT_DROP_JDBC_FILE_NAME = "dropDDL.jdbc";
    public static final String JAVASE_DB_INTERACTION = "INTERACT_WITH_DB";    
    
    /**
     * A default constructor is required by all Providers accoring the the EJB 3.0 specification
     */
    public EntityManagerFactoryProvider() {      
    }

    /**
     * This method can be used to ensure the session represented by emSetupImpl
     * is removed from the SessionManager. 
     */
    private void cleanUpSessionManager (EntityManagerSetupImpl emSetupImpl) {
        if (emSetupImpl.getDeployedSessionName() != null){
                AbstractSession tempSession = (AbstractSession)SessionManager.getManager().getSessions().get(emSetupImpl.getDeployedSessionName());
                if(tempSession != null) {
                    SessionManager.getManager().getSessions().remove(emSetupImpl.getDeployedSessionName());
                }
        }
    }

    /**
    * Called by Persistence class when an EntityManagerFactory
    * is to be created.
    *
    * @param emName The name of the persistence unit
    * @param map A Map of properties for use by the
    * persistence provider. These properties may be used to
    * override the values of the corresponding elements in
    * the persistence.xml file or specify values for
    * properties not specified in the persistence.xml.
    * @return EntityManagerFactory for the persistence unit,
    * or null if the provider is not the right provider
    */
	public EntityManagerFactory createEntityManagerFactory(String emName, Map properties){
        Map nonNullProperties = (properties == null) ? new HashMap() : properties;
        String name = emName;
        if (name == null){
            name = "";
        }
        ServerSession session = getServerSession(name, nonNullProperties);
        if (session != null){
            try {
                EntityManagerFactory emf = new EntityManagerFactoryImpl(session);
                login(session, getPersistenceUnitProperties(name, nonNullProperties));
                generateDDLFiles(session, getPersistenceUnitProperties(name, nonNullProperties), true);
                return emf;
            } catch (oracle.toplink.essentials.exceptions.ValidationException e)  {
                throw new javax.persistence.PersistenceException(e);
            }        
        }
        return null;
    }

    /**
    * Called by the container when an EntityManagerFactory
    * is to be created.
    *
    * @param info Metadata for use by the persistence provider
    * @return EntityManagerFactory for the persistence unit
    * specified by the metadata
    * @param map A Map of integration-level properties for use
    * by the persistence provider.
    */
    public EntityManagerFactory createContainerEntityManagerFactory(PersistenceUnitInfo info, Map properties){
        EntityManagerSetupImpl emSetupImpl = new EntityManagerSetupImpl();
        ClassLoader tempLoader = info.getNewTempClassLoader();
        Collection entities = buildEntityList(info, tempLoader, emSetupImpl);
        Map mergedProperties = mergePropertiesIntoMap(properties, info.getProperties());
        boolean validationOnly = ((String)getConfigPropertyAsString(TOPLINK_VALIDATION_ONLY_PROPERTY, mergedProperties, "false")).equalsIgnoreCase("true");
        emSetupImpl.setValidationOnly(validationOnly);
        ClassTransformer transformer = emSetupImpl.predeploy(entities, tempLoader, info, properties);
        if (transformer != null){
            info.addTransformer(transformer);
        }
        ClassLoader realLoader = info.getClassLoader();
        entities = buildEntityList(info, realLoader, emSetupImpl);
        ServerSession session = null;
        try{
            session = emSetupImpl.deploy(entities, realLoader);
            if(validationOnly) {
                session.initializeDescriptors();
            } else {
                login(session, mergedProperties);
                generateDDLFiles(session, mergedProperties, false);
            }
        } catch (oracle.toplink.essentials.exceptions.ValidationException exception) {
                cleanUpSessionManager(emSetupImpl);
                throw new javax.persistence.PersistenceException(exception);
        } catch (RuntimeException exception) {
                cleanUpSessionManager(emSetupImpl);
                throw exception;
        }        
        return new EntityManagerFactoryImpl(session);
    }


    /**
     * Logs in to given session. If user has not specified  <codeTOPLINK_PLATFORM_PROPERTY</code>
     * the plaform would be auto detected
     * @param session The session to login to.
     * @param persitenceUnitProperties User specified properties for the persistence unit
     */
    private void login(ServerSession session, Map persistenceUnitProperties) {
        String toplinkPlatform = (String)persistenceUnitProperties.get(TOPLINK_PLATFORM_PROPERTY);
        if (!session.isConnected()) {
            if (toplinkPlatform == null || toplinkPlatform.equals("")) {
                // if user has not specified a database platform, try to detect
                session.loginAndDetectDatasource();
            } else {
                session.login();
            }
        }
    }

    private void generateDDLFiles(ServerSession session, Map props,
            boolean inSEmode) {
        boolean createTables = false, shouldDropFirst = false;
        String appLocation; 
        String createDDLJdbc;
        String dropDDLJdbc;
        String ddlGeneration = NONE;
        
        if(null == props)
            return;

        ddlGeneration = (String)getConfigPropertyAsString(DDL_GENERATION, props, NONE);
        ddlGeneration = ddlGeneration.toLowerCase();
        if(ddlGeneration.equals(NONE)) 
            return;

        if(ddlGeneration.equals(CREATE_ONLY) || 
            ddlGeneration.equals(DROP_AND_CREATE)) {
            createTables = true;
            if(ddlGeneration.equals(DROP_AND_CREATE)) {
                shouldDropFirst = true;
            }
        } 

        if(createTables) {
            appLocation = (String)getConfigPropertyAsString( APP_LOCATION, props, DEFAULT_APP_LOCATION);
            createDDLJdbc = (String)getConfigPropertyAsString( CREATE_JDBC_DDL_FILE, props, DEFAULT_CREATE_JDBC_FILE_NAME);
            dropDDLJdbc = (String)getConfigPropertyAsString( DROP_JDBC_DDL_FILE, props,  DEFAULT_DROP_JDBC_FILE_NAME);

            SchemaManager mgr = new SchemaManager(session);

            if(inSEmode) {
                runInSEMode(mgr, shouldDropFirst);
            }
        
            writeDDLsToFiles(mgr, shouldDropFirst, appLocation,  createDDLJdbc,  dropDDLJdbc);
        }
    }

  /**
   * Create a list of the entities that will be deployed.  This list is build from the information
   * provided in the PersistenceUnitInfo argument.
   * The list contains Classes specified in the PersistenceUnitInfo's class list and also
   * files that are annotated with @Entity, @Embeddable and @MappedSuperclass in
   * the jar files provided in the persistence info.
   * This list of classes will used by TopLink to build a deployment project and to 
   * decide what classes to weave.
   * @param info 
   * @param loader 
   * @return 
   */
    public static Collection buildEntityList(PersistenceUnitInfo info, ClassLoader loader, EntityManagerSetupImpl emSetupImpl) {
        HashSet classNames = new HashSet();
        classNames.addAll(info.getManagedClassNames());
        Iterator i = info.getJarFileUrls().iterator();
        while (i.hasNext()){
            classNames.addAll(PersistenceUnitProcessor.getPersistentClassNamesFromURL((URL)i.next(), loader));
        }
        if (!info.excludeUnlistedClasses()){
            classNames.addAll(PersistenceUnitProcessor.getPersistentClassNamesFromURL(info.getPersistenceUnitRootUrl(), loader));
        }
        // append the list of entity classes that are defined in the XML descriptor
        classNames.addAll(emSetupImpl.buildPersistentClassSet(loader, info));

        Vector entityList = new Vector();
        i = classNames.iterator();
        String className = null;
        while (i.hasNext()){
            try{
                className = (String)i.next();
                Class entityClass = loader.loadClass(className);
                entityList.add(entityClass);
            } catch (ClassNotFoundException exc){
                AbstractSessionLog.getLog().log(SessionLog.CONFIG, "exception_loading_entity_class", className, exc);
            }
        }        
        return entityList;
    }
   
    /**
     * @see {@link Persistence#createEntityManagerFactory()}.
     */
    protected ServerSession getServerSession(String emName, Map m) {
        JavaSECMPInitializer initializer = JavaSECMPInitializer.getJavaSECMPInitializer(m);
        return initializer.getServerSession(emName, m);
    }

    protected Properties getPersistenceUnitProperties(String emName, Map m) {
        JavaSECMPInitializer initializer = JavaSECMPInitializer.getJavaSECMPInitializer(m);
        return initializer.getPersistenceUnitProperties(emName);
    }
            
    private void runInSEMode(SchemaManager mgr, boolean shouldDropFirst) {
        String str = getConfigPropertyAsString(JAVASE_DB_INTERACTION, null ,"true");
        boolean interactWithDB = Boolean.valueOf(str.toLowerCase()).booleanValue();
        if (!interactWithDB){
            return;
        }
        createOrReplaceDefaultTables(mgr, shouldDropFirst);
    }
    
  /**
   * Check the provided map for an object with the given key.  If that object is not available, check the
   * System properties.  If it is not available from either location, return the default value.
   * @param propertyKey 
   * @param map 
   * @param defaultValue 
   * @return 
   */
    public static String getConfigPropertyAsString(String propertyKey, Map overrides, String defaultValue){
        String value = null;
        if (overrides != null){
            value = (String)overrides.get(propertyKey);
        }
        if (value == null){
            value = System.getProperty(propertyKey);
        }
        if (value == null){
            return defaultValue;
        }
        return value;
    }
    
    private void createOrReplaceDefaultTables(
        SchemaManager mgr, boolean shouldDropFirst) {          
        if (shouldDropFirst){
            mgr.replaceDefaultTables(true); 
        } else { 
            mgr.createDefaultTables(); 
        }
    }

    private void writeDDLsToFiles(SchemaManager mgr,  
        boolean shouldDropFirst, String appLocation,
        String createDDLJdbc, String dropDDLJdbc) {
        // Ensure that the appLocation string ends with  File.seperator 
        appLocation = addFileSeperator(appLocation);
        if (null != createDDLJdbc) {
            String createJdbcFileName = appLocation + createDDLJdbc;
            mgr.outputCreateDDLToFile(createJdbcFileName);
        }

        if (null != dropDDLJdbc) {
            String dropJdbcFileName = appLocation + dropDDLJdbc;              
            mgr.outputDropDDLToFile(dropJdbcFileName);
        }

        mgr.setCreateSQLFiles(false);
        // When running in the application server environment always ensure that
        // we write out both the drop and create table files.
        createOrReplaceDefaultTables(mgr, true);
        mgr.closeDDLWriter();
    }
    
    private String addFileSeperator(String appLocation) {
        int strLength = appLocation.length();
        if (appLocation.substring(strLength -1, strLength).equals(File.separator)) 
            return appLocation;
        else
            return appLocation + File.separator;        
    }

  /**
   * Merge the properties from the source object into the target object.  If the property
   * exists in both objects, use the one from the target
   * @param target 
   * @param source 
   * @return the target object
   */
    public static Map mergePropertiesIntoMap(Map target, Properties source){
        Map map = new HashMap();
        if (target != null){
            map.putAll(target);
        }
        if (source != null){
            Enumeration sourceNames = source.propertyNames();
            while (sourceNames.hasMoreElements()){
                String key = (String)sourceNames.nextElement();
                if (map.get(key) == null){
                    map.put(key, source.getProperty(key));
                }
            }
        }
        return map;
    }

}
