/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the "License").  You may not use this file except 
 * in compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * glassfish/bootstrap/legal/CDDLv1.0.txt or 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html. 
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * HEADER in each file and include the License file at 
 * glassfish/bootstrap/legal/CDDLv1.0.txt.  If applicable, 
 * add the following below this CDDL HEADER, with the 
 * fields enclosed by brackets "[]" replaced with your 
 * own identifying information: Portions Copyright [yyyy] 
 * [name of copyright owner]
 */
// Copyright (c) 1998, 2006, Oracle. All rights reserved.  
package oracle.toplink.essentials.ejb.cmp3.persistence;

import java.net.URL;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.jar.JarFile;
import java.util.zip.ZipEntry;
import java.util.Enumeration;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.util.List;
import java.util.Vector;
import java.util.Iterator;
import java.util.Set;
import java.util.HashSet;

import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;

import org.xml.sax.XMLReader;
import org.xml.sax.InputSource;

import oracle.toplink.essentials.exceptions.PersistenceUnitLoadingException;
import oracle.toplink.essentials.exceptions.XMLParseException;
import oracle.toplink.essentials.internal.ejb.cmp3.xml.PersistenceContentHandler;
import oracle.toplink.essentials.internal.ejb.cmp3.xml.XMLConstants;
import oracle.toplink.essentials.internal.ejb.cmp3.xml.XMLException;
import oracle.toplink.essentials.internal.ejb.cmp3.xml.XMLExceptionHandler;

/**
 * INTERNAL:
 * Utility Class that deals with persistence archives for EJB 3.0
 * Provides functions like searching for persistence archives, processing persistence.xml
 * and searching for Entities in a Persistence archive
 */
public class PersistenceUnitProcessor  {
   
    /**
     * Get a list of persitence units from the file or directory at the given url
     * PersistenceUnits are built based on the presence of persistence.xml in a META-INF directory
     * at the base of the URL
     * @param url The url of a jar file or directory to check
     * @return 
     */
    public static List<PersistenceUnitInfo> getPersistenceUnits(URL url, ClassLoader loader){
        return processPersistenceArchive(url, loader);
    }
    
  /**
   * Search a directory for persistence.xml. It should be found in a
   * meta-inf directory contained within the passed directory.
   * The file passed into this method should be a jar file or null will be returned
   * @param file a File representing the directory to search
   */
    private static List<PersistenceUnitInfo> processDirectory(URL baseURL, File file, ClassLoader loader){
        FileInputStream inputStream = null;
        try{           
            String filePath = file.getPath();
            File persistenceXMLFile = new File(filePath + File.separator + "META-INF" + File.separator + "persistence.xml"); 
            
            if (!persistenceXMLFile.exists()){
                persistenceXMLFile = new File(filePath + File.separator + "meta-inf" + File.separator + "persistence.xml");
            }
            if (persistenceXMLFile.exists()){
                inputStream = new FileInputStream(persistenceXMLFile);
                return processPersistenceXML(baseURL, inputStream, loader);
            }
            return null;
        } catch (FileNotFoundException exc){
            throw PersistenceUnitLoadingException.exceptionLoadingFromDirectory(file, exc);            
        } finally {
            try{
                if (inputStream != null){
                    inputStream.close();                    
                }
            } catch (IOException exc){};

        }
    }
    
  /**
   * Search a jar file for persistence.xml.  It should be found in the
   * the meta-inf subdirectory.  When persistence.xml is found, process it.
   * The file passed into this method should be a jar file.
   * @param file 
   */
    protected static List<PersistenceUnitInfo> processJarFile(URL baseURL, File file, ClassLoader loader){
        JarFile jarFile = null;
        InputStream stream = null;
        try{
            jarFile = new JarFile(file);
            ZipEntry entry = jarFile.getEntry("META-INF/persistence.xml");
            if (entry == null){
                entry = jarFile.getEntry("meta-inf/persistence.xml");
            }
            if (entry != null){
                stream = jarFile.getInputStream(entry);
                return processPersistenceXML(baseURL, stream, loader);
            }
            return null;
        } catch (IOException exc){
            throw PersistenceUnitLoadingException.exceptionLoadingFromJar(file, exc);
        } finally {
            try{
                if (stream != null){
                    stream.close();                    
                }
                if (jarFile != null){
                    jarFile.close();                    
                }
            } catch (IOException exc){};

        }
    }
    
    /**
     * Go through the par file for this ParProcessor and process any XML provided in it
     */
    private static List<PersistenceUnitInfo> processPersistenceArchive(URL url, ClassLoader loader){
        // Only URL.toURI() handles spaces and special characters correctly
        File file = new File(convertURLToURI(url));
        
        if (file.isFile()){
            return processJarFile(url, file, loader);
        } else if (file.isDirectory()){
            return processDirectory(url, file, loader);
        }
        return null;
    }
    
  /**
   * Build a persistence.xml file into a PersistenceUnitInfo object 
   * @param input 
   */
    public static List<PersistenceUnitInfo> processPersistenceXML(URL baseURL, InputStream input, ClassLoader loader){
        SAXParserFactory spf = SAXParserFactory.newInstance();
        spf.setNamespaceAware(true);
        spf.setValidating(true);
        
        XMLReader xmlReader = null;
        SAXParser sp = null;
        XMLExceptionHandler xmlErrorHandler = new XMLExceptionHandler();

        // create a SAX parser
        try {
            sp = spf.newSAXParser();
	        sp.setProperty(XMLConstants.SCHEMA_LANGUAGE, XMLConstants.XML_SCHEMA);
	    } catch (javax.xml.parsers.ParserConfigurationException exc){
	    	throw XMLParseException.exceptionCreatingSAXParser(baseURL, exc);
	    } catch (org.xml.sax.SAXException exc){
	    	throw XMLParseException.exceptionCreatingSAXParser(baseURL, exc);
	    }
	        
	    // create an XMLReader
	    try {
            xmlReader = sp.getXMLReader();
	        xmlReader.setErrorHandler(xmlErrorHandler);
        } catch (org.xml.sax.SAXException exc){
        	throw XMLParseException.exceptionCreatingXMLReader(baseURL, exc);
        }
       
        // attempt to load the schema from the classpath
        URL schemaURL = loader.getResource(XMLConstants.PERSISTENCE_SCHEMA_NAME);
        if (schemaURL != null) {
            try {
            	sp.setProperty(XMLConstants.JAXP_SCHEMA_SOURCE, schemaURL.toString());
            } catch (org.xml.sax.SAXException exc){
            	throw XMLParseException.exceptionSettingSchemaSource(baseURL, schemaURL, exc);
            }
        }
        
        PersistenceContentHandler myContentHandler = new PersistenceContentHandler();
        xmlReader.setContentHandler(myContentHandler);

        InputSource inputSource = new InputSource(input);
        try{
            xmlReader.parse(inputSource);
        } catch (IOException exc){
            throw PersistenceUnitLoadingException.exceptionProcessingPersistenceXML(baseURL, exc);
        } catch (org.xml.sax.SAXException exc){
        	// XMLErrorHandler will handle SAX exceptions
        }
        
        // handle any parse exceptions
        XMLException xmlError = xmlErrorHandler.getXMLException();
        if (xmlError != null) {
            throw PersistenceUnitLoadingException.exceptionProcessingPersistenceXML(baseURL, xmlError);
        }

        Iterator<PersistenceUnitInfo> persistenceInfos = myContentHandler.getPersistenceUnits().iterator();
        while (persistenceInfos.hasNext()){
            PersistenceUnitInfo info = persistenceInfos.next();
            info.setPersistenceUnitRootUrl(baseURL);           
        }
        return myContentHandler.getPersistenceUnits();
    }

   /**
   * Entries in a zip file are directory entries using slashes to separate them.
   * Build a class name using '.' instead of slash and removing the '.class' extension.
   * @param classEntryString 
   * @return 
   */
    public static String buildClassNameFromEntryString(String classEntryString){
        String classNameForLoader = classEntryString;
        if (classEntryString.endsWith(".class")){
            classNameForLoader = classNameForLoader.substring(0, classNameForLoader.length() - 6);;
            classNameForLoader = classNameForLoader.replace("/", ".");              
        }
        return classNameForLoader;
    }

   /**
   * Build a set that contains all the class names at a URL
   * @return a Set of class name strings
   */
    public static Set<String> buildClassSet(PersistenceUnitInfo persistenceUnitInfo){
        Set<String> set = new HashSet<String>();
        set.addAll(persistenceUnitInfo.getManagedClassNames());
        Iterator i = persistenceUnitInfo.getJarFileUrls().iterator();
        while (i.hasNext()) {
            set.addAll(PersistenceUnitProcessor.getClassNamesFromURL((URL)i.next()));
        }
        if (!persistenceUnitInfo.excludeUnlistedClasses()){
            set.addAll(PersistenceUnitProcessor.getClassNamesFromURL(persistenceUnitInfo.getPersistenceUnitRootUrl()));
        }
       return set;
    }

  /**
   * Return a set of class names that are annotated as either @Entity, @Embeddable, or @MappedSuperclass
   * from the base URL of this PersistenceUnitProcessor
   * @param loader the class loader to load the classes with
   * @return 
   */
    public static Set<String> buildPersistentClassSet(PersistenceUnitInfo persistenceUnitInfo, ClassLoader loader){
        Set<String> set = new HashSet();
        set.addAll(persistenceUnitInfo.getManagedClassNames());
        Iterator i = persistenceUnitInfo.getJarFileUrls().iterator();
        while (i.hasNext()) {
            set.addAll(PersistenceUnitProcessor.getPersistentClassNamesFromURL((URL)i.next(), loader));
        }
        if (!persistenceUnitInfo.excludeUnlistedClasses()){
            set.addAll(PersistenceUnitProcessor.getPersistentClassNamesFromURL(persistenceUnitInfo.getPersistenceUnitRootUrl(), loader));
        }
       return set;
    }

  /**
   * Recursive method to look through a directory and build class names for all files
   * ending in '.class' in that directory and any subdirectory.  Strips out any extraneous
   * characters and returns a className with package names separated by '.'
   * @param directory 
   * @param leadingCharactersToRemove 
   * @return Returns a list of class names in a directory
   */
    protected static List<String> findClassesInDirectory(File directory, int leadingCharactersToRemove){
        Vector classes = new Vector();
        File[] files = directory.listFiles();
        for (File file: files){
            if (file.isDirectory()){
                classes.addAll(findClassesInDirectory(file, leadingCharactersToRemove));
            }
            if (file.isFile() && file.getName().endsWith(".class")){
                String className = file.getPath().substring(leadingCharactersToRemove + 1, file.getPath().length() - 6);
                className = className.replace("/", ".");
                className = className.replace("\\", ".");
                classes.add(className);
            }
        }
        return classes;
    }

    /**
     *  Search the classpath for persistence archives.  A persistence archive is defined as any
     *  part of the class path that contains a META-INF directory with a persistence.xml file in it. 
     *  Return a list of the URLs of those files.
     *  Use the current thread's context classloader to get the classpath.  We assume it is a URL class loader
     */
    public static Set<URL> findPersistenceArchives(){
        ClassLoader threadLoader = Thread.currentThread().getContextClassLoader();
        return findPersistenceArchives(threadLoader);
    }


    /**
     *  Search the classpath for persistence archives. A persistence archive is defined as any
     *  part of the class path that contains a META-INF directory with a persistence.xml file in it.. 
     *  Return a list of the URLs of those files.
     *  @param loader the class loader to get the class path from
     */
    public static Set<URL> findPersistenceArchives(ClassLoader loader){
         Set<URL> parURLs = new HashSet<URL>(); 
        try {
            Enumeration<URL> resources = loader.getResources("META-INF/persistence.xml");
            while (resources.hasMoreElements()){
                try{
                    URL url = resources.nextElement();
                    String newURLString = url.toString().substring(0, url.toString().length() - 25);
                    if (newURLString.endsWith("!")){
                        newURLString = newURLString.substring(0, newURLString.length() -1);
                    }
                    if (newURLString.startsWith("jar:")){
                        newURLString = newURLString.substring(4, newURLString.length());
                    }
                    URL newURL = new URL(newURLString);
                    parURLs.add(newURL);
                } catch (java.net.MalformedURLException exc){
                    throw PersistenceUnitLoadingException.exceptionSearchingForPersistenceResources(loader, exc);
                };
            }
        } catch (java.io.IOException exc){
            throw PersistenceUnitLoadingException.exceptionSearchingForPersistenceResources(loader, exc);
        }
        return parURLs;
    }

  /**
   * Return a list of all the names of classes stored in the jar stored at this URL.
   * Classes are assumed to be located in the base directory of the jar.
   * @param file 
   * @return 
   */
    private static List<String> getClassNamesFromJar(File file){
        List persistentClasses = new Vector();
        JarFile jarFile = null;
        try {
            // Only URL.toURI() handles spaces and special characters correctly
            jarFile = new JarFile(file);
            Enumeration e = jarFile.entries();
            while (e.hasMoreElements()){
                ZipEntry entry = (ZipEntry)e.nextElement();
                String classNameForLoader = buildClassNameFromEntryString(entry.getName());
                 if (entry.getName().endsWith(".class")){
                     persistentClasses.add(classNameForLoader);
                }
            }
        } catch (IOException exc){
            throw PersistenceUnitLoadingException.exceptionSearchingForEntities(file, exc);
        } finally {
            try{
                if (jarFile != null){
                    jarFile.close();                    
                }
            } catch (IOException exc){};
        }
        return persistentClasses;
    }

  /**
   * Return a list of class names from this URL.  This will work either with directories
   * or jar files and assume the classes are based in the base directory of the URL
   * @param url 
   * @return 
   */
    private static List<String> getClassNamesFromURL(URL url){                  
        // Only URL.toURI() handles spaces and special characters correctly
        File file = new File(convertURLToURI(url));
        if (file.isDirectory()){
            return getClassNamesFromDirectory(file);       
        } else {
            return getClassNamesFromJar(file);
        }      
    }

  /**
   *  Return a list of the names of classes that are stored in this directory.  Package
   *  name for the class will assume classes are based in the directoryURL
   * @param file the File representation of this directory
   * @return 
   */
    private static List<String> getClassNamesFromDirectory(File file){
        List classList = null;        

        if (!file.isDirectory()){
            return null;
        }
        int initialDirectoryNameLength = file.getPath().length();
        classList = findClassesInDirectory(file, initialDirectoryNameLength);        
        return classList;
    }

  /**
   *  Create a list of the classes that from a directory referenced by the given URL that return
   *  true from the isPersistent() call.
   * @param file the File representation of this directory
   * @param loader the ClassLoader to use
   * @return 
   */
    public static List<String> getPersistentClassNamesFromDirectory(File file, ClassLoader loader){
        List<String> persistentClasses = new Vector<String>();
        List<String> classList = getClassNamesFromDirectory(file);
        for (String className: classList){
            if (isClassPersistent(className, loader)){
                persistentClasses.add(className);
            }
        }
        return persistentClasses;
    }
    
  /**
   *  Return a list of persistent classes names accessible through a given URL
   *  If the URL refers to a .jar or .par file this method will look through the entries in 
   *  the jar for the class files.  If it refers to a directory, this method will recursively look
   *  for the classes in the directory.
   * @param url 
   * @return 
   */
    public static List getPersistentClassNamesFromURL(URL url, ClassLoader loader){                    
        // Only URL.toURI() handles spaces and special characters correctly
        File file = new File(convertURLToURI(url));            
        if (file.isDirectory()){
            return getPersistentClassNamesFromDirectory(file, loader);
        } else {
            return getPersistentClassNamesFromJar(file, loader);
        }
    }

  /**
   *  Create a list of the classes that from the jar with the given name.  That return
   *  true from the isPersistent() call.
   * @param file the File representation of this jar
   * @param loader the ClassLoader to use
   * @return 
   */
    public static List<String> getPersistentClassNamesFromJar(File file, ClassLoader loader){
        List<String> persistentClasses = new Vector();
        List<String> classList = getClassNamesFromJar(file);
        for (String className: classList){
            if (isClassPersistent(className, loader)){
                persistentClasses.add(className);
            }
        }
        return persistentClasses;
    }

  /**
   *  Return whether the class with the given name is persistent.
   *  A class is persistent if it is annotated with @Entity, @Embeddable or @MappedSuperclass
   * @param className 
   * @return 
   */
    public static boolean isClassPersistent(String className, ClassLoader loader){
        Class candidateClass = null;
        try{
            candidateClass = loader.loadClass(className);
        } catch (ClassNotFoundException exc){      
            throw PersistenceUnitLoadingException.exceptionLoadingClassWhileLookingForAnnotations(className, exc);
        }
        return isClassPersistent(candidateClass);
    }
    /**
    *  Return whether a given class is persistent
    *  A class is persistent if it is annotated with @Entity, @Embeddable or @MappedSuperclass
    * @param candidateClass 
    * @return 
    */
    public static boolean isClassPersistent(Class candidateClass){
        if (candidateClass.isAnnotationPresent(javax.persistence.Entity.class) 
            || candidateClass.isAnnotationPresent(javax.persistence.Embeddable.class)) {
            return true;
        }
        return false;
    }

    /**
    * Converts URL to a URI to handle spaces and special characters correctly.
    * Validates the URL to represent a valid file name.
    * @param url the URL to be converted
    * @return the corresponding URI
    */
    private static URI convertURLToURI(URL url) {
        String filePath = url.getFile();        
        if (filePath.equals("") || filePath == null) {
                throw PersistenceUnitLoadingException.filePathMissingException(filePath);
        }

        URI uri = null;
        try {
            uri = url.toURI();
        } catch (URISyntaxException e) {
            throw PersistenceUnitLoadingException.exceptionProcessingPersistenceUnit(url, e);
        }

        return uri;
    }

}
