/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the "License").  You may not use this file except 
 * in compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * glassfish/bootstrap/legal/CDDLv1.0.txt or 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html. 
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * HEADER in each file and include the License file at 
 * glassfish/bootstrap/legal/CDDLv1.0.txt.  If applicable, 
 * add the following below this CDDL HEADER, with the 
 * fields enclosed by brackets "[]" replaced with your 
 * own identifying information: Portions Copyright [yyyy] 
 * [name of copyright owner]
 */
// Copyright (c) 1998, 2006, Oracle. All rights reserved.  
package oracle.toplink.essentials.internal.ejb.cmp3.annotations;

import java.lang.annotation.Annotation;

import java.lang.reflect.Type;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.lang.reflect.AnnotatedElement;

import javax.persistence.Lob;
import javax.persistence.Column;
import javax.persistence.Version;
import javax.persistence.Embedded;
import javax.persistence.Temporal;
import javax.persistence.OneToOne;
import javax.persistence.OneToMany;
import javax.persistence.ManyToOne;
import javax.persistence.Embeddable;
import javax.persistence.Enumerated;
import javax.persistence.EmbeddedId;
import javax.persistence.ManyToMany;
import javax.persistence.JoinColumn;
import javax.persistence.JoinColumns;
import javax.persistence.SqlResultSetMapping;
import javax.persistence.PrimaryKeyJoinColumn;
import javax.persistence.PrimaryKeyJoinColumns;

import oracle.toplink.essentials.exceptions.ValidationException;
import oracle.toplink.essentials.internal.ejb.cmp3.metadata.MetadataHelper;
import oracle.toplink.essentials.internal.ejb.cmp3.metadata.MetadataAccessor;

/**
 * @author Guy Pelletier
 * @since TopLink 10.1.3/EJB 3.0 Preview
 */
public class AnnotationsAccessor extends MetadataAccessor {
    private String m_name;
    private Type m_relationType;
    private AnnotatedElement m_annotatedElement;
    
    private AnnotationsAccessor(EJBAnnotationsProcessor processor, AnnotatedElement annotatedElement, AnnotationsDescriptor descriptor) {
        setNeedsProcessing(true);
    
        m_isRelationship = null;    // Signifies that we don't know yet.
        m_metadataProcessor = processor;
        m_metadataDescriptor = descriptor;
        m_annotatedElement = annotatedElement;
    }
    
    public AnnotationsAccessor(Field field, EJBAnnotationsProcessor processor, AnnotationsDescriptor descriptor) {
        this(processor, field, descriptor);
        
        m_name = field.getName();
        m_attributeName = m_name;
        m_relationType = MetadataHelper.getGenericType(field);
    }
    
    public AnnotationsAccessor(Method method, EJBAnnotationsProcessor processor, AnnotationsDescriptor descriptor) {
        this(processor, method, descriptor);
        
        m_name = method.getName();
        m_attributeName = MetadataHelper.getAttributeNameFromMethodName(m_name);
        m_relationType = MetadataHelper.getGenericReturnType(method);
    }

    /**
     * INTERNAL:
     * Return the annotated element for this accessor.
     */
    public AnnotatedElement getAnnotatedElement() {
        return m_annotatedElement;
    }
    
    /**
     * INTERNAL:
     * Return the DescriptorMetadata for this accessor.
     */
    public AnnotationsDescriptor getMetadataDescriptor() {
        return (AnnotationsDescriptor) m_metadataDescriptor;
    }
    
    /**
     * INTERNAL:
     * Returns the name of this accessor. If it is a field, it will return 
     * the field name. For a method it will return the method name.
     */
    public String getName() {
        return m_name;
    }
    
    /**
     * INTERNAL:
     * Return the raw class for this accessor. 
     * Eg. For an accessor with a type of java.util.Collection<Employee>, this 
     * method will return java.util.Collection
     */
    public Class getRawClass() {
        if (m_rawClass == null) {
            if (MetadataHelper.isGenericCollectionType(m_relationType)) {
                // By default, the raw class is equal to the relation
                // class. But if the relation class is a generic we need to 
                // extract and set the actual raw class from the generic. 
                m_rawClass = MetadataHelper.getRawClassFromGeneric(m_relationType);
            } else {
                m_rawClass = (Class) m_relationType;
            }
        }
        
        return m_rawClass;    
    }
    
    /**
     * INTERNAL:
     * Return the reference class for this accessor.
     */
    public Class getReferenceClass() {
        if (m_referenceClass == null) {
            if (MetadataHelper.isGenericCollectionType(m_relationType)) {
                // By default, the reference class is equal to the relation
                // class. But if the relation class is a generic we need to 
                // extract and set the actual reference class from the generic. 
                m_referenceClass = MetadataHelper.getReturnTypeFromGeneric(m_relationType);
            } else {
                m_referenceClass = (Class) m_relationType;
            }
        }
        
        return m_referenceClass;
    }
    
    /**
     * INTERNAL:
     * Return the relation type of this accessor.
     */
    public Type getRelationType() {
        return m_relationType;
    }
    
    /**
     * INTERNAL:
	 * Method to check if an annotated element has @Column.
     */
	public boolean hasColumn() {
		return isAnnotationPresent(Column.class);
    }
    
    /**
     * INTERNAL:
	 * Method to check if an annotated element has an @Enumerated.
     */
	public boolean hasEnumerated() {
		return isAnnotationPresent(Enumerated.class);
    }
    
    /**
     * INTERNAL:
	 * Method to check if an annotated element has a @JoinColumn.
     */
	public boolean hasJoinColumn() {
		return isAnnotationPresent(JoinColumn.class);
    }
    
    /**
     * INTERNAL:
	 * Method to check if an annotated element has a @JoinColumns.
     */
	public boolean hasJoinColumns() {
		return isAnnotationPresent(JoinColumns.class);
    }
    
    /**
     * INTERNAL:
	 * Method to check if an annotated element has a @PrimaryKeyJoinColumn.
     */
	public boolean hasPrimaryKeyJoinColumn() {
		return isAnnotationPresent(PrimaryKeyJoinColumn.class);
    }
    
    /**
     * INTERNAL:
	 * Method to check if an annotated element has a @PrimaryKeyJoinColumns.
     */
	public boolean hasPrimaryKeyJoinColumns() {
		return isAnnotationPresent(PrimaryKeyJoinColumns.class);
    }
    
    /**
     * INTERNAL:
	 * Return true if this accessor represents an Sql result set mapping.
     */
	public boolean hasSqlResultSetMapping() {
        return isAnnotationPresent(SqlResultSetMapping.class);
    }
    
    /**
     * INTERNAL:
     * Return true if this accessor represents a temporal mapping.
     */
	public boolean hasTemporal() {
        return isAnnotationPresent(Temporal.class);
    }
    
    /** 
     * INTERNAL:
     * Indicates whether the specified annotation is present on the annotated
     * element for this accessor.
     */
    public boolean isAnnotationNotPresent(Class<? extends Annotation> annotation) {
        return AnnotationsHelper.isAnnotationNotPresent(annotation, m_annotatedElement);
    }
    
    /** 
     * INTERNAL:
     * Indicates whether the specified annotation is present on the annotated
     * element for this accessor. Method checks against the metadata complete
     * flag.
     */
    public boolean isAnnotationPresent(Class<? extends Annotation> annotation) {
        return AnnotationsHelper.isAnnotationPresent(annotation, m_annotatedElement, getMetadataDescriptor());
    }
    
    /**
     * INTERNAL:
     * Return true if this accessor represents an aggregate mapping. True is
     * returned id and @Embedded is found or if an @Embeddable is found on the
     * reference class.
     */
	public boolean isEmbedded() {
        if (isAnnotationNotPresent(Embedded.class) && isAnnotationNotPresent(EmbeddedId.class)) {
            return (AnnotationsHelper.isAnnotationPresent(Embeddable.class, getReferenceClass()));
        } else {
            // Still need to make the call since we may need to ignore it
            // because of meta-data complete.
            return isAnnotationPresent(Embedded.class);
        }
    }
    
    /**
     * INTERNAL:
     * Return true if this accessor represents an aggregate id mapping.
     */
	public boolean isEmbeddedId() {
        return isAnnotationPresent(EmbeddedId.class);
    }
     
    /**
     * INTERNAL:
     * Return true if this accessor represents an BLOB/CLOB mapping.
     */
	public boolean isLob() {
        return isAnnotationPresent(Lob.class);
    }
    
    /**
     * INTERNAL:
     * Return true if this accessor represents a m-m relationship.
     */
	public boolean isManyToMany() {
        if (isAnnotationPresent(ManyToMany.class)) {
            if (MetadataHelper.isSupportedCollectionClass(getRawClass())) {
                return true;
            } else {
                throw ValidationException.invalidCollectionTypeForRelationship(getRawClass(), m_annotatedElement);
            }
        } 
        
        return false;
    }
    
    /**
     * INTERNAL:
	 * Return true if this accessor represents a m-1 relationship.
     */
	public boolean isManyToOne() {
        return isAnnotationPresent(ManyToOne.class);
    }
    
    /**
     * INTERNAL:
	 * Return true if this accessor represents a 1-m relationship.
     */
	public boolean isOneToMany() {
        if (isAnnotationNotPresent(OneToMany.class)) {
            if (MetadataHelper.isGenericCollectionType(m_relationType) && MetadataHelper.isSupportedCollectionClass(getRawClass())) {
                m_metadataProcessor.getLogger().logConfigMessage(AnnotationsLogger.ONE_TO_MANY_MAPPING, m_annotatedElement);
                return true;
            }
        } else {
            // Still need to make the call since we may need to ignore it
            // because of meta-data complete.
            if (isAnnotationPresent(OneToMany.class)) {
                if (MetadataHelper.isSupportedCollectionClass(getRawClass())) {
                    return true;
                } else {
                    throw ValidationException.invalidCollectionTypeForRelationship(getRawClass(), m_annotatedElement);
                }
            }
        }
        
        return false;
    }
    
	/**
     * INTERNAL:
     * Return true if this accessor represents a 1-1 relationship.
     */
	public boolean isOneToOne() {
        if (isAnnotationNotPresent(OneToOne.class)) {
            if (((EJBAnnotationsProcessor) m_metadataProcessor).getClasses().contains(m_referenceClass)) {
                m_metadataProcessor.getLogger().logConfigMessage(AnnotationsLogger.ONE_TO_ONE_MAPPING, m_annotatedElement);
                return true;
            } else {
                return false;
            }
        } else {
            // Still need to make the call since we may need to ignore it
            // because of meta-data complete.
            return isAnnotationPresent(OneToOne.class);
        }
    }
    
    /**
     * INTERNAL:
	 * Return true if this accessor represents an optimistic locking field.
     */
	public boolean isVersion() {
        return isAnnotationPresent(Version.class);
    }
    
    /**
     * INTERNAL:
	 * Set the reference class for this accessor.
     */
    public void setReferenceClass(Class potentialReferenceClass, String context) {
        if (potentialReferenceClass == void.class) {
            // Log the defaulting column name based on the given context.
            m_metadataProcessor.getLogger().logConfigMessage(context, m_annotatedElement, m_referenceClass);
        } else {
            m_referenceClass = potentialReferenceClass;
        }
	}
    
    /**
     * INTERNAL:
	 * Store ourself on our respective MetadataDescriptor.
     */
    public void store() {
        getMetadataDescriptor().addAccessor(this);
	}
}
