/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the "License").  You may not use this file except 
 * in compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * glassfish/bootstrap/legal/CDDLv1.0.txt or 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html. 
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * HEADER in each file and include the License file at 
 * glassfish/bootstrap/legal/CDDLv1.0.txt.  If applicable, 
 * add the following below this CDDL HEADER, with the 
 * fields enclosed by brackets "[]" replaced with your 
 * own identifying information: Portions Copyright [yyyy] 
 * [name of copyright owner]
 */
package oracle.toplink.essentials.internal.ejb.cmp3.annotations;

import java.util.HashMap;
import java.util.Hashtable;
import java.util.ArrayList;
import java.lang.annotation.Annotation;
import java.lang.reflect.AnnotatedElement;

import oracle.toplink.essentials.internal.ejb.cmp3.metadata.MetadataHelper;
import oracle.toplink.essentials.internal.ejb.cmp3.metadata.MetadataAccessor;
import oracle.toplink.essentials.internal.ejb.cmp3.metadata.MetadataDescriptor;

/**
 * @author Guy Pelletier
 * @since TopLink 10.1.3/EJB 3.0 Preview
 */
public class AnnotationsHelper extends MetadataHelper {
    public static final String PERSISTENCE_PACKAGE_PREFIX = "javax.persistence";
    
    // Will cache the class access type for processing.
    protected static Hashtable<Class, Boolean> m_classAccessTypes; 
    
    /**
     * INTERNAL:
     * Wrapper to the getAnnotation() call using an Accessor.
     */
    public static <T extends Annotation> T getAnnotation(Class annotation, AnnotationsAccessor accessor) {
        return (T) getAnnotation(annotation, accessor.getAnnotatedElement(), accessor.getMetadataDescriptor());
    }
    
    /**
     * INTERNAL:
     * Method to read an annotation. I think there is a bug in the JDK when
     * reading annotations from classes. It returns the wrong type. Anyhow,
     * this method fixes that.
     */
    public static <T extends Annotation> T getAnnotation(Class annotation, AnnotatedElement annotatedElement) {
        return (T) annotatedElement.getAnnotation(annotation);
    }
    
    /**
     * INTERNAL:
     * Wrapper to the getAnnotation() call to check if we should ignore
     * annotations.
     */
    public static <T extends Annotation> T getAnnotation(Class annotation, AnnotatedElement annotatedElement, AnnotationsDescriptor descriptor) {
        // WIP - we should log a message to the fact of which annotations
        // we are ignoring.
        if (descriptor.shouldIgnoreAnnotations()) {
            return null;
        } else {
            return (T) getAnnotation(annotation, annotatedElement);
        }
    }
    
    /**
     * INTERNAL:
     * Wrapper to the getAnnotation() call using an MetadataDescriptor.
     */
    public static <T extends Annotation> T getAnnotation(Class annotation, AnnotationsDescriptor descriptor) {
        return (T) getAnnotation(annotation, descriptor.getJavaClass(), descriptor);
    }
    
    /**
     * INTERNAL:
     * Wrapper to the getAnnotation() call using an Accessor.
     */
    public static <T extends Annotation> T getAnnotation(Class annotation, MetadataAccessor accessor) {
        return (T) getAnnotation(annotation, (AnnotationsAccessor) accessor);
    }
    
    /**
     * INTERNAL:
     * Wrapper to the getAnnotation() call using an MetadataDescriptor.
     */
    public static <T extends Annotation> T getAnnotation(Class annotation, MetadataDescriptor md) {
        return (T) getAnnotation(annotation, (AnnotationsDescriptor) md);
    }
    
    /**
     * INTERNAL:
     */
    protected static Hashtable<Class, Boolean> getClassAccessTypes() {
        if (m_classAccessTypes == null) {
            m_classAccessTypes = new Hashtable<Class, Boolean>();
        }
        
        return m_classAccessTypes;
    }
    
    /**
     * INTERNAL:
     */
    protected static int getDeclaredAnnotationsCount(AnnotatedElement annotatedElement, AnnotationsDescriptor descriptor) {
        if (descriptor.shouldIgnoreAnnotations()) {
            return 0;
        } else {
            return annotatedElement.getDeclaredAnnotations().length;
        }
    }
    
    /**
     * INTERNAL:
     */
    protected static boolean havePersistenceAnnotationsDefined(AnnotatedElement[] annotatedElements) {
        for (AnnotatedElement annotatedElement : annotatedElements) {
            for (Annotation annotation : annotatedElement.getDeclaredAnnotations()) {
                if (annotation.annotationType().getName().startsWith(PERSISTENCE_PACKAGE_PREFIX)) {
                    return true;
                }
            }
        }
        
        return false;
    }
    
    /** 
     * INTERNAL:
     * Indicates whether the specified annotation is actually not present on 
     * the specified class. Used for defaulting. Need this check since the
     * isAnnotationPresent calls can return a false when true because of the
     * meta-data complete feature.
     */
    public static boolean isAnnotationNotPresent(Class annotation, AnnotatedElement annotatedElement) {
        return ! annotatedElement.isAnnotationPresent(annotation);
    }
    
    /** 
     * INTERNAL:
     * Indicates whether the specified annotation is present on the specified 
     * class.
     */
    public static boolean isAnnotationPresent(Class annotation, AnnotatedElement annotatedElement) {
        return annotatedElement.isAnnotationPresent(annotation);
    }
    
    /** 
     * INTERNAL:
     * Indicates whether the specified annotation is present on the specified 
     * class.
     */
    public static boolean isAnnotationPresent(Class annotation, AnnotatedElement annotatedElement, AnnotationsDescriptor descriptor) {
        // WIP - we should log a message to the fact of which annotations
        // we are ignoring.
        if (descriptor.shouldIgnoreAnnotations()) {
            return false;
        } else {
            return isAnnotationPresent(annotation, annotatedElement);
        }
    }
    
    /** 
     * INTERNAL:
     * Indicates whether the specified annotation is present on java class
     * for the given descriptor metadata. 
     */
    public static boolean isAnnotationPresent(Class annotation, AnnotationsDescriptor descriptor) {
        return isAnnotationPresent(annotation, descriptor.getJavaClass(), descriptor);
    }
    
    /** 
     * INTERNAL:
     * Indicates whether the class should ignore annotations. Note that such 
     * classes should already have their descriptors with PKs added to 
     * session's project.
     */
    public static boolean shouldIgnoreAnnotations(Class cls, HashMap<Class, AnnotationsDescriptor> metadataDescriptors) {
        AnnotationsDescriptor descriptor = metadataDescriptors.get(cls);
        
        if (descriptor != null) {
            return descriptor.shouldIgnoreAnnotations();
        } else {
            return false;
        }
    }
    
    /**
     * INTERNAL:
     * Return true if this class uses property access. Since this method may 
     * be called for the same class several times (e.g. in an inheritance 
     * hierarchy), its access type will be cached for quick retrieval and 
     * performance enhancement.
     * 
     * NOTE: in an inheritance hierarchy, the access type for the inheritance
     * subclasses is equal to that of the root class. 
     */
    public static Boolean usesPropertyAccess(AnnotationsDescriptor md) {
        Class entityClass = md.getJavaClass();
        
        if (! getClassAccessTypes().containsKey(entityClass)) {
            // We need to figure out the access type ...
            
            if (havePersistenceAnnotationsDefined(getFields(entityClass)) || md.isXmlFieldAccess()) {
                // We have persistence annotations defined on a field from the
                // entity or field access has been set via XML, set the access 
                // to FIELD.
                getClassAccessTypes().put(entityClass, false);
            } else if (havePersistenceAnnotationsDefined(getDeclaredMethods(entityClass)) || md.isXmlPropertyAccess()) {
                // We have persistence annotations defined on a method from the
                // entity or method access has been set via XML, set the access 
                // to PROPERTY.
                getClassAccessTypes().put(entityClass, true);
            } else {
                for (Class mappedSuperclass : (ArrayList<Class>) md.getMappedSuperclasses()) {
                    if (havePersistenceAnnotationsDefined(getFields(mappedSuperclass))) {
                        // We have persistence annotations defined on a field 
                        // from a mapped superclass, set the access to FIELD.
                        getClassAccessTypes().put(entityClass, false);
                        break;
                    } else if (havePersistenceAnnotationsDefined(getDeclaredMethods(mappedSuperclass))) {
                        // We have persistence annotations defined on a method 
                        // from a mapped superclass, set the access to FIELD.
                        getClassAccessTypes().put(entityClass, true);
                        break;
                    }
                }
                
                // We still found nothing ... we should throw an exception here,
                // but for now, set the access to PROPERTY. The user will 
                // eventually get an exception saying there is no primary key 
                // set.
                if (! getClassAccessTypes().containsKey(entityClass)) {
                    getClassAccessTypes().put(entityClass, true);
                }
            }
        }
        
        return getClassAccessTypes().get(entityClass);
    }
}
