/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the "License").  You may not use this file except 
 * in compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * glassfish/bootstrap/legal/CDDLv1.0.txt or 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html. 
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * HEADER in each file and include the License file at 
 * glassfish/bootstrap/legal/CDDLv1.0.txt.  If applicable, 
 * add the following below this CDDL HEADER, with the 
 * fields enclosed by brackets "[]" replaced with your 
 * own identifying information: Portions Copyright [yyyy] 
 * [name of copyright owner]
 */
// Copyright (c) 1998, 2005, Oracle. All rights reserved.  
package oracle.toplink.essentials.mappings.converters;

import java.util.*;
import oracle.toplink.essentials.sessions.Session;
import oracle.toplink.essentials.mappings.DatabaseMapping;
import oracle.toplink.essentials.exceptions.ValidationException;
import oracle.toplink.essentials.internal.security.PrivilegedAccessController;

/**
 * <b>Purpose</b>: Object type converter is used to match a fixed number of 
 * database data values to a Java enum object value. It can be used when the 
 * values on the database and in the Java differ. To create an object type 
 * converter, simply specify the set of conversion value pairs. A default value 
 * and one-way conversion are also supported for legacy data situations.
 *
 * @author Guy Pelletier
 * @since Toplink 10.1.4RI
 */
public class EnumTypeConverter extends ObjectTypeConverter {
    private Class m_enumClass;
    private String m_enumClassName;
	private boolean m_usesOrdinalValues;

    /**
     * PUBLIC:
     */
    public EnumTypeConverter(DatabaseMapping mapping, String enumClassName, boolean usesOrdinalValues) {
        super(mapping);

        m_enumClassName = enumClassName;
		m_usesOrdinalValues = usesOrdinalValues;
    }
    
    /**
     * INTERNAL:
     * Convert all the class-name-based settings in this converter to actual 
     * class-based settings. This method is used when converting a project 
     * that has been built with class names to a project with classes.
     * @param classLoader 
     */
    public void convertClassNamesToClasses(ClassLoader classLoader){
        try {
            m_enumClass = PrivilegedAccessController.getClassForName(m_enumClassName, true, classLoader);
        } catch (ClassNotFoundException exception){
            throw ValidationException.classNotFoundWhileConvertingClassNames(m_enumClassName, exception);
        }
    }
    
    /**
     * INTERNAL:
     */
    public void initialize(DatabaseMapping mapping, Session session) {
        Iterator<Enum> i = EnumSet.allOf(m_enumClass).iterator();
        
        while (i.hasNext()) {
            Enum theEnum = i.next();
            
            if (m_usesOrdinalValues) {
                addConversionValue(theEnum.ordinal(), theEnum);
            } else {
                addConversionValue(theEnum.name(), theEnum);
            }
        }
        
        super.initialize(mapping, session);
    }
    
    /**
     * PUBLIC:
     * Returns true if this converter uses ordinal values for the enum
     * conversion.
     */
	public boolean usesOrdinalValues() {
		return m_usesOrdinalValues;   
    }
}
